/*
 * Copyright (c) OSGi Alliance (2010). All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.osgi.jmx.service.coordinator;

import java.io.IOException;

import javax.management.openmbean.CompositeData;
import javax.management.openmbean.CompositeType;
import javax.management.openmbean.SimpleType;
import javax.management.openmbean.TabularData;
import javax.management.openmbean.TabularType;

import org.osgi.jmx.Item;
import org.osgi.jmx.JmxConstants;

/**
 * This MBean provides the management interface to the OSGi Coordinator Service
 * 
 * @version $Id: 4c8a1031c3e93d3d88a21faa48fdacedf5051de0 $
 * @ThreadSafe
 */
public interface CoordinatorMBean {
	/**
	 * User Admin MBean object name.
	 */
	String			OBJECTNAME			= JmxConstants.OSGI_COMPENDIUM
												+ ":service=coordinator,version=1.0";

	/**
	 * The key NAME, used in {@link #NAME_ITEM}.
	 */
	String			NAME				= "Name";

	/**
	 * The item for the user name for an authorization object. The key is
	 * {@link #NAME} and the type is {@link SimpleType#STRING}.
	 */
	Item			NAME_ITEM			= new Item(NAME,
												"The name of a Coordination",
												SimpleType.STRING);

	/**
	 * The key ID, used in {@link #ID_ITEM}.
	 */
	String			ID					= "Id";

	/**
	 * The item for the id of an Coordination object. The key is {@link #ID} and
	 * the type is {@link SimpleType#LONG}. The id must be generated by the
	 * Mbean and map to a unique Coordination (which should no be pinned in
	 * memory because of this).
	 */
	Item			ID_ITEM				= new Item(ID,
												"The id of a Coordination",
												SimpleType.LONG);

	/**
	 * The key TIMEOUT, used in {@link #TIMEOUT_ITEM}.
	 */
	String			TIMEOUT				= "Timeout";

	/**
	 * The item for the id of an Coordination object. The key is {@link #ID} and
	 * the type is {@link SimpleType#LONG}.
	 */
	Item			TIMEOUT_ITEM		= new Item(
												TIMEOUT,
												"The timeout of a Coordination",
												SimpleType.LONG);

	/**
	 */
	CompositeType	COORDINATION_TYPE	= Item.compositeType("COORDINATION",
												"Coordination mapping",
												ID_ITEM, NAME_ITEM,
												TIMEOUT_ITEM);

	/**
	 * Defines a list of {@link #COORDINATION_TYPE}
	 */
	TabularType		COORDINATIONS_TYPE	= Item.tabularType("COORDINATIONS", "A list of Coordinations", COORDINATION_TYPE);

	/**
	 * List the current coordinations.
	 * 
	 * The Composite Data is typed by {@link #COORDINATIONS_TYPE}.
	 * 
	 * @param regexFilter a regular expression filter on the coordination name
	 * @return the Coordinations typed by {@link #COORDINATIONS_TYPE}.
	 * @throws IOException if the operation fails
	 */
	TabularData listCoordinations(String regexFilter) throws IOException;

	/**
	 * Get a Coordination.
	 * 
	 * The Composite Data is typed by {@link #COORDINATION_TYPE}.
	 * 
	 * @param id The id of a Coordination
	 * @return the Coordinations typed by {@link #COORDINATION_TYPE}.
	 * @throws IOException if the operation fails
	 */
	CompositeData getCoordination(long id) throws IOException;

	/**
	 * Fail a Coordination.
	 * 
	 * @param id The id of the coordination to be failed.
	 * @param reason
	 * @return {@code true} if the coordination was failed by this call,
	 *         otherwise {@code false}.
	 * @throws IOException
	 */
	boolean fail(long id, String reason) throws IOException;

	/**
	 * Set/Change the timeout of a Coordination.
	 * 
	 * @param id The id of the Coordination
	 * @param timeout The nr of milliseconds for the next timeout.
	 * @throws IOException
	 */
	void addTimeout(long id, long timeout) throws IOException;

}
