/*
 * Copyright 2013 Canonical Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import QtQuick 2.0
import QtQuick.LocalStorage 2.0
import "dateTimeUtils.js" as DateTime
/*!
  \brief Notes database model.

  The model loads the notes in memory and builds up a model that can be used
  in Notes app.

  The very first element is an empty one instructing

  Roles:
    - id: unique identifier, should be used to touch and delete note
    - date: creation or last modification date
    - data: note data

  When executing operations such as touch, new and delete note, the model is
  automatically refreshed.

  The model keeps all the data in the memory. This could be memory consuming
  and should be optimized by loading the notes only for those items requesting
  for it.
  */

ListModel {
    id: model

    /*!
      Refreshes the model by loading the notes from the database
      */
    function refresh() {
        if (!__db)
            return;
        __db.readTransaction(function (tx) {
            model.clear();
            var rs = tx.executeSql("SELECT * FROM notes ORDER BY date DESC");
            if (rs.rows.length == 0) {
                // add fake element if we don't have any note in the database
                model.append({"id": -1, "date": DateTime.formatDateTime(new Date()), "note": ""});
            } else {
                for(var i = 0; i < rs.rows.length; i++) {
                    var item = rs.rows.item(i);
                    model.append({"id": parseInt(item.id), "date": DateTime.formatDateTime(item.date), "note": item.note});
                }
            }
        });
    }

    /*!
      Updates (touches) a note. The date is automatically updated.
      */
    function touchNote(index, note) {
        if (!__db)
            return;
        var id = model.get(index).id;
        __db.transaction(function (tx) {
            var dt = new Date();
            if (id !== -1) {
                tx.executeSql("UPDATE notes SET date=?, note=? WHERE id=?", [dt, note, id]);
            } else {
                var result = tx.executeSql("INSERT INTO notes VALUES(NULL, ?, ?)", [dt, note]);
                model.setProperty(index, "id", parseInt(result.insertId));
            }
            model.setProperty(index, "date", DateTime.formatDateTime(dt));
            if (index > 0)
                model.move(index, 0, 1);
        });
    }

    /*!
      Stores a new note into the database.
      */
    function newNote(note) {
        if (!__db)
            return;
        __db.transaction(function (tx) {
            var dt = new Date();
            var result = tx.executeSql("INSERT INTO notes VALUES(NULL, ?, ?)", [dt, ""]);
            model.insert(0, {"id": parseInt(result.insertId), "date": DateTime.formatDateTime(dt), "note": note})
        });
    }

    /*!
      Deletes a note identified by its database ID.
      */
    function deleteNote(index) {
        if (!__db)
            return;
        if (model.get(index) === undefined) {
            console.debug("undefined item to be removed!")
            return;
        }
        var id = model.get(index).id;
        __db.transaction(function (tx) {
            tx.executeSql("DELETE FROM notes WHERE id=?", [id]);
            model.remove(index);
        });
    }

    property var __db: null

    Component.onCompleted: {
        __db = LocalStorage.openDatabaseSync("notes", 1.0, "Notes database", 3 * 1024 * 1024);
        __db.transaction(function (tx){
            // create table if not exist
            tx.executeSql("CREATE TABLE IF NOT EXISTS notes (id INTEGER PRIMARY KEY, date DATETIME, note TEXT)");
        });
        refresh();
    }
}
