#ifndef _NUMBERS_H
#define _NUMBERS_H

/* numbers.h
  
   Data structures and functions for handling of numeric specifications.

   Copyright (C) 2007, 2008, 2009 Eloy Paris

   This is part of Network Expect.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <dnet.h>

enum numtype {
    /* Numbers */
    NUM_FIXED,
    NUM_LIST,
    NUM_RANGE,
    NUM_RANDOM,
    NUM_ADD,
    NUM_SUB,
    NUM_MUL,
    NUM_DIV,
    NUM_MOD,
    NUM_SHL,
    NUM_SHR,

    /* IP addresses */
    IP_FIXED,
    IP_LIST,
    IP_RANGE,
    IP_RANDOM,
    IP_ADD
};

typedef struct _numspec {
    enum numtype type;	/* Type of numeric specification */
    int sequence;	/* Order in which the numspec was created */
    int change_freq;	/* How often the number in a list or range changes */
    int change_counter;
    int nvalues;	/* Number of values if list or range */
    char *spec;		/* String representation, as the corresponding
			   xxx_init() function received. It is allocated
			   via strdup() so xxx_destroy() needs to free() it. */

    union {
	/* Fixed number */
	struct {
	    uint32_t value;
	} __u32;

	/* Number that changes based on an operation (add, sub, mul, etc.) */
	struct {
	    uint32_t current;
	    int operand;
	} __u32op;

	/* Number that changes within a range */
	struct {
	    uint32_t current;
	    uint32_t start;
	    uint32_t end;
	} __u32range;

	/* Fully random number, or random number within a range */
	struct {
	    uint32_t start;
	    uint32_t end;
	    uint32_t value;	    /* Used only for pseudo-random */
	    double probability;	    /* Used only for pseudo-random */
	} __u32rand;

	/* Number that changes within a list */
	struct {
	    uint32_t curr_index;
	    uint32_t *values;
	} __u32list;

	/* Fixed IP address */
	struct {
	    ip_addr_t value;
	} __ip;

	/* IP address that changes based on an operation (add, sub, mul, etc.) */
	struct {
	    ip_addr_t current;
	    int operand;
	} __ipop;

	/* IP address that changes within a range */
	struct {
	    ip_addr_t current;
	    ip_addr_t start;
	    ip_addr_t end;
	} __iprange;

	/* Fully random IP address, or random IP address within a range */
	struct {
	    ip_addr_t start;
	    ip_addr_t end;
	} __iprand;

	/* IP address that changes within a list */
	struct {
	    uint32_t curr_index;
	    ip_addr_t *values;
	} __iplist;
    } __num_u;
} numspec_t;

#define num_u32		__num_u.__u32
#define num_u32op	__num_u.__u32op
#define num_u32range	__num_u.__u32range
#define num_u32rand	__num_u.__u32rand
#define num_u32list	__num_u.__u32list

#define num_ip		__num_u.__ip
#define num_ipop	__num_u.__ipop
#define num_iprange	__num_u.__iprange
#define num_iprand	__num_u.__iprand
#define num_iplist	__num_u.__iplist

/*
 * The numeric specification parser interprets symbols differently based on the
 * xxx_init() function that was called. For instance, if num_init() is called
 * then symbols are interpreted as service names (from /etc/services). If
 * ip_init() is called then symbols are interpreted as hostnames. The following
 * enum is used to provide a hint regarding the symbol interpretation that is
 * being used.
 */
enum {
    HINT_NUM,
    HINT_IP
};

/*
 * This keeps track of the order in which numspecs are created. It must
 * be set to zero before numspecs begin to be created.
 */
extern int numspec_seq;

/* Numbers */
extern numspec_t *num_init(const char *);
extern numspec_t *num_initn(uint32_t);
extern uint32_t num_next(numspec_t *);

/* IPv4 addresses */
extern numspec_t *ip_init(const char *);
extern numspec_t *ip_initn(ip_addr_t);
extern ip_addr_t ip_next(numspec_t *);

/* Used by all types of numeric specifications, i.e. numbers and IP addrs. */
extern const char *num_info(numspec_t *);
extern void num_destroy(numspec_t *);
extern const char *num_spec(numspec_t *);
extern int num_getfreq(numspec_t *);
extern void num_setfreq(numspec_t *, int);
extern int num_nvalues(numspec_t *);
extern int num_seq(numspec_t *);

#endif /* _NUMBERS_H */
