/*!
	@file
	@author		Albert Semenov
	@date		07/2012
*/

#include "Precompiled.h"
#include "SkinTextureController.h"
#include "FactoryManager.h"
#include "DataSelectorManager.h"
#include "DataManager.h"
#include "PropertyUtility.h"
#include "ScopeManager.h"
#include "DataUtility.h"

namespace tools
{

	FACTORY_ITEM_ATTRIBUTE(SkinTextureController)

	void SkinTextureController::setTarget(Control* _control)
	{
		mControl = _control->findControl<ScopeTextureControl>();
	}

	void SkinTextureController::activate()
	{
		mParentTypeName = "Root";
		mScopeName = "Skin";

		ScopeManager::getInstance().eventChangeScope.connect(this, &SkinTextureController::notifyChangeScope);
		notifyChangeScope(ScopeManager::getInstance().getCurrentScope());
	}

	void SkinTextureController::deactivate()
	{
		ScopeManager::getInstance().eventChangeScope.disconnect(this);
	}

	void SkinTextureController::notifyChangeDataSelector(DataPtr _data, bool _changeOnlySelection)
	{
		mParentData = _data;
		if (mParentData != nullptr && mParentData->getType()->getName() != mParentTypeName)
			mParentData = nullptr;

		std::string texture;
		PropertyPtr property = PropertyUtility::getPropertyByName("Skin", "Texture");
		if (property != nullptr)
		{
			texture = property->getValue();

			if (!property->eventChangeProperty.exist(this, &SkinTextureController::notifyChangeProperty))
				property->eventChangeProperty.connect(this, &SkinTextureController::notifyChangeProperty);
		}

		std::string coord;
		property = PropertyUtility::getPropertyByName("Skin", "Size");
		if (property != nullptr)
		{
			coord = property->getValue();

			if (!property->eventChangeProperty.exist(this, &SkinTextureController::notifyChangeProperty))
				property->eventChangeProperty.connect(this, &SkinTextureController::notifyChangeProperty);
		}

		updateTexture(texture);
		updateCoords(coord);
	}

	void SkinTextureController::notifyChangeProperty(PropertyPtr _sender)
	{
		if (!mActivated || !PropertyUtility::isDataSelected(_sender->getOwner()))
			return;

		if (_sender->getOwner()->getType()->getName() == "Skin")
		{
			if (_sender->getType()->getName() == "Texture")
				updateTexture(_sender->getValue());
			else if (_sender->getType()->getName() == "Size")
				updateCoords(_sender->getValue());
		}
	}

	void SkinTextureController::notifyChangeValue(std::string_view _value)
	{
		PropertyPtr property = PropertyUtility::getPropertyByName("Skin", "Size");
		if (property != nullptr)
			PropertyUtility::executeAction(property, _value, true);
	}

	void SkinTextureController::notifyChangeScope(std::string_view _scope)
	{
		if (mControl == nullptr)
			return;

		if (_scope == mScopeName)
		{
			if (!mActivated)
			{
				mControl->eventChangeValue.connect(this, &SkinTextureController::notifyChangeValue);
				mControl->clearAll();

				DataSelectorManager::getInstance()
					.getEvent(mParentTypeName)
					->connect(this, &SkinTextureController::notifyChangeDataSelector);
				mParentData = DataUtility::getSelectedDataByType(mParentTypeName);
				notifyChangeDataSelector(mParentData, false);

				mControl->getRoot()->setUserString("CurrentScopeController", mScopeName);

				mActivated = true;
			}
		}
		else
		{
			if (mActivated)
			{
				mControl->eventChangeValue.disconnect(this);

				DataSelectorManager::getInstance().getEvent(mParentTypeName)->disconnect(this);
				mParentData = nullptr;

				// мы еще владельцы контрола сбрасываем его
				std::string_view value = mControl->getRoot()->getUserString("CurrentScopeController");
				if (value == mScopeName)
				{
					mControl->getRoot()->setUserString("CurrentScopeController", std::string_view{});
					notifyChangeDataSelector(mParentData, false);

					mControl->clearAll();
				}

				mActivated = false;
			}
		}
	}

	void SkinTextureController::updateCoords(std::string_view _value)
	{
		MyGUI::IntCoord coord;
		if (MyGUI::utility::parseComplex(_value, coord.left, coord.top, coord.width, coord.height))
		{
			mControl->setCoordValue(coord, ScopeTextureControl::SelectorCoord);
		}
		else
		{
			mControl->clearCoordValue();
		}
	}

	void SkinTextureController::updateTexture(std::string_view _value)
	{
		mControl->setTextureValue(MyGUI::UString(_value));
		mControl->resetTextureRegion();
	}

}
