/*
 * Copyright (C) 2009 Mathias Doenitz
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.parboiled;

import org.jetbrains.annotations.NotNull;
import org.parboiled.matchers.Matcher;
import org.parboiled.trees.TreeNode;

/**
 * Represents a node in the parse tree created during a parsing run.
 *
 * @param <V> the type of the value field of a parse tree node
 */
public interface Node<V> extends TreeNode<Node<V>> {

    /**
     * Returns the matcher that created this node.
     *
     * @return the matcher that created this node.
     */
    @NotNull
    Matcher<V> getMatcher();

    /**
     * Returns the label of this node which is equal to the name of the rule that created this node
     *
     * @return the label of this node
     */
    String getLabel();

    /**
     * Returns the start index of this nodes text in the underlying input buffer.
     *
     * @return the start index
     */
    int getStartIndex();

    /**
     * Returns the end index of this nodes text in the underlying input buffer, i.e. the index of the character
     * immediately following the last character matched by this node.
     *
     * @return the end index
     */
    int getEndIndex();

    /**
     * Returns the value object attached to this node.
     *
     * @return the value object
     */
    V getValue();

    /**
     * @return true if there were parse errors in the input range covered by this node
     */
    boolean hasError();

}
