/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2012 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

//
// Implementation of ianimator.h
//

#include "ianimator.h"

#include "ianimatorscript.h"
#include "icommoneventobservers.h"
#include "icontrolmodule.h"
#include "icrosssectionviewsubject.h"
#include "idatareader.h"
#include "idirectory.h"
#include "ierror.h"
#include "ierrorstatus.h"
#include "ifile.h"
#include "iimagecomposer.h"
#include "imath.h"
#include "ishell.h"
#include "ishellfactory.h"
#include "isystem.h"
#include "iviewmodule.h"
#include "iviewobject.h"
#include "iviewobjectfamily.h"

#include <vtkCamera.h>
#include <vtkMath.h>
#include <vtkImageBlend.h>
#include <vtkPolyData.h>
#include <vtkProperty.h>
#include <vtkRenderer.h>
#include <vtkRenderWindow.h>


using namespace iParameter;

//
//  Templates
//
#include "iarraytemplate.h"


#define REN this->GetViewModule()->GetRenderer()
#define DATAREADER this->GetViewModule()->GetReader()
#define CAM this->GetViewModule()->GetRenderer()->GetActiveCamera()
#define XSECTION this->GetViewModule()->GetCrossSectionViewSubject()

#define RAN_U (2.0*vtkMath::Random()-1.0)
#define RAN_N (1.4142*tan(2.0*RAN_U/3.1415926))
#define LEN(x) (sqrt(x[0]*x[0]+x[1]*x[1]+x[2]*x[2]))


IOBJECT_DEFINE_TYPE(iAnimator,Animator,a,iObjectType::_Helper);

IOBJECT_DEFINE_KEY(iAnimator,Phi,dp,Float,1);
IOBJECT_DEFINE_KEY(iAnimator,Roll,dr,Float,1);
IOBJECT_DEFINE_KEY(iAnimator,Theta,dt,Float,1);
IOBJECT_DEFINE_KEY(iAnimator,Zoom,dz,Float,1);
IOBJECT_DEFINE_KEY(iAnimator,FlybySpeed,fs,Float,1);
IOBJECT_DEFINE_KEY(iAnimator,LoadScriptFile,ls,String,1);
IOBJECT_DEFINE_KEY(iAnimator,LogoFile,lf,String,1);
IOBJECT_DEFINE_KEY(iAnimator,LogoLocation,ll,Int,1);
IOBJECT_DEFINE_KEY(iAnimator,LogoOpacity,lo,Float,1);
IOBJECT_DEFINE_KEY(iAnimator,NumberOfBlendedFrames,nb,Int,1);
IOBJECT_DEFINE_KEY(iAnimator,NumberOfFrames,nf,Int,1);
IOBJECT_DEFINE_KEY(iAnimator,NumberOfTransitionFrames,nt,Int,1);
IOBJECT_DEFINE_KEY(iAnimator,Style,s,Int,1);
IOBJECT_DEFINE_KEY(iAnimator,ScriptFile,sf,String,1);
IOBJECT_DEFINE_KEY(iAnimator,NumberOfTitlePageBlendedFrames,tbf,Int,1);
IOBJECT_DEFINE_KEY(iAnimator,TitlePageFile,tf,String,1);
IOBJECT_DEFINE_KEY(iAnimator,NumberOfTitlePageFrames,tnf,Int,1);

IOBJECT_DEFINE_KEY(iAnimator,CrossSectionSpeed,xs,Float,1);

IOBJECT_DEFINE_KEY(iAnimator,DebugFlag,-df,Int,1);
IOBJECT_DEFINE_KEY(iAnimator,Debugging,-dd,Bool,1); // need a bool key for GUI shell

//
//  Special keys that contain pointers to internal data. They should never be packed/unpacked,
//  only used for internal communication. Is there a better design?
//
IOBJECT_DEFINE_KEY(iAnimator,LogoImage,-li,Any,0);
IOBJECT_DEFINE_KEY(iAnimator,TitlePageImage,-ti,Any,0);


//
//  iAnimator class
//
iAnimator* iAnimator::New(iViewModule *vm)
{
	IERROR_ASSERT(vm);
	return new iAnimator(vm); // non-inheritable, so no need to use Object Factory
}


iAnimator::iAnimator(iViewModule *vm) : iObject("Animator"), mViewModule(vm)
{
	mStarted = mStartedRender = mUsingScript = false;
	mDebugFlag = 0;

//	scriptFile = ""; (this is automatic with iString)

	mNewRec = false;
	mCurRec = mPrevRec = -1;
	mCurFrame = mTotFrame = 0;

	mState.mode = 1;
	mState.nframes = 1;
	mState.dphi = 1.0;
	mState.dtheta = 0.0;
	mState.dscale = 1.0;
	mState.droll = 0.0;
	mState.flybySpeed = 0.0;
	mState.slideSpeed = 0.0;
	mState.nBlendedFrames = 0;
	mState.nTransitionFrames = 0;

	mTitlePageNumFrames = 0;
	mTitlePageNumBlendedFrames = 0;
	mLogoOpacity = 0.5;
	mLogoLocation = 0;

	mRandStep = 0.03;
	mSeed = 12345;

	mAnimatorScript = iAnimatorScript::New(this); IERROR_ASSERT(mAnimatorScript);
	mAnimatorObserver = iAnimatorEventObserver::New(this->GetViewModule()); IERROR_ASSERT(mAnimatorObserver);

	mLeader = 0;
}


iAnimator::~iAnimator()
{
	this->RemoveAllFollowers();
	if(mLeader != 0) mLeader->RemoveFollower(this);

	mAnimatorObserver->Delete();
	mAnimatorScript->Delete();
	while(mBlenderBase.Size() > 0) delete mBlenderBase.RemoveLast();
}


//
//  This is the main driver
//
void iAnimator::Animate()
{
	this->Start();
	while(this->Continue());
	this->Stop();
}


//
//  Start the animator
//
void iAnimator::Start()
{
	this->GetErrorStatus()->Clear();

	if(mStarted)
	{
		this->GetErrorStatus()->Set("Attempting to start an already started animation.");
		return;
	}
	//
	//  Disactivate Progress Dialog
	//
	iAbortRenderEventObserver::BlockAbortRenderEventObservers(true);
	//
	//  Are we using a script?
	//
	this->SaveState();
	if(mUsingScript)
	{
		mState.mode = 0;
		mState.nframes = 1;
		mState.dphi = 0.0;
		mState.dtheta = 0.0;
		mState.dscale = 1.0;
		mState.droll = 0.0;
		mState.flybySpeed = 0.01;
		mState.slideSpeed = 0.0;
		mState.nBlendedFrames = 0;
		mState.nTransitionFrames = 0;

		this->ClearCache();
	}
	else
	{
		mAnimatorScript->SetText("render -1");
	}

	this->GetViewModule()->StartAnimation();
	mAnimatorScript->StartExecute();

	mStarted = true;
	this->ClearCache();

	int i;
	for(i=0; i<mFollowers.Size(); i++) mFollowers[i]->Start();
}


//
//  Keep animating until stopped
//
bool iAnimator::Continue()
{
	if(!mStarted)
	{
		this->GetErrorStatus()->Set("Attempting to continue a never started animation.");
		return false;
	}

	this->GetErrorStatus()->Monitor(mAnimatorScript->GetErrorStatus());
	this->ClearCache();
	return mAnimatorScript->ExecuteOneLine();
}


//
//  Stop animation
//
void iAnimator::Stop()
{
	int i;
	for(i=0; i<mFollowers.Size(); i++) mFollowers[i]->Stop();

	if(!mStarted)
	{
		this->GetErrorStatus()->Set("Attempting to finish a never started animation.");
		return;
	}

	this->GetViewModule()->FinishAnimation();
	mAnimatorScript->StopExecute();

	//
	//  Reset to initial state
	//
	mStartedRender = false;
	this->GetViewModule()->JustifyLabelLeft(false);

	//
	//  Restore the original state after execution of a script - must do it after reset()
	//  so that projection is Set properly in case it was changed in a script before the
	//  first render
	//
	this->RestoreState(true);

	if(!mUsingScript) mAnimatorScript->SetText("");

	//
	//  Activate Progress Dialog
	//
	iAbortRenderEventObserver::BlockAbortRenderEventObservers(false);

	mStarted = false;
	this->ClearCache();
}


void iAnimator::AddFollower(iAnimator *f)
{
	if(f!=0 && f!=this)
	{
		mFollowers.AddUnique(f);
		f->SetLeader(this);
	}
}


void iAnimator::RemoveFollower(iAnimator *f)
{
	if(f != 0)
	{
		mFollowers.Remove(f);
		f->SetLeader(0);
	}
}


void iAnimator::SetLeader(iAnimator *l)
{
	if(l != this)
	{
		mLeader = l;
	}
}


void iAnimator::RemoveAllFollowers()
{
	int i;
	for(i=0; i<mFollowers.Size(); i++) mFollowers[i]->SetLeader(0);
	mFollowers.Clear();
}


//
//  Internal functions
//
void iAnimator::SetDebugFlag(int s) 
{ 
	mDebugFlag = s;
	this->GetViewModule()->SetDebugMode(s > 1);
	this->ClearCache();
}


void iAnimator::ResetCurrentFile()
{
	mNewRec = true;
	mPrevRec = mCurRec;
	mCurRec = DATAREADER->GetRecordNumber();
	mCurFrame = mTotFrame = 0;
	this->ClearCache();
}


void iAnimator::SaveState()
{
	if(XSECTION != 0) mState.xsecPos = XSECTION->GetLocation(); else mState.xsecPos = 0.0;
	mState.ifBoundingBox = this->GetViewModule()->IsBoundingBoxVisible();
	mState.ifColorBars = this->GetViewModule()->IsColorBarsVisible();
	mState.ifTimeLabel = this->GetViewModule()->IsLabelVisible();

	if(DATAREADER != 0) mState.currec = this->GetViewModule()->GetReader()->GetRecordNumber();

	mState.cameraProjection = CAM->GetParallelProjection();
	CAM->GetPosition(mState.cameraPosition);
	CAM->GetFocalPoint(mState.cameraFocalPoint);
	CAM->GetViewUp(mState.cameraViewUp);
	mState.cameraParallelScale = CAM->GetParallelScale();
	CAM->GetClippingRange(mState.cameraClippingRange);

	mState2 = mState;
	
	this->ClearCache();
}


void iAnimator::RestoreState(bool with_camera)
{
	mState = mState2;

	if(DATAREADER!=0 && mDebugFlag<2)
	{
		DATAREADER->LoadRecord(mState.currec,0,true);
	}

	if(XSECTION != 0) XSECTION->SetLocation(mState.xsecPos);
	this->GetViewModule()->ShowBoundingBox(mState.ifBoundingBox );
	this->GetViewModule()->ShowColorBars(mState.ifColorBars);
	this->GetViewModule()->ShowLabel(mState.ifTimeLabel);

	if(with_camera)
	{
		CAM->SetParallelProjection(mState.cameraProjection);
		CAM->SetPosition(mState.cameraPosition);
		CAM->SetFocalPoint(mState.cameraFocalPoint);
		CAM->SetViewUp(mState.cameraViewUp);
		CAM->SetParallelScale(mState.cameraParallelScale);
	}

	this->ClearCache();
}


void iAnimator::CopyState(iAnimator *anim)
{
	//
	//  Copy state but not camera
	//
	anim->SaveState();
	mState2 = anim->mState;
	this->RestoreState(false);

	this->ClearCache();
}


void iAnimator::SetStyle(int ma)
{ 
	if(ma>=0 && ma<=3) 
	{
		mState.mode = ma; 
		this->ClearCache();
	}
}


void iAnimator::SetNumberOfFrames(int na)
{
	if(na > 0)
	{
		mState.nframes = na;
		this->ClearCache();
	}
}


void iAnimator::SetPhi(float va)
{
	mState.dphi = va; 
	this->ClearCache();
}


void iAnimator::SetTheta(float va)
{
	mState.dtheta = va; 
	this->ClearCache();
}


void iAnimator::SetZoom(float va)
{
	mState.dscale = va; 
	this->ClearCache();
}


void iAnimator::SetRoll(float va)
{
	mState.droll = va; 
	this->ClearCache();
}


void iAnimator::SetCrossSectionSpeed(float va)
{
	mState.slideSpeed = va; 
	this->ClearCache();
}


void iAnimator::SetFlybySpeed(float va)
{
	mState.flybySpeed = va; 
	this->ClearCache();
}


void iAnimator::SetNumberOfBlendedFrames(int na)
{
	if(na >= 0)
	{
		mState.nBlendedFrames = na; 
		this->ClearCache();
	}
}


void iAnimator::SetNumberOfTransitionFrames(int na)
{
	if(na >= 0)
	{
		mState.nTransitionFrames = na; 
		this->ClearCache();
	}
}


void iAnimator::SetNumberOfTitlePageFrames(int n)
{
	if(n >= 0)
	{
		mTitlePageNumFrames = n; 
		this->ClearCache();
	}
}


void iAnimator::SetNumberOfTitlePageBlendedFrames(int n)
{
	if(n >= 0)
	{
		mTitlePageNumBlendedFrames = n; 
		this->ClearCache();
	}
}


void iAnimator::SetLogoLocation(int n)
{
	if(n>0 && n<5)
	{
		mLogoLocation = n; 
		this->ClearCache();
	}
}


void iAnimator::SetLogoOpacity(float v)
{
	mLogoOpacity = v; 
	this->ClearCache();
}


bool iAnimator::SetLogoFile(const iString &s)
{
	this->ClearCache();
	bool ok = mLogoImage.LoadFromFile(s);
	if(ok) mLogoFile = s; else 
	{ 
		mLogoFile.Clear();
		mLogoImage.Clear();
	}
	return ok;
}


bool iAnimator::SetTitlePageFile(const iString &s)
{
	this->ClearCache();
	bool ok = mTitlePageImage.LoadFromFile(s);
	if(ok) mTitlePageFile = s; else 
	{
		mTitlePageFile.Clear();
		mTitlePageImage.Clear();
	}
	return ok;
}


bool iAnimator::SetScriptFile(const iString &name)
{
	iString fname(name), text;

	mUsingScript = false;
	if(!name.IsEmpty())
	{
		iDirectory::ExpandFileName(fname,this->GetViewModule()->GetControlModule()->GetShell()->GetEnvironment(Environment::Script));

		iFile f(fname);
		if(!f.Open(iFile::_Read,iFile::_Text))
		{
			this->GetErrorStatus()->Set("Animation script file is not found.");
			return false;
		}

		iString st, tmp;
		while(f.ReadLine(tmp)) text += tmp + "\n";
		if(text.IsEmpty())
		{
			this->GetErrorStatus()->Set("Animation script file is empty or unreadable.");
			f.Close();
			return false;
		}

		f.Close();
		mUsingScript = true;
	}

	mAnimatorScript->SetText(text);
	mScriptFile = fname;
	this->ClearCache();

	return true;
}


bool iAnimator::Frame(bool dumpImage)
{
	static float Pi = 3.1415927;
	double xc[3], x0[3];
	float v0[3], r0[3], r1[3], vA[3], vB[3];
	int i;

	mNewRec = false;
	this->ClearCache();
	this->GetErrorStatus()->Clear();

	if(!mStartedRender)
	{
		if(!DATAREADER->IsFileAnimatable())
		{
			this->GetErrorStatus()->Set("File is not animatable.");
			return false;
		}
		this->GetViewModule()->JustifyLabelLeft(true);
		
		vtkMath::RandomSeed(mSeed);

		if(mState.mode == 2)
		{
			wData.dphl0 = mState.dphi;
			wData.dthl0 = mState.dtheta;
			wData.r = 0.0;
			wData.ramp = RAN_N;
		}
		
		if(mState.mode == 3)
		{
			for(i=0; i<3; i++)
			{
				wData.xc1[i] = 0.5*RAN_U;
				wData.xc2[i] = 0.5*RAN_U;
			}
			CAM->SetParallelProjection(0);
			CAM->GetPosition(wData.x);
			if(mState.cameraProjection == 1)
			{
				for(i=0; i<3; i++) wData.x[i] = 0.5*wData.x[i];
				CAM->SetPosition(wData.x);
			}
			else
			{
				CAM->GetFocalPoint(wData.xc1);
			}
			CAM->SetFocalPoint(wData.xc1);
			float d = LEN(wData.x);
			wData.v[0] = d*0.5*RAN_U;
			wData.v[1] = d*0.5*RAN_U;
			wData.v[2] = 0.0;
			wData.t = 0.0;
			wData.dt0 = 0.1*Pi;
			wData.dt = wData.dt0;
		}
		
		mCurFrame = mTotFrame = 0;

		mPrevRec = -1;
		mCurRec = DATAREADER->GetRecordNumber();
		mStartedRender = true;
	} 

	if(mCurFrame == mState.nframes)
	{
		this->GetErrorStatus()->Monitor(DATAREADER->GetErrorStatus());
		
		iProgressEventObserver *obs = DATAREADER->GetProgressEventObserver(DATAREADER->GetFileSetDataType());
		if(obs != 0) obs->AttachScript(mAnimatorScript);
		DATAREADER->LoadRecord(-1,1,mDebugFlag>1);
		if(obs != 0) obs->AttachScript(0);

		if(this->GetErrorStatus()->IsError())
		{
			if(this->GetErrorStatus()->Level()==10)
			{
				this->GetErrorStatus()->Clear();
				if(mUsingScript)
				{
					this->GetErrorStatus()->Set("Not enough data files to complete the script.",-1);
				}
			}
			return false;
		}
		this->GetViewModule()->UpdateLabel();
		mNewRec = true;
		mPrevRec = mCurRec;
		mCurRec = DATAREADER->GetRecordNumber();
		mCurFrame = 0;
	} 
		
	if(this->GetErrorStatus()->IsError()) 
	{
		return false;
	}

	mCurFrame++;
	mTotFrame++;

	if(fabs(mState.slideSpeed)>1.0e-30 && XSECTION!=0)
	{
		double p = XSECTION->GetLocation();
		p = p + mState.slideSpeed;
		XSECTION->SetLocation(p);
		if(XSECTION->GetOverTheEdgeFlag()) mState.slideSpeed = -mState.slideSpeed;
	}
	//
	//	Add transformations for rotate & tumble
	//	
	if(mState.mode==1 || mState.mode==2)
	{		
		CAM->Azimuth(-mState.dphi);
		CAM->Elevation(-mState.dtheta);
		CAM->Zoom(mState.dscale);
		CAM->Roll(mState.droll);
		CAM->OrthogonalizeViewUp();

		if(mState.mode == 2)
		{
			wData.r = wData.r + mRandStep;
			float cr = cos(wData.r*wData.ramp);
			float sr = sin(wData.r*wData.ramp);
			mState.dphi =  wData.dphl0*cr + wData.dthl0*sr;
			mState.dtheta =  wData.dthl0*cr - wData.dphl0*sr;
			if(wData.r > 1.0)
			{
				wData.r = 0.0;
				wData.ramp = RAN_N;
				wData.dphl0 =  mState.dphi;
				wData.dthl0 =  mState.dtheta;
			}
		}
	}
	//
	//  Add transformations for flyby
	//
	if(mState.mode == 3)
	{
		for(i=0; i<3; i++)
		{
			xc[i] = wData.xc1[i] + (wData.xc2[i]-wData.xc1[i])*wData.t;
			x0[i] = wData.x[i];
			v0[i] = wData.v[i];
		}

		CAM->SetFocalPoint(xc);

		wData.dt *= 2;
		float d0, d1, cot, sot;
		do
		{
			wData.dt *= 0.5;

			for(i=0; i<3; i++)
			{
				r0[i] = x0[i] - xc[i];
				vA[i] = r0[i];
				vB[i] = v0[i];
			}

			cot = cos(wData.dt);
			sot = sin(wData.dt);
			for(i=0; i<3; i++) r1[i] = vA[i]*cot + vB[i]*sot - r0[i];

			d0 = d1 = 0.0;
			for(i=0; i<3; i++)
			{
				d0 = d0 + r0[i]*r0[i];
				d1 = d1 + r1[i]*r1[i];
			}
			d1 = sqrt(d1/d0);
		}
		while(d1>mState.flybySpeed && wData.dt>0.001*wData.dt0);

		if(d1 < 0.2*mState.flybySpeed) wData.dt = 1.5*wData.dt;

		for(i=0; i<3; i++)
		{
			wData.v[i] = vB[i]*cot-vA[i]*sot;
			wData.x[i] = r0[i] + r1[i] + xc[i];
		}

		CAM->SetPosition(wData.x);
		CAM->OrthogonalizeViewUp();
		wData.t = wData.t + wData.dt;

		if(wData.t > 1.0)
		{
			wData.t = 0.0;
			for(i=0; i<3; i++)
			{
				wData.xc1[i] = wData.xc2[i];
				wData.xc2[i] = 0.5*RAN_U;
			}
		}
	}

	//
	//  Image data holder
	//
	iStereoImageArray images;

	//
	//  Create the primary image set
	//
	this->GetErrorStatus()->Monitor(this->GetViewModule()->GetErrorStatus(),true);
	this->GetViewModule()->CreateImages(images);
	if(this->GetErrorStatus()->IsError()) return false;

	//
	//  Title page
	//
	if(mTotFrame==1 && dumpImage && !mTitlePageImage.IsEmpty() && mDebugFlag==0)
	{
		iImage tmp = mTitlePageImage;
		//
		//  We must update composer here because there is no automatic way to call
		//  composer->Update() when vtkRenderWindow changes its size (it does not invoke an event).
		//
		this->GetViewModule()->GetControlModule()->GetImageComposer()->Update();
		tmp.Scale(this->GetViewModule()->GetFullImageWidth(),this->GetViewModule()->GetFullImageHeight());

		iStereoImageArray tmpset;
		tmpset.Copy(images);

		int n;
		for(n=0; n<mTitlePageNumFrames; n++)
		{
			tmpset.Fill(tmp);
			this->DumpImages(tmpset);
			if(this->GetErrorStatus()->IsError()) return false;
		}

		for(n=0; n<mTitlePageNumBlendedFrames; n++)
		{
			//
			//  Dissolve it; image already contains the correct image
			//
			tmpset.Copy(images);
			for(i=0; i<images.Size(); i++) tmpset[i].Blend(tmp,float(n)/mTitlePageNumBlendedFrames);

			this->DumpImages(tmpset);
			if(this->GetErrorStatus()->IsError()) return false;
		}
	}

	//
	//  Transition effects
	//
	bool doTransitionFrames = dumpImage && mTotFrame>0 && mPrevRec>0 && mCurFrame<=mState.nTransitionFrames && mState.nTransitionFrames>0 && mDebugFlag==0;
	if(doTransitionFrames)
	{
		iStereoImageArray tmpset;
		//
		//  Objects for transition effects (blending with previous record)
		//
		this->GetErrorStatus()->Monitor(DATAREADER->GetErrorStatus(),true);
		DATAREADER->LoadRecord(mPrevRec,0,false);
		if(this->GetErrorStatus()->IsError()) return false;

		this->GetViewModule()->UpdateLabel();
		this->GetErrorStatus()->Monitor(this->GetViewModule()->GetErrorStatus(),true);
		this->GetViewModule()->CreateImages(tmpset);
		if(this->GetErrorStatus()->IsError()) return false;

		float ops = (float)mCurFrame/mState.nTransitionFrames;

		this->GetErrorStatus()->Monitor(DATAREADER->GetErrorStatus(),true);
		DATAREADER->LoadRecord(mCurRec,0,false);	
		if(this->GetErrorStatus()->IsError()) return false;
	}

	//
	//  Blending of images
	//
	bool doBlendedFrames = dumpImage && mTotFrame >0 && mState.nBlendedFrames>0 && mDebugFlag==0;
	if(doBlendedFrames)
	{
		int k;
		//
		//  Update the image list
		//
		while(mBlenderBase.Size() < mState.nBlendedFrames)
		{
			iStereoImageArray *ptmparr = new iStereoImageArray; IERROR_ASSERT(ptmparr);
			ptmparr->Copy(images);
			mBlenderBase.Add(ptmparr);
		}
		while(mBlenderBase.Size() > mState.nBlendedFrames)
		{
			delete mBlenderBase.RemoveLast();
		}

		delete mBlenderBase[0];
		for(k=0; k<mBlenderBase.MaxIndex(); k++) mBlenderBase[k]->Copy(*mBlenderBase[k+1]);
		mBlenderBase.Last()->Copy(images);

		//
		//  Make sure that all the arrays are of the same size
		//
		for(k=0; k<mBlenderBase.MaxIndex(); k++)
		{
			while(mBlenderBase[k]->Size() > images.Size()) mBlenderBase[k]->RemoveLast();
			while(mBlenderBase[k]->Size() < images.Size()) mBlenderBase[k]->Add(images[mBlenderBase[k]->Size()]);
		}

		//
		//  Blend the arrays
		//
		int n = 1;
		float ops;
		images.Copy(*mBlenderBase[0]);
		for(k=1; k<mBlenderBase.Size(); k++)
		{
			n += (k+1);
			ops = float(k+1)/n;
			for(i=0; i<images.Size(); i++)
			{
				images[i].Blend((*mBlenderBase[k])[i],ops);
			}
		}
	}
	
	for(i=0; i<mFollowers.Size(); i++)
	{
		this->GetErrorStatus()->Monitor(mFollowers[i]->GetErrorStatus(),true,"Follower Animator from window #"+iString::FromNumber(1+mFollowers[i]->GetViewModule()->GetWindowNumber()));
		if(!mFollowers[i]->Frame(false)) break;
	}

	if(mDebugFlag==0 && dumpImage)
	{
		this->DumpImages(images);
	}

	return this->GetErrorStatus()->NoError();
}


//
//  Dump an image array, optionally adding a logo
//
void iAnimator::DumpImages(iStereoImageArray &images)
{
	if(images.Size()>0 && mDebugFlag==0 && !mLogoImage.IsEmpty())
	{
		//
		//  If the logo is more than 20% of the image, scale it down.
		//  Use tmp as a temp storage
		//
		iImage tmp = mLogoImage;
		if(tmp.Width()>images[0].Width()/5 || tmp.Height()>images[0].Height()/5)
		{
			tmp.Scale(images[0].Width()/5,images[0].Height()/5);
		}

		if(tmp.Width()>=2 && tmp.Height()>=2)
		{
			//
			//  tmp is now the proper logo image
			//
			int i, ioff, joff;
			//
			//  Where do we place the logo?
			//
			ioff = tmp.Width()/5;
			joff = tmp.Height()/5;
			switch(mLogoLocation)
			{
			case 1:
				{
					//  upper right corner 
					ioff = images[0].Width() - tmp.Width() - ioff;
					joff = images[0].Height() - tmp.Height() - joff;
					break;
				}
			case 2:
				{
					//  lower left right corner 
					break;
				}
			case 3:
				{
					//  lower right corner 
					ioff = images[0].Width() - tmp.Width() - ioff;
					break;
				}
			default:
				{
					//  upper left corner - the default choice
					joff = images[0].Height() - tmp.Height() - joff;
					break;
				}
			}
			for(i=0; i<images.Size(); i++) images[i].Overlay(ioff,joff,tmp,mLogoOpacity);
		}
	}

	this->GetErrorStatus()->Monitor(this->GetViewModule()->GetErrorStatus(),true);
	this->GetViewModule()->DumpImages(images,ImageType::AnimationFrame);
}


//
//  Two functions used in saving/restoring the state and in creating new instances with
//
void iAnimator::PackStateBody(iString &s) const
{
	this->PackValue(s,KeyDebugFlag(),mDebugFlag);
	this->PackValue(s,KeyDebugging(),mDebugFlag!=0);

	this->PackValue(s,KeyScriptFile(),mScriptFile);

	this->PackValue(s,KeyStyle(),mState.mode);
	this->PackValue(s,KeyNumberOfFrames(),mState.nframes);
	this->PackValue(s,KeyNumberOfBlendedFrames(),mState.nBlendedFrames);
	this->PackValue(s,KeyNumberOfTransitionFrames(),mState.nTransitionFrames);

	this->PackValue(s,KeyPhi(),mState.dphi);
	this->PackValue(s,KeyTheta(),mState.dtheta);
	this->PackValue(s,KeyRoll(),mState.droll);
	this->PackValue(s,KeyZoom(),mState.dscale);
	this->PackValue(s,KeyCrossSectionSpeed(),mState.slideSpeed);
	this->PackValue(s,KeyFlybySpeed(),mState.flybySpeed);

	this->PackValue(s,KeyTitlePageFile(),mTitlePageFile);
	this->PackValue(s,KeyNumberOfTitlePageFrames(),mTitlePageNumFrames);
	this->PackValue(s,KeyNumberOfTitlePageBlendedFrames(),mTitlePageNumBlendedFrames);
	this->PackValue(s,KeyLogoFile(),mLogoFile);
	this->PackValue(s,KeyLogoOpacity(),mLogoOpacity);
	this->PackValue(s,KeyLogoLocation(),mLogoLocation);
}


void iAnimator::UnPackStateBody(const iString &s)
{
	int i; float f; iString ss;

	if(this->UnPackValue(s,KeyDebugFlag(),i)) this->SetDebugFlag(i);

	if(this->UnPackValue(s,KeyNumberOfFrames(),i)) this->SetNumberOfFrames(i);
	if(this->UnPackValue(s,KeyNumberOfBlendedFrames(),i)) this->SetNumberOfBlendedFrames(i);
	if(this->UnPackValue(s,KeyNumberOfTransitionFrames(),i)) this->SetNumberOfTransitionFrames(i);

	if(this->UnPackValue(s,KeyPhi(),f)) this->SetPhi(f);
	if(this->UnPackValue(s,KeyTheta(),f)) this->SetTheta(f);
	if(this->UnPackValue(s,KeyRoll(),f)) this->SetRoll(f);
	if(this->UnPackValue(s,KeyZoom(),f)) this->SetZoom(f);
	if(this->UnPackValue(s,KeyCrossSectionSpeed(),f)) this->SetCrossSectionSpeed(f);
	if(this->UnPackValue(s,KeyFlybySpeed(),f)) this->SetFlybySpeed(f);

	if(this->UnPackValue(s,KeyScriptFile(),ss)) this->SetScriptFile(ss);
	if(this->UnPackValue(s,KeyTitlePageFile(),ss)) this->SetTitlePageFile(ss);
	if(this->UnPackValue(s,KeyNumberOfTitlePageFrames(),i)) this->SetNumberOfTitlePageFrames(i);
	if(this->UnPackValue(s,KeyNumberOfTitlePageBlendedFrames(),i)) this->SetNumberOfTitlePageBlendedFrames(i);
	if(this->UnPackValue(s,KeyLogoFile(),ss)) this->SetLogoFile(ss);
	if(this->UnPackValue(s,KeyLogoOpacity(),f)) this->SetLogoOpacity(f);
	if(this->UnPackValue(s,KeyLogoLocation(),i)) this->SetLogoLocation(i);

	if(this->UnPackValue(s,KeyStyle(),i)) this->SetStyle(i);
	
	//
	//  Action keys
	//
	iObject::ReportMissingKeys(false); //action keys are not part of the states

	if(this->UnPackValue(s,KeyLoadScriptFile(),ss)) this->SetScriptFile(ss);

	iObject::ReportMissingKeys(true);
}

