/**
 *
 * /brief RR iterator over wireformat DNS packet
 */
/*
 * Copyright (c) 2013, NLnet Labs, Verisign, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * * Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 * * Neither the names of the copyright holders nor the
 *   names of its contributors may be used to endorse or promote products
 *   derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL Verisign, Inc. BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef RR_ITER_H_
#define RR_ITER_H_

#include "getdns/getdns.h"
#include "rr-dict.h"
#include "gldns/pkthdr.h"
#include "gldns/gbuffer.h"

typedef struct _getdns_rr_iter {
	const uint8_t *pkt;
	const uint8_t *pkt_end;

	/* Which RR are we currently at */
	size_t   n;

	/* pos points to start of the owner name the RR.
	 * Or is NULL when there are no RR's left.
	 */
	const uint8_t *pos;

	/* rr_type will point to the rr_type right after the RR's owner name.
	 * rr_type is guaranteed to have a value when pos has a value
	 */
	const uint8_t *rr_type;

	/* nxt point to the owner name of the next RR or to pkt_end */
	const uint8_t *nxt;

} _getdns_rr_iter;

_getdns_rr_iter *_getdns_rr_iter_init(_getdns_rr_iter *i,
    const uint8_t *pkt, const size_t pkt_len);

_getdns_rr_iter *_getdns_single_rr_iter_init(_getdns_rr_iter *i,
    const uint8_t *wire, const size_t wire_len);

_getdns_rr_iter *_getdns_rr_iter_rewind(_getdns_rr_iter *i);

_getdns_rr_iter *_getdns_rr_iter_next(_getdns_rr_iter *i);

const uint8_t *_getdns_owner_if_or_as_decompressed(
    _getdns_rr_iter *i, uint8_t *ff_bytes, size_t *len);

static inline gldns_pkt_section
_getdns_rr_iter_section(_getdns_rr_iter *i)
{
	return !i->pkt ? (i->nxt - i->rr_type == 4 ? GLDNS_SECTION_QUESTION
	                                           : GLDNS_SECTION_ANSWER  )
             : i->n < GLDNS_QDCOUNT(i->pkt) ? GLDNS_SECTION_QUESTION
	     : i->n < GLDNS_QDCOUNT(i->pkt)
	            + GLDNS_ANCOUNT(i->pkt) ? GLDNS_SECTION_ANSWER
	     : i->n < GLDNS_QDCOUNT(i->pkt)
	            + GLDNS_ANCOUNT(i->pkt)
	            + GLDNS_NSCOUNT(i->pkt) ? GLDNS_SECTION_AUTHORITY
	     : i->n < GLDNS_QDCOUNT(i->pkt)
	            + GLDNS_ANCOUNT(i->pkt)
	            + GLDNS_NSCOUNT(i->pkt)
	            + GLDNS_ARCOUNT(i->pkt) ? GLDNS_SECTION_ADDITIONAL
	                                    : GLDNS_SECTION_ANY;
}

typedef struct piv_getdns_rdf_iter {
	const uint8_t           *pkt;
	const uint8_t           *pkt_end;
	const _getdns_rdata_def *rdd_pos;
	const _getdns_rdata_def *rdd_end;
	const _getdns_rdata_def *rdd_repeat;
	const uint8_t           *pos;
	const uint8_t           *end;
	const uint8_t           *nxt;
} _getdns_rdf_iter;

_getdns_rdf_iter *_getdns_rdf_iter_init(_getdns_rdf_iter *i,
    _getdns_rr_iter *rr);

_getdns_rdf_iter *_getdns_rdf_iter_next(_getdns_rdf_iter *i);

_getdns_rdf_iter *_getdns_rdf_iter_init_at(_getdns_rdf_iter *i,
    _getdns_rr_iter *rr, size_t pos);

const uint8_t *_getdns_rdf_if_or_as_decompressed(
    _getdns_rdf_iter *i, uint8_t *ff_bytes, size_t *len);

#endif
