/***********************************************************************
* ftermlinux-test.cpp - FTermLinux unit tests                          *
*                                                                      *
* This file is part of the FINAL CUT widget toolkit                    *
*                                                                      *
* Copyright 2019-2021 Markus Gans                                      *
*                                                                      *
* FINAL CUT is free software; you can redistribute it and/or modify    *
* it under the terms of the GNU Lesser General Public License as       *
* published by the Free Software Foundation; either version 3 of       *
* the License, or (at your option) any later version.                  *
*                                                                      *
* FINAL CUT is distributed in the hope that it will be useful, but     *
* WITHOUT ANY WARRANTY; without even the implied warranty of           *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
* GNU Lesser General Public License for more details.                  *
*                                                                      *
* You should have received a copy of the GNU Lesser General Public     *
* License along with this program.  If not, see                        *
* <http://www.gnu.org/licenses/>.                                      *
***********************************************************************/

#include <term.h>
#undef buttons  // from term.h

#include <limits>
#include <string>

#include <cppunit/BriefTestProgressListener.h>
#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/HelperMacros.h>
#include <cppunit/TestFixture.h>
#include <cppunit/TestResult.h>
#include <cppunit/TestResultCollector.h>
#include <cppunit/TestRunner.h>
#include <unistd.h>

#if defined(__linux__)
  #include <linux/version.h>
#endif

#include <conemu.h>
#include <final/final.h>

#define CPPUNIT_ASSERT_CSTRING(expected, actual) \
            check_c_string (expected, actual, CPPUNIT_SOURCELINE())

//----------------------------------------------------------------------
void check_c_string ( const char* s1
                    , const char* s2
                    , const CppUnit::SourceLine& sourceLine )
{
  if ( s1 == nullptr && s2 == nullptr )  // Strings are equal
    return;

  if ( s1 && s2 && std::strcmp (s1, s2) == 0 )  // Strings are equal
      return;

  ::CppUnit::Asserter::fail ("Strings are not equal", sourceLine);
}


namespace test
{

//----------------------------------------------------------------------
// class FSystemTest
//----------------------------------------------------------------------

class FSystemTest : public finalcut::FSystem
{
  public:
    // Typedefs and Enumerations
    struct ShiftState
    {
      uChar shift  : 1;
      uChar alt_gr : 1;
      uChar ctrl   : 1;
      uChar alt    : 1;
      uChar        : 4;  // padding bits
    };

    struct RGB
    {
      uChar red;
      uChar green;
      uChar blue;
    };

    struct ColorMap
    {
      RGB color[16];
    };

    enum class AccessMode
    {
      index,
      data
    };

    enum class Codeset
    {
      cp437,
      lat15
    };

    // Constructor
    FSystemTest();

    // Destructor
    ~FSystemTest() override;

    // Methods
    auto            inPortByte (uShort) -> uChar override;
    void             outPortByte (uChar, uShort) override;
    auto              isTTY (int) const -> int override;
    auto              ioctl (int, uLong, ...) -> int override;
    auto              open (const char*, int, ...) -> int override;
    auto              close (int) -> int override;
    auto            fopen (const char*, const char*) -> FILE* override;
    auto              fclose (FILE*) -> int override;
    auto              fputs (const char*, FILE*) -> int override;
    auto              putchar (int) -> int override;
    auto            getuid() -> uid_t override;
    auto            geteuid() -> uid_t override;
    auto              getpwuid_r ( uid_t, struct passwd*, char*
                                , size_t, struct passwd** ) -> int override;
    auto            realpath (const char*, char*) -> char* override;
    auto             getRGB (finalcut::FColor) -> RGB&;
    auto getConsoleFont() -> console_font_op&;
    auto      getShiftState() -> ShiftState&;
    auto     getCharacters() -> std::string&;
    void             setCodeset (Codeset);

  private:
    // Methods
    static void initVScreenInfo();
    static void initFScreenInfo();

    // Data members
    std::string characters;
    static ShiftState shift_state;
    static RGB terminal_color[16];
    static RGB defaultColor[16];
    static struct console_font_op terminal_font;
    static unimapdesc terminal_unicode_map;
    static struct fb_var_screeninfo fb_terminal_info;
    static struct fb_fix_screeninfo fb_terminal_fix_info;
    static bool vga_port_access;

    AccessMode attribute_controller_mode = AccessMode::index;
    uChar ac_address_register[21] = \
    {
      0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
      0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d,
      0x0e, 0x0f, 0x0c, 0x00, 0x0f, 0x08, 0x00
    };
    uChar ac_index{0};
    bool palette_addr_source_field{true};
    uChar port_3cc{0x67};  // Miscellaneous output
    uChar port_3da{0};     // Input status 1
    Codeset codeset{Codeset::cp437};
    static uChar vga8x16[];
    static struct unipair unicode_cp437_pairs[];
    static struct unipair unicode_lat15_pairs[];
};


// private data member of FSystemTest
//----------------------------------------------------------------------
uChar FSystemTest::vga8x16[] =
{
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 0
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7e, 0x81, 0xa5, 0x81, 0x81, 0xbd, 0x99, 0x81, 0x81, 0x7e, 0x00, 0x00, 0x00, 0x00,  // 1
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7e, 0xff, 0xdb, 0xff, 0xff, 0xc3, 0xe7, 0xff, 0xff, 0x7e, 0x00, 0x00, 0x00, 0x00,  // 2
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x6c, 0xfe, 0xfe, 0xfe, 0xfe, 0x7c, 0x38, 0x10, 0x00, 0x00, 0x00, 0x00,  // 3
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x10, 0x38, 0x7c, 0xfe, 0x7c, 0x38, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00,  // 4
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x18, 0x3c, 0x3c, 0xe7, 0xe7, 0xe7, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,  // 5
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x18, 0x3c, 0x7e, 0xff, 0xff, 0x7e, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,  // 6
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x3c, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 7
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xe7, 0xc3, 0xc3, 0xe7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,  // 8
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x3c, 0x66, 0x42, 0x42, 0x66, 0x3c, 0x00, 0x00, 0x00, 0x00, 0x00,  // 9
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xc3, 0x99, 0xbd, 0xbd, 0x99, 0xc3, 0xff, 0xff, 0xff, 0xff, 0xff,  // 10
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x1e, 0x0e, 0x1a, 0x32, 0x78, 0xcc, 0xcc, 0xcc, 0xcc, 0x78, 0x00, 0x00, 0x00, 0x00,  // 11
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x66, 0x66, 0x66, 0x66, 0x3c, 0x18, 0x7e, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,  // 12
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3f, 0x33, 0x3f, 0x30, 0x30, 0x30, 0x30, 0x70, 0xf0, 0xe0, 0x00, 0x00, 0x00, 0x00,  // 13
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7f, 0x63, 0x7f, 0x63, 0x63, 0x63, 0x63, 0x67, 0xe7, 0xe6, 0xc0, 0x00, 0x00, 0x00,  // 14
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x18, 0x18, 0xdb, 0x3c, 0xe7, 0x3c, 0xdb, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,  // 15
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x80, 0xc0, 0xe0, 0xf0, 0xf8, 0xfe, 0xf8, 0xf0, 0xe0, 0xc0, 0x80, 0x00, 0x00, 0x00, 0x00,  // 16
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x02, 0x06, 0x0e, 0x1e, 0x3e, 0xfe, 0x3e, 0x1e, 0x0e, 0x06, 0x02, 0x00, 0x00, 0x00, 0x00,  // 17
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x3c, 0x7e, 0x18, 0x18, 0x18, 0x7e, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,  // 18
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x00, 0x66, 0x66, 0x00, 0x00, 0x00, 0x00,  // 19
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7f, 0xdb, 0xdb, 0xdb, 0x7b, 0x1b, 0x1b, 0x1b, 0x1b, 0x1b, 0x00, 0x00, 0x00, 0x00,  // 20
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x7c, 0xc6, 0x60, 0x38, 0x6c, 0xc6, 0xc6, 0x6c, 0x38, 0x0c, 0xc6, 0x7c, 0x00, 0x00, 0x00,  // 21
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0xfe, 0xfe, 0xfe, 0x00, 0x00, 0x00, 0x00,  // 22
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x3c, 0x7e, 0x18, 0x18, 0x18, 0x7e, 0x3c, 0x18, 0x7e, 0x00, 0x00, 0x00, 0x00,  // 23
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x3c, 0x7e, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,  // 24
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x7e, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x00,  // 25
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x0c, 0xfe, 0x0c, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 26
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x60, 0xfe, 0x60, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 27
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0xc0, 0xc0, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 28
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x6c, 0xfe, 0x6c, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 29
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x10, 0x38, 0x38, 0x7c, 0x7c, 0xfe, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00,  // 30
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xfe, 0xfe, 0x7c, 0x7c, 0x38, 0x38, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00,  // 31
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 32
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x3c, 0x3c, 0x3c, 0x18, 0x18, 0x18, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,  // 33
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x66, 0x66, 0x66, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 34
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x6c, 0x6c, 0xfe, 0x6c, 0x6c, 0x6c, 0xfe, 0x6c, 0x6c, 0x00, 0x00, 0x00, 0x00,  // 35
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x7c, 0xc6, 0xc2, 0xc0, 0x7c, 0x06, 0x06, 0x86, 0xc6, 0x7c, 0x18, 0x18, 0x00, 0x00,  // 36
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xc2, 0xc6, 0x0c, 0x18, 0x30, 0x60, 0xc6, 0x86, 0x00, 0x00, 0x00, 0x00,  // 37
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x38, 0x6c, 0x6c, 0x38, 0x76, 0xdc, 0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,  // 38
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x30, 0x30, 0x30, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 39
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x0c, 0x18, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x18, 0x0c, 0x00, 0x00, 0x00, 0x00,  // 40
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x30, 0x18, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x18, 0x30, 0x00, 0x00, 0x00, 0x00,  // 41
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x66, 0x3c, 0xff, 0x3c, 0x66, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 42
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x7e, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 43
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x18, 0x30, 0x00, 0x00, 0x00,  // 44
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 45
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,  // 46
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x02, 0x06, 0x0c, 0x18, 0x30, 0x60, 0xc0, 0x80, 0x00, 0x00, 0x00, 0x00,  // 47
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x38, 0x6c, 0xc6, 0xc6, 0xd6, 0xd6, 0xc6, 0xc6, 0x6c, 0x38, 0x00, 0x00, 0x00, 0x00,  // 48
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x38, 0x78, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x7e, 0x00, 0x00, 0x00, 0x00,  // 49
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0x06, 0x0c, 0x18, 0x30, 0x60, 0xc0, 0xc6, 0xfe, 0x00, 0x00, 0x00, 0x00,  // 50
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0x06, 0x06, 0x3c, 0x06, 0x06, 0x06, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,  // 51
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x0c, 0x1c, 0x3c, 0x6c, 0xcc, 0xfe, 0x0c, 0x0c, 0x0c, 0x1e, 0x00, 0x00, 0x00, 0x00,  // 52
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfe, 0xc0, 0xc0, 0xc0, 0xfc, 0x06, 0x06, 0x06, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,  // 53
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x38, 0x60, 0xc0, 0xc0, 0xfc, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,  // 54
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfe, 0xc6, 0x06, 0x06, 0x0c, 0x18, 0x30, 0x30, 0x30, 0x30, 0x00, 0x00, 0x00, 0x00,  // 55
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,  // 56
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0x7e, 0x06, 0x06, 0x06, 0x0c, 0x78, 0x00, 0x00, 0x00, 0x00,  // 57
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,  // 58
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x18, 0x18, 0x30, 0x00, 0x00, 0x00, 0x00,  // 59
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x06, 0x0c, 0x18, 0x30, 0x60, 0x30, 0x18, 0x0c, 0x06, 0x00, 0x00, 0x00, 0x00,  // 60
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0x00, 0x00, 0x7e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 61
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x60, 0x30, 0x18, 0x0c, 0x06, 0x0c, 0x18, 0x30, 0x60, 0x00, 0x00, 0x00, 0x00,  // 62
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0x0c, 0x18, 0x18, 0x18, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,  // 63
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xde, 0xde, 0xde, 0xdc, 0xc0, 0x7c, 0x00, 0x00, 0x00, 0x00,  // 64
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x10, 0x38, 0x6c, 0xc6, 0xc6, 0xfe, 0xc6, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,  // 65
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfc, 0x66, 0x66, 0x66, 0x7c, 0x66, 0x66, 0x66, 0x66, 0xfc, 0x00, 0x00, 0x00, 0x00,  // 66
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x66, 0xc2, 0xc0, 0xc0, 0xc0, 0xc0, 0xc2, 0x66, 0x3c, 0x00, 0x00, 0x00, 0x00,  // 67
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xf8, 0x6c, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x6c, 0xf8, 0x00, 0x00, 0x00, 0x00,  // 68
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfe, 0x66, 0x62, 0x68, 0x78, 0x68, 0x60, 0x62, 0x66, 0xfe, 0x00, 0x00, 0x00, 0x00,  // 69
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfe, 0x66, 0x62, 0x68, 0x78, 0x68, 0x60, 0x60, 0x60, 0xf0, 0x00, 0x00, 0x00, 0x00,  // 70
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x66, 0xc2, 0xc0, 0xc0, 0xde, 0xc6, 0xc6, 0x66, 0x3a, 0x00, 0x00, 0x00, 0x00,  // 71
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xfe, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,  // 72
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,  // 73
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x1e, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0xcc, 0xcc, 0xcc, 0x78, 0x00, 0x00, 0x00, 0x00,  // 74
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xe6, 0x66, 0x66, 0x6c, 0x78, 0x78, 0x6c, 0x66, 0x66, 0xe6, 0x00, 0x00, 0x00, 0x00,  // 75
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xf0, 0x60, 0x60, 0x60, 0x60, 0x60, 0x60, 0x62, 0x66, 0xfe, 0x00, 0x00, 0x00, 0x00,  // 76
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xee, 0xfe, 0xfe, 0xd6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,  // 77
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xe6, 0xf6, 0xfe, 0xde, 0xce, 0xc6, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,  // 78
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,  // 79
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfc, 0x66, 0x66, 0x66, 0x7c, 0x60, 0x60, 0x60, 0x60, 0xf0, 0x00, 0x00, 0x00, 0x00,  // 80
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xd6, 0xde, 0x7c, 0x0c, 0x0e, 0x00, 0x00,  // 81
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfc, 0x66, 0x66, 0x66, 0x7c, 0x6c, 0x66, 0x66, 0x66, 0xe6, 0x00, 0x00, 0x00, 0x00,  // 82
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7c, 0xc6, 0xc6, 0x60, 0x38, 0x0c, 0x06, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,  // 83
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x7e, 0x7e, 0x5a, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,  // 84
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,  // 85
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x6c, 0x38, 0x10, 0x00, 0x00, 0x00, 0x00,  // 86
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xd6, 0xd6, 0xd6, 0xfe, 0xee, 0x6c, 0x00, 0x00, 0x00, 0x00,  // 87
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0xc6, 0x6c, 0x7c, 0x38, 0x38, 0x7c, 0x6c, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,  // 88
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x66, 0x66, 0x66, 0x66, 0x3c, 0x18, 0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,  // 89
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfe, 0xc6, 0x86, 0x0c, 0x18, 0x30, 0x60, 0xc2, 0xc6, 0xfe, 0x00, 0x00, 0x00, 0x00,  // 90
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x3c, 0x00, 0x00, 0x00, 0x00,  // 91
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x80, 0xc0, 0xe0, 0x70, 0x38, 0x1c, 0x0e, 0x06, 0x02, 0x00, 0x00, 0x00, 0x00,  // 92
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x3c, 0x00, 0x00, 0x00, 0x00,  // 93
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x10, 0x38, 0x6c, 0xc6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 94
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00,  // 95
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x30, 0x30, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 96
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,  // 97
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xe0, 0x60, 0x60, 0x78, 0x6c, 0x66, 0x66, 0x66, 0x66, 0x7c, 0x00, 0x00, 0x00, 0x00,  // 98
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc0, 0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,  // 99
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x1c, 0x0c, 0x0c, 0x3c, 0x6c, 0xcc, 0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,  // 100
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xfe, 0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,  // 101
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x38, 0x6c, 0x64, 0x60, 0xf0, 0x60, 0x60, 0x60, 0x60, 0xf0, 0x00, 0x00, 0x00, 0x00,  // 102
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0x7c, 0x0c, 0xcc, 0x78, 0x00,  // 103
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xe0, 0x60, 0x60, 0x6c, 0x76, 0x66, 0x66, 0x66, 0x66, 0xe6, 0x00, 0x00, 0x00, 0x00,  // 104
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x18, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,  // 105
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x06, 0x06, 0x00, 0x0e, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x66, 0x66, 0x3c, 0x00,  // 106
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xe0, 0x60, 0x60, 0x66, 0x6c, 0x78, 0x78, 0x6c, 0x66, 0xe6, 0x00, 0x00, 0x00, 0x00,  // 107
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,  // 108
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xec, 0xfe, 0xd6, 0xd6, 0xd6, 0xd6, 0xc6, 0x00, 0x00, 0x00, 0x00,  // 109
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xdc, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x00, 0x00, 0x00, 0x00,  // 110
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,  // 111
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xdc, 0x66, 0x66, 0x66, 0x66, 0x66, 0x7c, 0x60, 0x60, 0xf0, 0x00,  // 112
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0x7c, 0x0c, 0x0c, 0x1e, 0x00,  // 113
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xdc, 0x76, 0x66, 0x60, 0x60, 0x60, 0xf0, 0x00, 0x00, 0x00, 0x00,  // 114
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0x60, 0x38, 0x0c, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,  // 115
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x10, 0x30, 0x30, 0xfc, 0x30, 0x30, 0x30, 0x30, 0x36, 0x1c, 0x00, 0x00, 0x00, 0x00,  // 116
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,  // 117
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x66, 0x66, 0x66, 0x66, 0x66, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x00,  // 118
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0xc6, 0xd6, 0xd6, 0xd6, 0xfe, 0x6c, 0x00, 0x00, 0x00, 0x00,  // 119
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0x6c, 0x38, 0x38, 0x38, 0x6c, 0xc6, 0x00, 0x00, 0x00, 0x00,  // 120
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7e, 0x06, 0x0c, 0xf8, 0x00,  // 121
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0xcc, 0x18, 0x30, 0x60, 0xc6, 0xfe, 0x00, 0x00, 0x00, 0x00,  // 122
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x0e, 0x18, 0x18, 0x18, 0x70, 0x18, 0x18, 0x18, 0x18, 0x0e, 0x00, 0x00, 0x00, 0x00,  // 123
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 0x00, 0x18, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,  // 124
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x70, 0x18, 0x18, 0x18, 0x0e, 0x18, 0x18, 0x18, 0x18, 0x70, 0x00, 0x00, 0x00, 0x00,  // 125
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x76, 0xdc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 126
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x10, 0x38, 0x6c, 0xc6, 0xc6, 0xc6, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00,  // 127
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3c, 0x66, 0xc2, 0xc0, 0xc0, 0xc0, 0xc2, 0x66, 0x3c, 0x0c, 0x06, 0x7c, 0x00, 0x00,  // 128
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xcc, 0x00, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,  // 129
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x0c, 0x18, 0x30, 0x00, 0x7c, 0xc6, 0xfe, 0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,  // 130
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x10, 0x38, 0x6c, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,  // 131
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xcc, 0x00, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,  // 132
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x60, 0x30, 0x18, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,  // 133
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x38, 0x6c, 0x38, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,  // 134
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x3c, 0x66, 0x60, 0x60, 0x66, 0x3c, 0x0c, 0x06, 0x3c, 0x00, 0x00, 0x00,  // 135
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x10, 0x38, 0x6c, 0x00, 0x7c, 0xc6, 0xfe, 0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,  // 136
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0x00, 0x00, 0x7c, 0xc6, 0xfe, 0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,  // 137
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x60, 0x30, 0x18, 0x00, 0x7c, 0xc6, 0xfe, 0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,  // 138
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x66, 0x00, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,  // 139
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x18, 0x3c, 0x66, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,  // 140
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x60, 0x30, 0x18, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,  // 141
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xc6, 0x00, 0x10, 0x38, 0x6c, 0xc6, 0xc6, 0xfe, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,  // 142
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x38, 0x6c, 0x38, 0x00, 0x38, 0x6c, 0xc6, 0xc6, 0xfe, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,  // 143
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x30, 0x60, 0x00, 0xfe, 0x66, 0x60, 0x7c, 0x60, 0x60, 0x66, 0xfe, 0x00, 0x00, 0x00, 0x00,  // 144
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xcc, 0x76, 0x36, 0x7e, 0xd8, 0xd8, 0x6e, 0x00, 0x00, 0x00, 0x00,  // 145
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x3e, 0x6c, 0xcc, 0xcc, 0xfe, 0xcc, 0xcc, 0xcc, 0xcc, 0xce, 0x00, 0x00, 0x00, 0x00,  // 146
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x10, 0x38, 0x6c, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,  // 147
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,  // 148
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x60, 0x30, 0x18, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,  // 149
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x30, 0x78, 0xcc, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,  // 150
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x60, 0x30, 0x18, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,  // 151
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xc6, 0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7e, 0x06, 0x0c, 0x78, 0x00,  // 152
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xc6, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,  // 153
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xc6, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,  // 154
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x18, 0x18, 0x3c, 0x66, 0x60, 0x60, 0x60, 0x66, 0x3c, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,  // 155
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x38, 0x6c, 0x64, 0x60, 0xf0, 0x60, 0x60, 0x60, 0x60, 0xe6, 0xfc, 0x00, 0x00, 0x00, 0x00,  // 156
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x66, 0x66, 0x3c, 0x18, 0x7e, 0x18, 0x7e, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,  // 157
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xf8, 0xcc, 0xcc, 0xf8, 0xc4, 0xcc, 0xde, 0xcc, 0xcc, 0xcc, 0xc6, 0x00, 0x00, 0x00, 0x00,  // 158
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x0e, 0x1b, 0x18, 0x18, 0x18, 0x7e, 0x18, 0x18, 0x18, 0x18, 0x18, 0xd8, 0x70, 0x00, 0x00,  // 159
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x18, 0x30, 0x60, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,  // 160
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x0c, 0x18, 0x30, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00,  // 161
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x18, 0x30, 0x60, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,  // 162
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x18, 0x30, 0x60, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00,  // 163
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x76, 0xdc, 0x00, 0xdc, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x00, 0x00, 0x00, 0x00,  // 164
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x76, 0xdc, 0x00, 0xc6, 0xe6, 0xf6, 0xfe, 0xde, 0xce, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,  // 165
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x3c, 0x6c, 0x6c, 0x3e, 0x00, 0x7e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 166
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x38, 0x6c, 0x6c, 0x38, 0x00, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 167
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x30, 0x30, 0x00, 0x30, 0x30, 0x60, 0xc0, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00,  // 168
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0xc0, 0xc0, 0xc0, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00,  // 169
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x06, 0x06, 0x06, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00,  // 170
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xc0, 0xc0, 0xc2, 0xc6, 0xcc, 0x18, 0x30, 0x60, 0xdc, 0x86, 0x0c, 0x18, 0x3e, 0x00, 0x00,  // 171
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xc0, 0xc0, 0xc2, 0xc6, 0xcc, 0x18, 0x30, 0x66, 0xce, 0x9e, 0x3e, 0x06, 0x06, 0x00, 0x00,  // 172
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x18, 0x18, 0x00, 0x18, 0x18, 0x18, 0x3c, 0x3c, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x00,  // 173
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x36, 0x6c, 0xd8, 0x6c, 0x36, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 174
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xd8, 0x6c, 0x36, 0x6c, 0xd8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 175
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x11, 0x44, 0x11, 0x44, 0x11, 0x44, 0x11, 0x44, 0x11, 0x44, 0x11, 0x44, 0x11, 0x44, 0x11, 0x44,  // 176
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa,  // 177
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77,  // 178
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,  // 179
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xf8, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,  // 180
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x18, 0xf8, 0x18, 0xf8, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,  // 181
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xf6, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,  // 182
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,  // 183
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x18, 0xf8, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,  // 184
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x36, 0x36, 0x36, 0x36, 0x36, 0xf6, 0x06, 0xf6, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,  // 185
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,  // 186
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x06, 0xf6, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,  // 187
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x36, 0x36, 0x36, 0x36, 0x36, 0xf6, 0x06, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 188
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 189
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x18, 0xf8, 0x18, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 190
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,  // 191
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x1f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 192
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 193
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,  // 194
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x1f, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,  // 195
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 196
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xff, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,  // 197
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x1f, 0x18, 0x1f, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,  // 198
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x37, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,  // 199
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x37, 0x30, 0x3f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 200
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0x30, 0x37, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,  // 201
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x36, 0x36, 0x36, 0x36, 0x36, 0xf7, 0x00, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 202
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0xf7, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,  // 203
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x37, 0x30, 0x37, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,  // 204
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 205
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x36, 0x36, 0x36, 0x36, 0x36, 0xf7, 0x00, 0xf7, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,  // 206
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x18, 0xff, 0x00, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 207
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 208
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0xff, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,  // 209
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,  // 210
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x3f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 211
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x1f, 0x18, 0x1f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 212
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x18, 0x1f, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,  // 213
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,  // 214
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xff, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36,  // 215
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x18, 0xff, 0x18, 0xff, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,  // 216
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 217
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,  // 218
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,  // 219
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,  // 220
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,  // 221
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f,  // 222
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 223
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xdc, 0xd8, 0xd8, 0xd8, 0xdc, 0x76, 0x00, 0x00, 0x00, 0x00,  // 224
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x78, 0xcc, 0xcc, 0xcc, 0xd8, 0xcc, 0xc6, 0xc6, 0xc6, 0xcc, 0x00, 0x00, 0x00, 0x00,  // 225
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0xfe, 0xc6, 0xc6, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0x00, 0x00, 0x00, 0x00,  // 226
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xfe, 0x6c, 0x6c, 0x6c, 0x6c, 0x6c, 0x6c, 0x6c, 0x00, 0x00, 0x00, 0x00,  // 227
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0xfe, 0xc6, 0x60, 0x30, 0x18, 0x30, 0x60, 0xc6, 0xfe, 0x00, 0x00, 0x00, 0x00,  // 228
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0xd8, 0xd8, 0xd8, 0xd8, 0xd8, 0x70, 0x00, 0x00, 0x00, 0x00,  // 229
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x66, 0x66, 0x66, 0x66, 0x66, 0x7c, 0x60, 0x60, 0xc0, 0x00, 0x00, 0x00,  // 230
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x76, 0xdc, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,  // 231
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x7e, 0x18, 0x3c, 0x66, 0x66, 0x66, 0x3c, 0x18, 0x7e, 0x00, 0x00, 0x00, 0x00,  // 232
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x38, 0x6c, 0xc6, 0xc6, 0xfe, 0xc6, 0xc6, 0x6c, 0x38, 0x00, 0x00, 0x00, 0x00,  // 233
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x38, 0x6c, 0xc6, 0xc6, 0xc6, 0x6c, 0x6c, 0x6c, 0x6c, 0xee, 0x00, 0x00, 0x00, 0x00,  // 234
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x1e, 0x30, 0x18, 0x0c, 0x3e, 0x66, 0x66, 0x66, 0x66, 0x3c, 0x00, 0x00, 0x00, 0x00,  // 235
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0xdb, 0xdb, 0xdb, 0x7e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 236
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x03, 0x06, 0x7e, 0xdb, 0xdb, 0xf3, 0x7e, 0x60, 0xc0, 0x00, 0x00, 0x00, 0x00,  // 237
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x1c, 0x30, 0x60, 0x60, 0x7c, 0x60, 0x60, 0x60, 0x30, 0x1c, 0x00, 0x00, 0x00, 0x00,  // 238
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00,  // 239
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0xfe, 0x00, 0x00, 0xfe, 0x00, 0x00, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00,  // 240
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x7e, 0x18, 0x18, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0x00,  // 241
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x30, 0x18, 0x0c, 0x06, 0x0c, 0x18, 0x30, 0x00, 0x7e, 0x00, 0x00, 0x00, 0x00,  // 242
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x0c, 0x18, 0x30, 0x60, 0x30, 0x18, 0x0c, 0x00, 0x7e, 0x00, 0x00, 0x00, 0x00,  // 243
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x0e, 0x1b, 0x1b, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,  // 244
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xd8, 0xd8, 0xd8, 0x70, 0x00, 0x00, 0x00, 0x00,  // 245
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x7e, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,  // 246
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xdc, 0x00, 0x76, 0xdc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 247
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x38, 0x6c, 0x6c, 0x38, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 248
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 249
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 250
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x0f, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0xec, 0x6c, 0x6c, 0x3c, 0x1c, 0x00, 0x00, 0x00, 0x00,  // 251
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0xd8, 0x6c, 0x6c, 0x6c, 0x6c, 0x6c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 252
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x70, 0xd8, 0x30, 0x60, 0xc8, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 253
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x7c, 0x7c, 0x7c, 0x7c, 0x7c, 0x7c, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00,  // 254
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  // 255
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

struct unipair FSystemTest::unicode_cp437_pairs[] =
{
  //  .----------- unicode
  //  |      .---- fontpos
  //  |      |
  {0x0020, 0x20},
  {0x0021, 0x21},
  {0x0022, 0x22},
  {0x0023, 0x23},
  {0x0024, 0x24},
  {0x0025, 0x25},
  {0x0026, 0x26},
  {0x0027, 0x27},
  {0x0028, 0x28},
  {0x0029, 0x29},
  {0x002a, 0x2a},
  {0x002b, 0x2b},
  {0x002c, 0x2c},
  {0x002d, 0x2d},
  {0x002e, 0x2e},
  {0x002f, 0x2f},
  {0x0030, 0x30},
  {0x0031, 0x31},
  {0x0032, 0x32},
  {0x0033, 0x33},
  {0x0034, 0x34},
  {0x0035, 0x35},
  {0x0036, 0x36},
  {0x0037, 0x37},
  {0x0038, 0x38},
  {0x0039, 0x39},
  {0x003a, 0x3a},
  {0x003b, 0x3b},
  {0x003c, 0x3c},
  {0x003d, 0x3d},
  {0x003e, 0x3e},
  {0x003f, 0x3f},
  {0x0040, 0x40},
  {0x0041, 0x41},
  {0x0042, 0x42},
  {0x0043, 0x43},
  {0x0044, 0x44},
  {0x0045, 0x45},
  {0x0046, 0x46},
  {0x0047, 0x47},
  {0x0048, 0x48},
  {0x0049, 0x49},
  {0x004a, 0x4a},
  {0x004b, 0x4b},
  {0x004c, 0x4c},
  {0x004d, 0x4d},
  {0x004e, 0x4e},
  {0x004f, 0x4f},
  {0x0050, 0x50},
  {0x0051, 0x51},
  {0x0052, 0x52},
  {0x0053, 0x53},
  {0x0054, 0x54},
  {0x0055, 0x55},
  {0x0056, 0x56},
  {0x0057, 0x57},
  {0x0058, 0x58},
  {0x0059, 0x59},
  {0x005a, 0x5a},
  {0x005b, 0x5b},
  {0x005c, 0x5c},
  {0x005d, 0x5d},
  {0x005e, 0x5e},
  {0x005f, 0x5f},
  {0x0060, 0x60},
  {0x0061, 0x61},
  {0x0062, 0x62},
  {0x0063, 0x63},
  {0x0064, 0x64},
  {0x0065, 0x65},
  {0x0066, 0x66},
  {0x0067, 0x67},
  {0x0068, 0x68},
  {0x0069, 0x69},
  {0x006a, 0x6a},
  {0x006b, 0x6b},
  {0x006c, 0x6c},
  {0x006d, 0x6d},
  {0x006e, 0x6e},
  {0x006f, 0x6f},
  {0x0070, 0x70},
  {0x0071, 0x71},
  {0x0072, 0x72},
  {0x0073, 0x73},
  {0x0074, 0x74},
  {0x0075, 0x75},
  {0x0076, 0x76},
  {0x0077, 0x77},
  {0x0078, 0x78},
  {0x0079, 0x79},
  {0x007a, 0x7a},
  {0x007b, 0x7b},
  {0x007c, 0x7c},
  {0x007d, 0x7d},
  {0x007e, 0x7e},
  {0x00a0, 0xff},
  {0x00a1, 0xad},
  {0x00a2, 0x9b},
  {0x00a3, 0x9c},
  {0x00a5, 0x9d},
  {0x00a7, 0x15},
  {0x00aa, 0xa6},
  {0x00ab, 0xae},
  {0x00ac, 0xaa},
  {0x00b0, 0xf8},
  {0x00b1, 0xf1},
  {0x00b2, 0xfd},
  {0x00b5, 0xe6},
  {0x00b6, 0x14},
  {0x00b7, 0xfa},
  {0x00ba, 0xa7},
  {0x00bb, 0xaf},
  {0x00bc, 0xac},
  {0x00bd, 0xab},
  {0x00bf, 0xa8},
  {0x00c4, 0x8e},
  {0x00c5, 0x8f},
  {0x00c6, 0x92},
  {0x00c7, 0x80},
  {0x00c9, 0x90},
  {0x00d1, 0xa5},
  {0x00d6, 0x99},
  {0x00dc, 0x9a},
  {0x00df, 0xe1},
  {0x00e0, 0x85},
  {0x00e1, 0xa0},
  {0x00e2, 0x83},
  {0x00e4, 0x84},
  {0x00e5, 0x86},
  {0x00e6, 0x91},
  {0x00e7, 0x87},
  {0x00e8, 0x8a},
  {0x00e9, 0x82},
  {0x00ea, 0x88},
  {0x00eb, 0x89},
  {0x00ec, 0x8d},
  {0x00ed, 0xa1},
  {0x00ee, 0x8c},
  {0x00ef, 0x8b},
  {0x00f1, 0xa4},
  {0x00f2, 0x95},
  {0x00f3, 0xa2},
  {0x00f4, 0x93},
  {0x00f6, 0x94},
  {0x00f7, 0xf6},
  {0x00f9, 0x97},
  {0x00fa, 0xa3},
  {0x00fb, 0x96},
  {0x00fc, 0x81},
  {0x00ff, 0x98},
  {0x0192, 0x9f},
  {0x0393, 0xe2},
  {0x0398, 0xe9},
  {0x03a3, 0xe4},
  {0x03a6, 0xe8},
  {0x03a9, 0xea},
  {0x03b1, 0xe0},
  {0x03b2, 0xe1},
  {0x03b4, 0xeb},
  {0x03b5, 0xee},
  {0x03bc, 0xe6},
  {0x03c0, 0xe3},
  {0x03c3, 0xe5},
  {0x03c4, 0xe7},
  {0x03c6, 0xed},
  {0x2008, 0x00},
  {0x2022, 0x07},
  {0x203c, 0x13},
  {0x207f, 0xfc},
  {0x20a7, 0x9e},
  {0x2126, 0xea},
  {0x212b, 0x8f},
  {0x2190, 0x1b},
  {0x2191, 0x18},
  {0x2192, 0x1a},
  {0x2193, 0x19},
  {0x2194, 0x1d},
  {0x2195, 0x12},
  {0x21a8, 0x17},
  {0x2205, 0xed},
  {0x2208, 0xee},
  {0x220e, 0xfe},
  {0x2219, 0xf9},
  {0x221a, 0xfb},
  {0x221e, 0xec},
  {0x221f, 0x1c},
  {0x2229, 0xef},
  {0x2248, 0xf7},
  {0x2261, 0xf0},
  {0x2264, 0xf3},
  {0x2265, 0xf2},
  {0x22c5, 0xf9},
  {0x2300, 0xed},
  {0x2302, 0x7f},
  {0x2310, 0xa9},
  {0x2319, 0x1c},
  {0x2320, 0xf4},
  {0x2321, 0xf5},
  {0x2500, 0xc4},
  {0x2502, 0xb3},
  {0x250c, 0xda},
  {0x2510, 0xbf},
  {0x2514, 0xc0},
  {0x2518, 0xd9},
  {0x251c, 0xc3},
  {0x2524, 0xb4},
  {0x252c, 0xc2},
  {0x2534, 0xc1},
  {0x253c, 0xc5},
  {0x2550, 0xcd},
  {0x2551, 0xba},
  {0x2552, 0xd5},
  {0x2553, 0xd6},
  {0x2554, 0xc9},
  {0x2555, 0xb8},
  {0x2556, 0xb7},
  {0x2557, 0xbb},
  {0x2558, 0xd4},
  {0x2559, 0xd3},
  {0x255a, 0xc8},
  {0x255b, 0xbe},
  {0x255c, 0xbd},
  {0x255d, 0xbc},
  {0x255e, 0xc6},
  {0x255f, 0xc7},
  {0x2560, 0xcc},
  {0x2561, 0xb5},
  {0x2562, 0xb6},
  {0x2563, 0xb9},
  {0x2564, 0xd1},
  {0x2565, 0xd2},
  {0x2566, 0xcb},
  {0x2567, 0xcf},
  {0x2568, 0xd0},
  {0x2569, 0xca},
  {0x256a, 0xd8},
  {0x256b, 0xd7},
  {0x256c, 0xce},
  {0x2580, 0xdf},
  {0x2584, 0xdc},
  {0x2588, 0xdb},
  {0x258c, 0xdd},
  {0x2590, 0xde},
  {0x2591, 0xb0},
  {0x2592, 0xb1},
  {0x2593, 0xb2},
  {0x25a0, 0xfe},
  {0x25ac, 0x16},
  {0x25b2, 0x1e},
  {0x25b6, 0x10},
  {0x25ba, 0x10},
  {0x25bc, 0x1f},
  {0x25c0, 0x11},
  {0x25c4, 0x11},
  {0x25cb, 0x09},
  {0x25d8, 0x08},
  {0x25d9, 0x0a},
  {0x263a, 0x01},
  {0x263b, 0x02},
  {0x263c, 0x0f},
  {0x2640, 0x0c},
  {0x2642, 0x0b},
  {0x2660, 0x06},
  {0x2663, 0x05},
  {0x2665, 0x03},
  {0x2666, 0x04},
  {0x266a, 0x0d},
  {0x266b, 0x0e},
  {0x266c, 0x0e}
};

struct unipair FSystemTest::unicode_lat15_pairs[] =
{
  //  .----------- unicode
  //  |      .---- fontpos
  //  |      |
  {0x0020, 0x20},
  {0x0021, 0x21},
  {0x0022, 0x22},
  {0x0023, 0x23},
  {0x0024, 0x24},
  {0x0025, 0x25},
  {0x0026, 0x26},
  {0x0027, 0x27},
  {0x0028, 0x28},
  {0x0029, 0x29},
  {0x002a, 0x2a},
  {0x002b, 0x2b},
  {0x002c, 0x2c},
  {0x002d, 0x2d},
  {0x002e, 0x2e},
  {0x002f, 0x2f},
  {0x0030, 0x30},
  {0x0031, 0x31},
  {0x0032, 0x32},
  {0x0033, 0x33},
  {0x0034, 0x34},
  {0x0035, 0x35},
  {0x0036, 0x36},
  {0x0037, 0x37},
  {0x0038, 0x38},
  {0x0039, 0x39},
  {0x003a, 0x3a},
  {0x003b, 0x3b},
  {0x003c, 0x3c},
  {0x003d, 0x3d},
  {0x003e, 0x3e},
  {0x003f, 0x3f},
  {0x0040, 0x40},
  {0x0041, 0x41},
  {0x0042, 0x42},
  {0x0043, 0x43},
  {0x0044, 0x44},
  {0x0045, 0x45},
  {0x0046, 0x46},
  {0x0047, 0x47},
  {0x0048, 0x48},
  {0x0049, 0x49},
  {0x004a, 0x4a},
  {0x004b, 0x4b},
  {0x004c, 0x4c},
  {0x004d, 0x4d},
  {0x004e, 0x4e},
  {0x004f, 0x4f},
  {0x0050, 0x50},
  {0x0051, 0x51},
  {0x0052, 0x52},
  {0x0053, 0x53},
  {0x0054, 0x54},
  {0x0055, 0x55},
  {0x0056, 0x56},
  {0x0057, 0x57},
  {0x0058, 0x58},
  {0x0059, 0x59},
  {0x005a, 0x5a},
  {0x005b, 0x5b},
  {0x005c, 0x5c},
  {0x005d, 0x5d},
  {0x005e, 0x5e},
  {0x005f, 0x5f},
  {0x0060, 0x60},
  {0x0061, 0x61},
  {0x0062, 0x62},
  {0x0063, 0x63},
  {0x0064, 0x64},
  {0x0065, 0x65},
  {0x0066, 0x66},
  {0x0067, 0x67},
  {0x0068, 0x68},
  {0x0069, 0x69},
  {0x006a, 0x6a},
  {0x006b, 0x6b},
  {0x006c, 0x6c},
  {0x006d, 0x6d},
  {0x006e, 0x6e},
  {0x006f, 0x6f},
  {0x0070, 0x70},
  {0x0071, 0x71},
  {0x0072, 0x72},
  {0x0073, 0x73},
  {0x0074, 0x74},
  {0x0075, 0x75},
  {0x0076, 0x76},
  {0x0077, 0x77},
  {0x0078, 0x78},
  {0x0079, 0x79},
  {0x007a, 0x7a},
  {0x007b, 0x7b},
  {0x007c, 0x7c},
  {0x007d, 0x7d},
  {0x007e, 0x7e},
  {0x00a0, 0x20},
  {0x00a1, 0xad},
  {0x00a2, 0x9b},
  {0x00a3, 0x9c},
  {0x00a4, 0x11},
  {0x00a5, 0x9d},
  {0x00a6, 0x12},
  {0x00a7, 0x15},
  {0x00a8, 0x16},
  {0x00a9, 0x00},
  {0x00aa, 0xa6},
  {0x00ab, 0xae},
  {0x00ac, 0xaa},
  {0x00ad, 0x2d},
  {0x00ae, 0x01},
  {0x00af, 0x17},
  {0x00b0, 0xf8},
  {0x00b1, 0xf1},
  {0x00b2, 0xfd},
  {0x00b3, 0x1c},
  {0x00b4, 0x1d},
  {0x00b5, 0xe6},
  {0x00b6, 0x14},
  {0x00b7, 0xfa},
  {0x00b8, 0x1e},
  {0x00b9, 0x1f},
  {0x00ba, 0xa7},
  {0x00bb, 0xaf},
  {0x00bc, 0xac},
  {0x00bd, 0xab},
  {0x00be, 0x7f},
  {0x00bf, 0xa8},
  {0x00c0, 0xa9},
  {0x00c1, 0xb2},
  {0x00c2, 0xb5},
  {0x00c3, 0xb6},
  {0x00c4, 0x8e},
  {0x00c5, 0x8f},
  {0x00c6, 0x92},
  {0x00c7, 0x80},
  {0x00c8, 0xb7},
  {0x00c9, 0x90},
  {0x00ca, 0xb8},
  {0x00cb, 0xb9},
  {0x00cc, 0xba},
  {0x00cd, 0xbb},
  {0x00ce, 0xbc},
  {0x00cf, 0xbd},
  {0x00d0, 0xbe},
  {0x00d1, 0xa5},
  {0x00d2, 0xc6},
  {0x00d3, 0xc7},
  {0x00d4, 0xc8},
  {0x00d5, 0xc9},
  {0x00d6, 0x99},
  {0x00d7, 0xca},
  {0x00d8, 0xcb},
  {0x00d9, 0xcc},
  {0x00da, 0xcd},
  {0x00db, 0xcf},
  {0x00dc, 0x9a},
  {0x00dd, 0x02},
  {0x00de, 0xd0},
  {0x00df, 0xe1},
  {0x00e0, 0x85},
  {0x00e1, 0xa0},
  {0x00e2, 0x83},
  {0x00e3, 0xd1},
  {0x00e4, 0x84},
  {0x00e5, 0x86},
  {0x00e6, 0x91},
  {0x00e7, 0x87},
  {0x00e8, 0x8a},
  {0x00e9, 0x82},
  {0x00ea, 0x88},
  {0x00eb, 0x89},
  {0x00ec, 0x8d},
  {0x00ed, 0xa1},
  {0x00ee, 0x8c},
  {0x00ef, 0x8b},
  {0x00f0, 0xd2},
  {0x00f1, 0xa4},
  {0x00f2, 0x95},
  {0x00f3, 0xa2},
  {0x00f4, 0x93},
  {0x00f5, 0xd3},
  {0x00f6, 0x94},
  {0x00f7, 0xf6},
  {0x00f8, 0xd4},
  {0x00f9, 0x97},
  {0x00fa, 0xa3},
  {0x00fb, 0x96},
  {0x00fc, 0x81},
  {0x00fd, 0xd5},
  {0x00fe, 0xd6},
  {0x00ff, 0x98},
  {0x0110, 0xbe},
  {0x011e, 0xd7},
  {0x011f, 0xdc},
  {0x0130, 0xdd},
  {0x0131, 0xde},
  {0x0152, 0x03},
  {0x0153, 0x05},
  {0x015e, 0xdf},
  {0x015f, 0xe0},
  {0x0160, 0xe2},
  {0x0161, 0xe4},
  {0x0178, 0x06},
  {0x017d, 0xe5},
  {0x017e, 0xe7},
  {0x0192, 0x9f},
  {0x02c6, 0xe8},
  {0x02c9, 0x17},
  {0x02dc, 0xe9},
  {0x0391, 0x41},
  {0x0392, 0x42},
  {0x0395, 0x45},
  {0x0396, 0x5a},
  {0x0397, 0x48},
  {0x0399, 0x49},
  {0x039a, 0x4b},
  {0x039c, 0x4d},
  {0x039d, 0x4e},
  {0x039f, 0x4f},
  {0x03a1, 0x50},
  {0x03a4, 0x54},
  {0x03a7, 0x58},
  {0x03bc, 0xe6},
  {0x03c0, 0xe3},
  {0x0401, 0xb9},
  {0x0405, 0x53},
  {0x0406, 0x49},
  {0x0407, 0xbd},
  {0x0408, 0x4a},
  {0x0410, 0x41},
  {0x0412, 0x42},
  {0x0415, 0x45},
  {0x041a, 0x4b},
  {0x041c, 0x4d},
  {0x041d, 0x48},
  {0x041e, 0x4f},
  {0x0420, 0x50},
  {0x0421, 0x43},
  {0x0422, 0x54},
  {0x0425, 0x58},
  {0x0430, 0x61},
  {0x0435, 0x65},
  {0x043e, 0x6f},
  {0x0440, 0x70},
  {0x0441, 0x63},
  {0x0443, 0x79},
  {0x0445, 0x78},
  {0x0451, 0x89},
  {0x0455, 0x73},
  {0x0456, 0x69},
  {0x0457, 0x8b},
  {0x0458, 0x6a},
  {0x04ae, 0x59},
  {0x2000, 0x20},
  {0x2001, 0x20},
  {0x2002, 0x20},
  {0x2003, 0x20},
  {0x2004, 0x20},
  {0x2005, 0x20},
  {0x2006, 0x20},
  {0x2007, 0x20},
  {0x2008, 0x20},
  {0x2009, 0x20},
  {0x200a, 0x20},
  {0x2010, 0x2d},
  {0x2011, 0x2d},
  {0x2012, 0x2d},
  {0x2013, 0x2d},
  {0x2014, 0x07},
  {0x2015, 0x07},
  {0x2018, 0xea},
  {0x2019, 0xeb},
  {0x201a, 0xec},
  {0x201c, 0xed},
  {0x201d, 0xee},
  {0x201e, 0xef},
  {0x2020, 0x08},
  {0x2021, 0x09},
  {0x2022, 0x0a},
  {0x2026, 0x0b},
  {0x202f, 0x20},
  {0x2030, 0x0c},
  {0x2039, 0xf4},
  {0x203a, 0xf5},
  {0x203c, 0x13},
  {0x207f, 0xfc},
  {0x20a7, 0x9e},
  {0x20ac, 0xf9},
  {0x20ae, 0x0f},
  {0x2116, 0x0e},
  {0x2122, 0x0d},
  {0x212a, 0x4b},
  {0x212b, 0x8f},
  {0x2190, 0x1b},
  {0x2191, 0x18},
  {0x2192, 0x1a},
  {0x2193, 0x19},
  {0x2205, 0x10},
  {0x2208, 0xff},
  {0x2212, 0x2d},
  {0x2248, 0xf7},
  {0x2260, 0xfb},
  {0x2261, 0xf0},
  {0x2264, 0xf3},
  {0x2265, 0xf2},
  {0x226a, 0xae},
  {0x226b, 0xaf},
  {0x2295, 0x25},
  {0x2296, 0x2d},
  {0x2298, 0x2f},
  {0x2299, 0x2e},
  {0x229b, 0x2a},
  {0x229c, 0x3d},
  {0x2460, 0x31},
  {0x2461, 0x32},
  {0x2462, 0x33},
  {0x2463, 0x34},
  {0x2464, 0x35},
  {0x2465, 0x36},
  {0x2466, 0x37},
  {0x2467, 0x38},
  {0x2468, 0x39},
  {0x24b6, 0x41},
  {0x24b7, 0x42},
  {0x24b8, 0x43},
  {0x24b9, 0x44},
  {0x24ba, 0x45},
  {0x24bb, 0x46},
  {0x24bc, 0x47},
  {0x24bd, 0x48},
  {0x24be, 0x49},
  {0x24bf, 0x4a},
  {0x24c0, 0x4b},
  {0x24c1, 0x4c},
  {0x24c2, 0x4d},
  {0x24c3, 0x4e},
  {0x24c4, 0x4f},
  {0x24c5, 0x50},
  {0x24c6, 0x51},
  {0x24c7, 0x52},
  {0x24c8, 0x53},
  {0x24c9, 0x54},
  {0x24ca, 0x55},
  {0x24cb, 0x56},
  {0x24cc, 0x57},
  {0x24cd, 0x58},
  {0x24ce, 0x59},
  {0x24cf, 0x5a},
  {0x24d0, 0x61},
  {0x24d1, 0x62},
  {0x24d2, 0x63},
  {0x24d3, 0x64},
  {0x24d4, 0x65},
  {0x24d5, 0x66},
  {0x24d6, 0x67},
  {0x24d7, 0x68},
  {0x24d8, 0x69},
  {0x24d9, 0x6a},
  {0x24da, 0x6b},
  {0x24db, 0x6c},
  {0x24dc, 0x6d},
  {0x24dd, 0x6e},
  {0x24de, 0x6f},
  {0x24df, 0x70},
  {0x24e0, 0x71},
  {0x24e1, 0x72},
  {0x24e2, 0x73},
  {0x24e3, 0x74},
  {0x24e4, 0x75},
  {0x24e5, 0x76},
  {0x24e6, 0x77},
  {0x24e7, 0x78},
  {0x24e8, 0x79},
  {0x24e9, 0x7a},
  {0x24ea, 0x30},
  {0x2500, 0xc4},
  {0x2501, 0xc4},
  {0x2502, 0xb3},
  {0x2503, 0xb3},
  {0x250c, 0xda},
  {0x250d, 0xda},
  {0x250e, 0xda},
  {0x250f, 0xda},
  {0x2510, 0xbf},
  {0x2511, 0xbf},
  {0x2512, 0xbf},
  {0x2513, 0xbf},
  {0x2514, 0xc0},
  {0x2515, 0xc0},
  {0x2516, 0xc0},
  {0x2517, 0xc0},
  {0x2518, 0xd9},
  {0x2519, 0xd9},
  {0x251a, 0xd9},
  {0x251b, 0xd9},
  {0x251c, 0xc3},
  {0x251d, 0xc3},
  {0x251e, 0xc3},
  {0x251f, 0xc3},
  {0x2520, 0xc3},
  {0x2521, 0xc3},
  {0x2522, 0xc3},
  {0x2523, 0xc3},
  {0x2524, 0xb4},
  {0x2525, 0xb4},
  {0x2526, 0xb4},
  {0x2527, 0xb4},
  {0x2528, 0xb4},
  {0x2529, 0xb4},
  {0x252a, 0xb4},
  {0x252b, 0xb4},
  {0x252c, 0xc2},
  {0x252d, 0xc2},
  {0x252e, 0xc2},
  {0x252f, 0xc2},
  {0x2530, 0xc2},
  {0x2531, 0xc2},
  {0x2532, 0xc2},
  {0x2533, 0xc2},
  {0x2534, 0xc1},
  {0x2535, 0xc1},
  {0x2536, 0xc1},
  {0x2537, 0xc1},
  {0x2538, 0xc1},
  {0x2539, 0xc1},
  {0x253a, 0xc1},
  {0x253b, 0xc1},
  {0x253c, 0xc5},
  {0x253d, 0xc5},
  {0x253e, 0xc5},
  {0x253f, 0xc5},
  {0x2540, 0xc5},
  {0x2541, 0xc5},
  {0x2542, 0xc5},
  {0x2543, 0xc5},
  {0x2544, 0xc5},
  {0x2545, 0xc5},
  {0x2546, 0xc5},
  {0x2547, 0xc5},
  {0x2548, 0xc5},
  {0x2549, 0xc5},
  {0x254a, 0xc5},
  {0x254b, 0xc5},
  {0x256a, 0xd8},
  {0x256c, 0xce},
  {0x2574, 0xc4},
  {0x2575, 0xb3},
  {0x2576, 0xc4},
  {0x2577, 0xb3},
  {0x2578, 0xc4},
  {0x2579, 0xb3},
  {0x257a, 0xc4},
  {0x257b, 0xb3},
  {0x257c, 0xc4},
  {0x257d, 0xb3},
  {0x257e, 0xc4},
  {0x257f, 0xb3},
  {0x2588, 0xdb},
  {0x2591, 0xb0},
  {0x2592, 0xb1},
  {0x25a0, 0xfe},
  {0x25ae, 0xfe},
  {0x25b2, 0x18},
  {0x25b4, 0x18},
  {0x25b6, 0x1a},
  {0x25b8, 0x1a},
  {0x25bc, 0x19},
  {0x25be, 0x19},
  {0x25c0, 0x1b},
  {0x25c2, 0x1b},
  {0x25c8, 0x04},
  {0x25cf, 0x0a},
  {0x2666, 0x04},
  {0xfffd, 0x04}
};

FSystemTest::RGB FSystemTest::terminal_color[16] { };

FSystemTest::RGB FSystemTest::defaultColor[16]
{
  {0x00, 0x00, 0x00}, {0xaa, 0x00, 0x00},
  {0x00, 0xaa, 0x00}, {0xaa, 0x55, 0x00},
  {0x00, 0x00, 0xaa}, {0xaa, 0x00, 0xaa},
  {0x00, 0xaa, 0xaa}, {0xaa, 0xaa, 0xaa},
  {0x55, 0x55, 0x55}, {0xff, 0x55, 0x55},
  {0x55, 0xff, 0x55}, {0xff, 0xff, 0x55},
  {0x55, 0x55, 0xff}, {0xff, 0x55, 0xff},
  {0x55, 0xff, 0xff}, {0xff, 0xff, 0xff}
};


// static class attributes
//----------------------------------------------------------------------
FSystemTest::ShiftState   FSystemTest::shift_state{};
struct console_font_op    FSystemTest::terminal_font{};
unimapdesc                FSystemTest::terminal_unicode_map{};
struct fb_var_screeninfo  FSystemTest::fb_terminal_info{};
struct fb_fix_screeninfo  FSystemTest::fb_terminal_fix_info{};
bool                      FSystemTest::vga_port_access{false};


// constructors and destructor
//----------------------------------------------------------------------
FSystemTest::FSystemTest()  // constructor
{
  // fill bit field with 0
  memset (&shift_state, 0x00, sizeof(shift_state));
  memset (&terminal_font, 0x00, sizeof(terminal_font));
  memset (&terminal_unicode_map, 0x00, sizeof(terminal_unicode_map));

  constexpr std::size_t font_data_size = 4 * 32 * 512;
  terminal_font.data = new uChar[font_data_size]{ };

  // init framebuffer
  initVScreenInfo();
  initFScreenInfo();
}

//----------------------------------------------------------------------
FSystemTest::~FSystemTest()  // destructor
{
  if ( terminal_font.data )
  {
    delete[] terminal_font.data;
    terminal_font.data = nullptr;
  }

  if ( terminal_unicode_map.entries )
  {
    delete[] terminal_unicode_map.entries;
    terminal_unicode_map.entries = nullptr;
  }
}


// public methods of FSystemTest
//----------------------------------------------------------------------
auto FSystemTest::inPortByte (uShort port) -> uChar
{
  std::cerr << "Call: im (port=" << port << ")\n";

  switch ( port )
  {
    case 0x3c1:
      if ( attribute_controller_mode == AccessMode::data )
        return ac_address_register[ac_index];
      break;

    case 0x3cc:
      return port_3cc;

    case 0x3da:
      attribute_controller_mode = AccessMode::index;
      return port_3da;
  }

  return 0;
}

//----------------------------------------------------------------------
void FSystemTest::outPortByte (uChar value, uShort port)
{
  std::cerr << "Call: outb (value=" << value
            << ", port=" << port << ")\n";

  switch ( port )
  {
    case 0x3c0:
      if ( attribute_controller_mode == AccessMode::index )
      {
        ac_index = value & 0x1f;
        palette_addr_source_field = bool(value & 0x20);
        attribute_controller_mode = AccessMode::data;
      }
      else
      {
        ac_address_register[ac_index] = value;
        attribute_controller_mode = AccessMode::index;
      }
      break;

    default:
      return;
  }
}

//----------------------------------------------------------------------
auto FSystemTest::isTTY (int fd) const -> int
{
  std::cerr << "Call: isatty (fd=" << fd << ")\n";
  return 1;
}

//----------------------------------------------------------------------
auto FSystemTest::ioctl (int fd, uLong request, ...) -> int
{
  va_list args{};
  void* argp{};
  std::string req_string{};
  int ret_val{-1};

  va_start (args, request);
  argp = va_arg (args, void*);

  switch ( request )
  {
    case TIOCLINUX:
    {
      req_string = "TIOCLINUX";
      auto subcode = static_cast<char*>(argp);
      auto state = reinterpret_cast<uChar*>(&shift_state);

      if ( *subcode == 6 )
        *subcode = static_cast<char>(*state);

      ret_val = 0;
      break;
    }

    case KDFONTOP:
    {
      req_string = "KDFONTOP";
      constexpr std::size_t font_data_size = 4 * 32 * 512;
      auto fn = static_cast<console_font_op*>(argp);

      if ( fn->op == KD_FONT_OP_GET )
      {
        // If data is empty
        if ( ! terminal_font.data )
        {
          terminal_font.data = new uChar[font_data_size]{ };
        }

        // Set Default
        if ( terminal_font.data[219 * 32] == 0 )
        {
          terminal_font.width = 8;
          terminal_font.height = 16;
          terminal_font.charcount = 256;

          if ( fn->data )
            std::memcpy (terminal_font.data, &vga8x16, sizeof(vga8x16));
        }

        fn->flags     = terminal_font.flags;
        fn->width     = terminal_font.width;
        fn->height    = terminal_font.height;
        fn->charcount = terminal_font.charcount;

        if ( fn->data && terminal_font.data )
          std::memcpy (fn->data, terminal_font.data, font_data_size);

        terminal_font.op = KD_FONT_OP_GET;
      }

      if ( fn->op == KD_FONT_OP_SET )
      {
        terminal_font.flags      = fn->flags;
        terminal_font.width      = fn->width;
        terminal_font.height     = fn->height;
        terminal_font.charcount  = fn->charcount;
        constexpr int fix_height = 32;  // This value is identical for all fonts
        auto size = fn->width / 8 * fix_height * fn->charcount;

        if ( ! terminal_font.data )  // If data is empty on a second run
        {
          try
          {
            terminal_font.data = new uChar[font_data_size]{ };
          }
          catch (const std::bad_alloc&)
          {
            return -1;
          }
        }

        if ( fn->data )
          std::memcpy (terminal_font.data, fn->data, size);

        terminal_font.op = KD_FONT_OP_SET;
      }

      ret_val = 0;
      break;
    }

    case KDGKBTYPE:
    {
      req_string = "KDGKBTYPE";
      char* keyboard_type = static_cast<char*>(argp);
      *keyboard_type = KB_101;
      ret_val = 0;
      break;
    }

    case GIO_CMAP:
    {
      req_string = "GIO_CMAP";
      auto cmap = static_cast<ColorMap*>(argp);
      // Set Default
      if ( terminal_color[15].red   == 0
        && terminal_color[15].green == 0
        && terminal_color[15].blue  == 0 )
      {
        for (std::size_t index = 0; index < 16; index++)
        {
          terminal_color[index].red   = defaultColor[index].red;
          terminal_color[index].green = defaultColor[index].green;
          terminal_color[index].blue  = defaultColor[index].blue;
        }
      }

      for (std::size_t index = 0; index < 16; index++)
      {
        cmap->color[index].red   = terminal_color[index].red;
        cmap->color[index].green = terminal_color[index].green;
        cmap->color[index].blue  = terminal_color[index].blue;
      }

      ret_val = 0;
      break;
    }

    case PIO_CMAP:
    {
      req_string = "PIO_CMAP";
      auto cmap = static_cast<ColorMap*>(argp);

      for (std::size_t index = 0; index < 16; index++)
      {
        terminal_color[index].red   = cmap->color[index].red;
        terminal_color[index].green = cmap->color[index].green;
        terminal_color[index].blue  = cmap->color[index].blue;
      }

      ret_val = 0;
      break;
    }

    case GIO_UNIMAP:
    {
      req_string = "GIO_UNIMAP";
      auto umap = static_cast<unimapdesc*>(argp);
      std::size_t unipair_size = sizeof(unipair);
      std::size_t pairs = ( codeset == Codeset::cp437 )
                        ? sizeof(unicode_cp437_pairs) / unipair_size
                        : sizeof(unicode_lat15_pairs) / unipair_size;
      std::size_t pairs_size = pairs * unipair_size;

      // Sets the default unicode map of the terminal on the first call
      if ( ! terminal_unicode_map.entries )
      {
        terminal_unicode_map.entry_ct = pairs;
        terminal_unicode_map.entries = new unipair[pairs]();

        if ( codeset == Codeset::cp437 )
          std::memcpy (terminal_unicode_map.entries, &unicode_cp437_pairs, pairs_size);
        else
          std::memcpy (terminal_unicode_map.entries, &unicode_lat15_pairs, pairs_size);
      }

      umap->entry_ct = terminal_unicode_map.entry_ct;

      if ( umap->entries && terminal_unicode_map.entries )
      {
        std::memcpy (umap->entries, terminal_unicode_map.entries, pairs_size);
        errno = 0;
        ret_val = 0;
      }
      else
      {
        errno = ENOMEM;
        ret_val = -1;
      }

      break;
    }

    case PIO_UNIMAP:
    {
      req_string = "PIO_UNIMAP";
      auto umap = static_cast<unimapdesc*>(argp);
      std::size_t pairs = umap->entry_ct;
      std::size_t pairs_size = pairs * sizeof(unipair);
      terminal_unicode_map.entry_ct = umap->entry_ct;

      if ( terminal_unicode_map.entries )
      {
        delete[] terminal_unicode_map.entries;
        terminal_unicode_map.entries = nullptr;
      }

      try
      {
        terminal_unicode_map.entries = new unipair[pairs]();
      }
      catch (const std::bad_alloc&)
      {
        return -1;
      }

      if ( umap->entries )
      {
        std::memcpy (terminal_unicode_map.entries, umap->entries, pairs_size);
        errno = 0;
        ret_val = 0;
      }
      else
      {
        errno = ENOMEM;
        ret_val = 0;
      }

      break;
    }

    case PIO_UNIMAPCLR:
      req_string = "PIO_UNIMAPCLR";
      ret_val = 0;
      break;

    case FBIOGET_VSCREENINFO:
    {
      req_string = "FBIOGET_VSCREENINFO";
      auto fb_var = static_cast<fb_var_screeninfo*>(argp);
      std::memcpy (fb_var, &fb_terminal_info, sizeof(fb_terminal_info));
      ret_val = 0;
      break;
    }

    case FBIOGET_FSCREENINFO:
    {
      req_string = "FBIOGET_FSCREENINFO";
      auto fb_fix = static_cast<fb_fix_screeninfo*>(argp);
      std::memcpy (fb_fix, &fb_terminal_fix_info, sizeof(fb_terminal_fix_info));
      ret_val = 0;
      break;
    }

    case KDENABIO:
    {
      req_string = "KDENABIO";
      vga_port_access = true;
      ret_val = 0;
      break;
    }

    case KDDISABIO:
    {
      req_string = "KDDISABIO";
      vga_port_access = false;
      ret_val = 0;
      break;
    }

    case TIOCGWINSZ:
    {
      req_string = "TIOCGWINSZ";
      auto win_size = static_cast<winsize*>(argp);
      win_size->ws_col = 96;
      win_size->ws_row = 36;
      ret_val = 0;
      break;
    }
  }

  va_end (args);

  std::cerr << "Call: ioctl (fd=" << fd
            << ", request=" << req_string
            << "(0x" << std::hex << request << ")"
            << ", argp=" << argp << std::dec << ")\n";
  return ret_val;
}

//----------------------------------------------------------------------
auto FSystemTest::open (const char* pathname, int flags, ...) -> int
{
  va_list args{};
  va_start (args, flags);
  auto mode = static_cast<mode_t>(va_arg (args, int));
  va_end (args);

  std::cerr << "Call: open (pathname=\"" << pathname
            << "\", flags=" << flags
            << ", mode=" << mode << ")\n";

  if ( std::memcmp(pathname, "/dev/fb0", 8) == 0
    || std::memcmp(pathname, "/dev/fb/0", 9) == 0 )
    return 99;  // File descriptor

  return 0;
}

//----------------------------------------------------------------------
auto FSystemTest::close (int fildes) -> int
{
  std::cerr << "Call: close (fildes=" << fildes << ")\n";
  return 0;
}

//----------------------------------------------------------------------
auto FSystemTest::fopen (const char* path, const char* mode) -> FILE*
{
  std::cerr << "Call: fopen (path=" << path
            << ", mode=" << mode << ")\n";
  return nullptr;
}

//----------------------------------------------------------------------
auto FSystemTest::fclose (FILE* fp) -> int
{
  std::cerr << "Call: fclose (fp=" << fp << ")\n";
  return 0;
}

//----------------------------------------------------------------------
auto FSystemTest::fputs (const char* str, FILE* stream) -> int
{
  std::cerr << "Call: fputs (" << str << ", " << stream << ")\n";
  std::string string = str;
  int count = 0;

  for (auto&& ch : string)
  {
    characters.push_back(ch);
    count++;
  }

  return count;
}

//----------------------------------------------------------------------
auto FSystemTest::putchar (int c) -> int
{
  std::cerr << "Call: putchar (" << c << ")\n";
  characters.push_back(c);
  return 1;
}

//----------------------------------------------------------------------
auto FSystemTest::getuid() -> uid_t
{
  return 0;
}

//----------------------------------------------------------------------
auto FSystemTest::geteuid() -> uid_t
{
  return 0;
}

//----------------------------------------------------------------------
auto FSystemTest::getpwuid_r ( uid_t, struct passwd*, char*
                            , size_t, struct passwd** ) -> int
{
  return 0;
}

//----------------------------------------------------------------------
auto FSystemTest::realpath (const char*, char*) -> char*
{
  return const_cast<char*>("");
}

//----------------------------------------------------------------------
auto FSystemTest::getRGB (finalcut::FColor c) -> RGB&
{
  auto i = std::size_t(c);

  if ( i < 16 )
    return terminal_color[i];

  return terminal_color[0];
}

//----------------------------------------------------------------------
auto FSystemTest::getConsoleFont() -> console_font_op&
{
  return terminal_font;
}

//----------------------------------------------------------------------
auto FSystemTest::getShiftState() -> ShiftState&
{
  return shift_state;
}

//----------------------------------------------------------------------
auto FSystemTest::getCharacters() -> std::string&
{
  return characters;
}

//----------------------------------------------------------------------
void FSystemTest::setCodeset (Codeset cs)
{
  codeset = cs;
}


// private methods of FSystemTest
//----------------------------------------------------------------------
void FSystemTest::initVScreenInfo()
{
  fb_terminal_info.xres = 800;
  fb_terminal_info.yres = 600;
  fb_terminal_info.xres_virtual = 1176;
  fb_terminal_info.yres_virtual = 885;
  fb_terminal_info.xoffset = 0;
  fb_terminal_info.yoffset = 0;
  fb_terminal_info.bits_per_pixel = 32;
  fb_terminal_info.grayscale = 0;
  fb_terminal_info.red.offset = 16;
  fb_terminal_info.red.length = 8;
  fb_terminal_info.red.msb_right = 0;
  fb_terminal_info.green.offset = 8;
  fb_terminal_info.green.length = 8,
  fb_terminal_info.green.msb_right = 0;
  fb_terminal_info.blue.offset = 0;
  fb_terminal_info.blue.length = 8;
  fb_terminal_info.blue.msb_right = 0;
  fb_terminal_info.transp.offset = 0;
  fb_terminal_info.transp.length = 0;
  fb_terminal_info.transp.msb_right = 0;
  fb_terminal_info.nonstd = 0;
  fb_terminal_info.activate = 0;
  fb_terminal_info.height = 0xffffffff;
  fb_terminal_info.width = 0xffffffff;
  fb_terminal_info.accel_flags = 0;
  fb_terminal_info.pixclock = 0;
  fb_terminal_info.left_margin = 0;
  fb_terminal_info.right_margin = 0;
  fb_terminal_info.upper_margin = 0;
  fb_terminal_info.lower_margin = 0;
  fb_terminal_info.hsync_len = 0;
  fb_terminal_info.vsync_len = 0;
  fb_terminal_info.sync = 0;
  fb_terminal_info.vmode = 0;
  fb_terminal_info.rotate = 0;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,2,0)
  fb_terminal_info.colorspace = 0;
#endif
  fb_terminal_info.reserved[0] = 0;
  fb_terminal_info.reserved[1] = 0;
  fb_terminal_info.reserved[2] = 0;
  fb_terminal_info.reserved[3] = 0;
}

//----------------------------------------------------------------------
void FSystemTest::initFScreenInfo()
{
  char id[16] { "VESA VGA" };
  std::strncpy (fb_terminal_fix_info.id, id, sizeof(fb_terminal_fix_info.id));
  fb_terminal_fix_info.smem_start = 0xf9000000;
  fb_terminal_fix_info.smem_len = 0x00500000;
  fb_terminal_fix_info.type = 0;
  fb_terminal_fix_info.type_aux = 0;
  fb_terminal_fix_info.visual = 2;
  fb_terminal_fix_info.xpanstep = 0;
  fb_terminal_fix_info.ypanstep = 0;
  fb_terminal_fix_info.ywrapstep = 0;
  fb_terminal_fix_info.line_length = 5120;
  fb_terminal_fix_info.mmio_start = 0;
  fb_terminal_fix_info.mmio_len = 0;
  fb_terminal_fix_info.accel = 0;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,2,0)
  fb_terminal_fix_info.capabilities = 0;
#endif
  fb_terminal_fix_info.reserved[0] = 0;
  fb_terminal_fix_info.reserved[1] = 0;
}

}  // namespace test


//----------------------------------------------------------------------
// class FTermLinuxTest
//----------------------------------------------------------------------

class FTermLinuxTest : public CPPUNIT_NS::TestFixture
                     , test::ConEmu
{
  public:
    FTermLinuxTest() = default;

  protected:
    void classNameTest();
    void linuxConsoleTest();
    void linuxConsoleLat15Test();
    void linuxCursorStyleTest();
    void linuxColorPaletteTest();
    void linuxFontTest();
    void modifierKeyTest();

  private:
    // Adds code needed to register the test suite
    CPPUNIT_TEST_SUITE (FTermLinuxTest);

    // Add a methods to the test suite
    CPPUNIT_TEST (classNameTest);
    CPPUNIT_TEST (linuxConsoleTest);
    CPPUNIT_TEST (linuxConsoleLat15Test);
    CPPUNIT_TEST (linuxCursorStyleTest);
    CPPUNIT_TEST (linuxColorPaletteTest);
    CPPUNIT_TEST (linuxFontTest);
    CPPUNIT_TEST (modifierKeyTest);

    // End of test suite definition
    CPPUNIT_TEST_SUITE_END();
};

//----------------------------------------------------------------------
void FTermLinuxTest::classNameTest()
{
  const finalcut::FTermLinux p{};
  const finalcut::FString& classname = p.getClassName();
  CPPUNIT_ASSERT ( classname == "FTermLinux" );
}

//----------------------------------------------------------------------
void FTermLinuxTest::linuxConsoleTest()
{
  std::unique_ptr<finalcut::FSystem> fsys = std::make_unique<test::FSystemTest>();
  finalcut::FTerm::setFSystem(fsys);
  std::cout << "\n";
  auto& data = finalcut::FTermData::getInstance();

  auto& encoding_list = data.getEncodingList();
  encoding_list["UTF-8"] = finalcut::Encoding::UTF8;
  encoding_list["UTF8"]  = finalcut::Encoding::UTF8;
  encoding_list["VT100"] = finalcut::Encoding::VT100;
  encoding_list["PC"]    = finalcut::Encoding::PC;
  encoding_list["ASCII"] = finalcut::Encoding::ASCII;

  data.setTermEncoding(finalcut::Encoding::PC);
  data.setBaudrate(38400);
  data.setTermType("linux");
  data.setTermFileName("/dev/tty2");

#if DEBUG
  data.setFramebufferBpp(32);
#endif

  data.supportShadowCharacter (false);
  data.supportHalfBlockCharacter (false);
  data.supportCursorOptimisation (true);
  data.setCursorHidden (true);
  data.useAlternateScreen (false);
  data.setASCIIConsole (true);
  data.setVT100Console (false);
  data.setUTF8Console (false);
  data.setUTF8 (false);
  data.setNewFont (false);
  data.setVGAFont (false);
  data.setMonochron (false);
  data.setTermResized (false);
  setenv ("TERM", "linux", 1);

  auto& term_detection = finalcut::FTermDetection::getInstance();
  finalcut::FTermcap::init();
  finalcut::FTermLinux linux;

  // setupterm is needed for tputs in ncurses >= 6.1
  setupterm (static_cast<char*>(nullptr), 1, static_cast<int*>(nullptr));
  data.setTermType (finalcut::FTermType::linux_con);

  pid_t pid = forkConEmu();

  if ( isConEmuChildProcess(pid) )
  {
    // (gdb) set follow-fork-mode child
    setenv ("TERM", "linux", 1);
    setenv ("COLUMNS", "90", 1);
    setenv ("LINES", "30", 1);
    unsetenv("TERMCAP");
    unsetenv("COLORTERM");
    unsetenv("COLORFGBG");
    unsetenv("VTE_VERSION");
    unsetenv("XTERM_VERSION");
    unsetenv("ROXTERM_ID");
    unsetenv("KONSOLE_DBUS_SESSION");
    unsetenv("KONSOLE_DCOP");
    unsetenv("TMUX");

    term_detection.detect();
    linux.init();

    CPPUNIT_ASSERT ( isatty(3) == 0 );
    CPPUNIT_ASSERT ( data.isTermType(finalcut::FTermType::linux_con) );
    CPPUNIT_ASSERT ( data.getTerminalGeometry().getWidth() == 96 );
    CPPUNIT_ASSERT ( data.getTerminalGeometry().getHeight() == 36 );
    CPPUNIT_ASSERT ( data.hasShadowCharacter() );
    CPPUNIT_ASSERT ( data.hasHalfBlockCharacter() );
    CPPUNIT_ASSERT ( linux.getFramebufferBpp() == 32 );

    const auto& fsystem = finalcut::FSystem::getInstance();
    auto fsystest = static_cast<test::FSystemTest*>(fsystem.get());
    std::string& characters = fsystest->getCharacters();
    linux.setUTF8 (false);

    CPPUNIT_ASSERT ( characters == ESC "%@" );
    characters.clear();

    linux.setUTF8 (true);
    CPPUNIT_ASSERT ( characters == ESC "%G" );
    characters.clear();

    linux.setBeep (20, 100);     // Hz < 21
    CPPUNIT_ASSERT ( characters.empty() );
    linux.setBeep (32767, 100);  // Hz > 32766
    CPPUNIT_ASSERT ( characters.empty() );
    linux.setBeep (200, -1);     // ms < 0
    CPPUNIT_ASSERT ( characters.empty() );
    linux.setBeep (200, 2000);   // ms > 1999
    CPPUNIT_ASSERT ( characters.empty() );
    linux.setBeep (200, 100);    // 200 Hz - 100 ms
    CPPUNIT_ASSERT ( characters == CSI "10;200]" CSI "11;100]" );
    characters.clear();

    linux.resetBeep();
    CPPUNIT_ASSERT ( characters == CSI "10;750]" CSI "11;125]" );
    characters.clear();

    linux.initCharMap();
    auto& character_map = data.getCharSubstitutionMap();
    CPPUNIT_ASSERT ( ! character_map.isEmpty() );
    CPPUNIT_ASSERT ( character_map.getMappedChar(wchar_t(finalcut::UniChar::BlackCircle)) == L'*' );
    CPPUNIT_ASSERT ( character_map.getMappedChar(wchar_t(finalcut::UniChar::Times)) == L'x' );
    CPPUNIT_ASSERT ( character_map.getMappedChar(L'ˣ') == L'ⁿ' );
    linux.finish();

    closeConEmuStdStreams();
    exit(EXIT_SUCCESS);
  }
  else  // Parent
  {
    // Start the terminal emulation
    startConEmuTerminal (ConEmu::console::linux_con);
    int wstatus;

    if ( waitpid(pid, &wstatus, WUNTRACED) != pid )
      std::cerr << "waitpid error" << std::endl;

    if ( WIFEXITED(wstatus) )
      CPPUNIT_ASSERT ( WEXITSTATUS(wstatus) == 0 );
  }
}

//----------------------------------------------------------------------
void FTermLinuxTest::linuxConsoleLat15Test()
{
  std::unique_ptr<finalcut::FSystem> fsys = std::make_unique<test::FSystemTest>();
  auto fsystest = static_cast<test::FSystemTest*>(fsys.get());
  fsystest->setCodeset(test::FSystemTest::Codeset::lat15);
  finalcut::FTerm::setFSystem(fsys);
  std::cout << "\n";
  auto& data = finalcut::FTermData::getInstance();

  auto& encoding_list = data.getEncodingList();
  encoding_list["UTF-8"] = finalcut::Encoding::UTF8;
  encoding_list["UTF8"]  = finalcut::Encoding::UTF8;
  encoding_list["VT100"] = finalcut::Encoding::VT100;
  encoding_list["PC"]    = finalcut::Encoding::PC;
  encoding_list["ASCII"] = finalcut::Encoding::ASCII;

  data.setTermEncoding(finalcut::Encoding::PC);
  data.setBaudrate(38400);
  data.setTermType("linux");
  data.setTermFileName("/dev/tty3");
  data.setTTYFileDescriptor(0);

#if DEBUG
  data.setFramebufferBpp(32);
#endif

  data.supportShadowCharacter (true);
  data.supportHalfBlockCharacter (true);
  data.supportCursorOptimisation (true);
  data.setCursorHidden (true);
  data.useAlternateScreen (false);
  data.setASCIIConsole (true);
  data.setVT100Console (false);
  data.setUTF8Console (false);
  data.setUTF8 (false);
  data.setNewFont (false);
  data.setVGAFont (false);
  data.setMonochron (false);
  data.setTermResized (false);

  auto& term_detection = finalcut::FTermDetection::getInstance();
  finalcut::FTermLinux linux;

  // setupterm is needed for tputs in ncurses >= 6.1
  setupterm (static_cast<char*>(nullptr), 1, static_cast<int*>(nullptr));
  data.setTermType (finalcut::FTermType::linux_con);

  pid_t pid = forkConEmu();

  if ( isConEmuChildProcess(pid) )
  {
    // (gdb) set follow-fork-mode child
    setenv ("TERM", "linux", 1);
    setenv ("COLUMNS", "90", 1);
    setenv ("LINES", "30", 1);
    unsetenv("TERMCAP");
    unsetenv("COLORTERM");
    unsetenv("COLORFGBG");
    unsetenv("VTE_VERSION");
    unsetenv("XTERM_VERSION");
    unsetenv("ROXTERM_ID");
    unsetenv("KONSOLE_DBUS_SESSION");
    unsetenv("KONSOLE_DCOP");
    unsetenv("TMUX");

    term_detection.detect();

    linux.init();
    linux.initCharMap();
    CPPUNIT_ASSERT ( data.isTermType(finalcut::FTermType::linux_con) );
    CPPUNIT_ASSERT ( ! data.hasShadowCharacter() );
    CPPUNIT_ASSERT ( ! data.hasHalfBlockCharacter() );
    auto& character_map = data.getCharSubstitutionMap();
    CPPUNIT_ASSERT ( ! character_map.isEmpty() );
    CPPUNIT_ASSERT ( character_map.getMappedChar(wchar_t(finalcut::UniChar::SquareRoot)) == L'x' );
    CPPUNIT_ASSERT ( character_map.getMappedChar(wchar_t(finalcut::UniChar::BlackLeftPointingPointer)) == L'◀' );
    CPPUNIT_ASSERT ( character_map.getMappedChar(wchar_t(finalcut::UniChar::BlackRightPointingPointer)) == L'▶' );
    CPPUNIT_ASSERT ( character_map.getMappedChar(L'ˣ') == L'ⁿ' );
    linux.finish();

    closeConEmuStdStreams();
    exit(EXIT_SUCCESS);
  }
  else  // Parent
  {
    // Start the terminal emulation
    startConEmuTerminal (ConEmu::console::linux_con);
    int wstatus;

    if ( waitpid(pid, &wstatus, WUNTRACED) != pid )
      std::cerr << "waitpid error" << std::endl;

    if ( WIFEXITED(wstatus) )
      CPPUNIT_ASSERT ( WEXITSTATUS(wstatus) == 0 );
  }
}

//----------------------------------------------------------------------
void FTermLinuxTest::linuxCursorStyleTest()
{
  std::unique_ptr<finalcut::FSystem> fsys = std::make_unique<test::FSystemTest>();
  finalcut::FTerm::setFSystem(fsys);
  std::cout << "\n";
  auto& data = finalcut::FTermData::getInstance();

  auto& encoding_list = data.getEncodingList();
  encoding_list["UTF-8"] = finalcut::Encoding::UTF8;
  encoding_list["UTF8"]  = finalcut::Encoding::UTF8;
  encoding_list["VT100"] = finalcut::Encoding::VT100;
  encoding_list["PC"]    = finalcut::Encoding::PC;
  encoding_list["ASCII"] = finalcut::Encoding::ASCII;

  data.setTermEncoding(finalcut::Encoding::PC);
  data.setBaudrate(38400);
  data.setTermType("linux");
  data.setTermFileName("/dev/tty2");

#if DEBUG
  data.setFramebufferBpp(32);
#endif

  data.supportShadowCharacter (false);
  data.supportHalfBlockCharacter (false);
  data.supportCursorOptimisation (true);
  data.setCursorHidden (false);
  data.useAlternateScreen (false);
  data.setASCIIConsole (true);
  data.setVT100Console (false);
  data.setUTF8Console (false);
  data.setUTF8 (false);
  data.setNewFont (false);
  data.setVGAFont (false);
  data.setMonochron (false);
  data.setTermResized (false);

  // setupterm is needed for tputs in ncurses >= 6.1
  setupterm (static_cast<char*>(nullptr), 1, static_cast<int*>(nullptr));
  auto& term_detection = finalcut::FTermDetection::getInstance();
  finalcut::FTermcap::init();
  finalcut::FTermLinux linux;

  pid_t pid = forkConEmu();

  if ( isConEmuChildProcess(pid) )
  {
    // (gdb) set follow-fork-mode child
    setenv ("TERM", "linux", 1);
    setenv ("COLUMNS", "90", 1);
    setenv ("LINES", "30", 1);
    unsetenv("TERMCAP");
    unsetenv("COLORTERM");
    unsetenv("COLORFGBG");
    unsetenv("VTE_VERSION");
    unsetenv("XTERM_VERSION");
    unsetenv("ROXTERM_ID");
    unsetenv("KONSOLE_DBUS_SESSION");
    unsetenv("KONSOLE_DCOP");
    unsetenv("TMUX");

    term_detection.detect();
    linux.init();

    const auto& fsystem = finalcut::FSystem::getInstance();
    auto fsystest = static_cast<test::FSystemTest*>(fsystem.get());
    std::string& characters = fsystest->getCharacters();
    characters.clear();
    linux.setCursorStyle (finalcut::LinuxConsoleCursorStyle::Default);
    CPPUNIT_ASSERT ( characters == CSI "?0c" );
    CPPUNIT_ASSERT ( linux.getCursorStyle() == finalcut::LinuxConsoleCursorStyle::Default );
    CPPUNIT_ASSERT_CSTRING ( linux.getCursorStyleString(), CSI "?0c" );
    characters.clear();
    linux.setCursorStyle (finalcut::LinuxConsoleCursorStyle::Invisible);
    CPPUNIT_ASSERT ( characters == CSI "?1c" );
    CPPUNIT_ASSERT ( linux.getCursorStyle() == finalcut::LinuxConsoleCursorStyle::Invisible );
    CPPUNIT_ASSERT_CSTRING ( linux.getCursorStyleString(), CSI "?1c" );
    characters.clear();
    linux.setCursorStyle (finalcut::LinuxConsoleCursorStyle::Underscore);
    CPPUNIT_ASSERT ( characters == CSI "?2c" );
    CPPUNIT_ASSERT_CSTRING ( linux.getCursorStyleString(), CSI "?2c" );
    CPPUNIT_ASSERT ( linux.getCursorStyle() == finalcut::LinuxConsoleCursorStyle::Underscore );
    characters.clear();
    linux.setCursorStyle (finalcut::LinuxConsoleCursorStyle::LowerThird);
    CPPUNIT_ASSERT ( characters == CSI "?3c" );
    CPPUNIT_ASSERT_CSTRING ( linux.getCursorStyleString(), CSI "?3c" );
    CPPUNIT_ASSERT ( linux.getCursorStyle() == finalcut::LinuxConsoleCursorStyle::LowerThird );
    characters.clear();
    linux.setCursorStyle (finalcut::LinuxConsoleCursorStyle::LowerHalf);
    CPPUNIT_ASSERT ( characters == CSI "?4c" );
    CPPUNIT_ASSERT_CSTRING ( linux.getCursorStyleString(), CSI "?4c" );
    CPPUNIT_ASSERT ( linux.getCursorStyle() == finalcut::LinuxConsoleCursorStyle::LowerHalf );
    characters.clear();
    linux.setCursorStyle (finalcut::LinuxConsoleCursorStyle::TwoThirds);
    CPPUNIT_ASSERT ( characters == CSI "?5c" );
    CPPUNIT_ASSERT_CSTRING ( linux.getCursorStyleString(), CSI "?5c" );
    CPPUNIT_ASSERT ( linux.getCursorStyle() == finalcut::LinuxConsoleCursorStyle::TwoThirds );
    characters.clear();
    linux.setCursorStyle (finalcut::LinuxConsoleCursorStyle::FullBlock);
    CPPUNIT_ASSERT ( characters == CSI "?6c" );
    CPPUNIT_ASSERT_CSTRING ( linux.getCursorStyleString(), CSI "?6c" );
    CPPUNIT_ASSERT ( linux.getCursorStyle() == finalcut::LinuxConsoleCursorStyle::FullBlock );
    characters.clear();

    linux.setCursorStyle (finalcut::LinuxConsoleCursorStyle::Default);
    CPPUNIT_ASSERT ( characters == CSI "?0c" );
    CPPUNIT_ASSERT_CSTRING ( linux.getCursorStyleString(), CSI "?0c" );
    CPPUNIT_ASSERT ( linux.getCursorStyle() == finalcut::LinuxConsoleCursorStyle::Default );
    characters.clear();
    linux.setCursorStyle (linux.getCursorStyle());
    CPPUNIT_ASSERT ( characters == CSI "?0c" );
    CPPUNIT_ASSERT_CSTRING ( linux.getCursorStyleString(), CSI "?0c" );
    CPPUNIT_ASSERT ( linux.getCursorStyle() == finalcut::LinuxConsoleCursorStyle::Default );
    characters.clear();
    data.setCursorHidden (true);
    linux.setCursorStyle (finalcut::LinuxConsoleCursorStyle::Invisible);
    data.setCursorHidden (false);
    CPPUNIT_ASSERT ( characters == "" );
    CPPUNIT_ASSERT ( linux.getCursorStyle() == finalcut::LinuxConsoleCursorStyle::Invisible );
    characters.clear();
    linux.setCursorStyle (linux.getCursorStyle());
    CPPUNIT_ASSERT ( characters == CSI "?1c" );
    CPPUNIT_ASSERT_CSTRING ( linux.getCursorStyleString(), CSI "?1c" );
    characters.clear();
    data.setCursorHidden (true);
    linux.setCursorStyle (finalcut::LinuxConsoleCursorStyle::Underscore);
    data.setCursorHidden (false);
    CPPUNIT_ASSERT ( characters == "" );
    CPPUNIT_ASSERT ( linux.getCursorStyle() == finalcut::LinuxConsoleCursorStyle::Underscore );
    characters.clear();
    linux.setCursorStyle (linux.getCursorStyle());
    CPPUNIT_ASSERT ( characters == CSI "?2c" );
    CPPUNIT_ASSERT_CSTRING ( linux.getCursorStyleString(), CSI "?2c" );
    characters.clear();
    data.setCursorHidden (true);
    linux.setCursorStyle (finalcut::LinuxConsoleCursorStyle::LowerThird);
    data.setCursorHidden (false);
    CPPUNIT_ASSERT ( characters == "" );
    CPPUNIT_ASSERT ( linux.getCursorStyle() == finalcut::LinuxConsoleCursorStyle::LowerThird );
    characters.clear();
    linux.setCursorStyle (linux.getCursorStyle());
    CPPUNIT_ASSERT ( characters == CSI "?3c" );
    CPPUNIT_ASSERT_CSTRING ( linux.getCursorStyleString(), CSI "?3c" );
    characters.clear();
    data.setCursorHidden (true);
    linux.setCursorStyle (finalcut::LinuxConsoleCursorStyle::LowerHalf);
    data.setCursorHidden (false);
    CPPUNIT_ASSERT ( characters == "" );
    CPPUNIT_ASSERT ( linux.getCursorStyle() == finalcut::LinuxConsoleCursorStyle::LowerHalf );
    characters.clear();
    linux.setCursorStyle (linux.getCursorStyle());
    CPPUNIT_ASSERT ( characters == CSI "?4c" );
    CPPUNIT_ASSERT_CSTRING ( linux.getCursorStyleString(), CSI "?4c" );
    characters.clear();
    data.setCursorHidden (true);
    linux.setCursorStyle (finalcut::LinuxConsoleCursorStyle::TwoThirds);
    data.setCursorHidden (false);
    CPPUNIT_ASSERT ( characters == "" );
    CPPUNIT_ASSERT ( linux.getCursorStyle() == finalcut::LinuxConsoleCursorStyle::TwoThirds );
    characters.clear();
    linux.setCursorStyle (linux.getCursorStyle());
    CPPUNIT_ASSERT ( characters == CSI "?5c" );
    CPPUNIT_ASSERT_CSTRING ( linux.getCursorStyleString(), CSI "?5c" );
    characters.clear();
    data.setCursorHidden (true);
    linux.setCursorStyle (finalcut::LinuxConsoleCursorStyle::FullBlock);
    data.setCursorHidden (false);
    CPPUNIT_ASSERT ( characters == "" );
    CPPUNIT_ASSERT ( linux.getCursorStyle() == finalcut::LinuxConsoleCursorStyle::FullBlock );
    characters.clear();
    linux.setCursorStyle (linux.getCursorStyle());
    CPPUNIT_ASSERT ( characters == CSI "?6c" );
    CPPUNIT_ASSERT_CSTRING ( linux.getCursorStyleString(), CSI "?6c" );
    characters.clear();

    linux.finish();

    closeConEmuStdStreams();
    exit(EXIT_SUCCESS);
  }
  else  // Parent
  {
    // Start the terminal emulation
    startConEmuTerminal (ConEmu::console::linux_con);
    int wstatus;

    if ( waitpid(pid, &wstatus, WUNTRACED) != pid )
      std::cerr << "waitpid error" << std::endl;

    if ( WIFEXITED(wstatus) )
      CPPUNIT_ASSERT ( WEXITSTATUS(wstatus) == 0 );
  }
}

//----------------------------------------------------------------------
void FTermLinuxTest::linuxColorPaletteTest()
{
  std::unique_ptr<finalcut::FSystem> fsys = std::make_unique<test::FSystemTest>();
  finalcut::FTerm::setFSystem(fsys);
  std::cout << "\n";
  auto& data = finalcut::FTermData::getInstance();

  auto& encoding_list = data.getEncodingList();
  encoding_list["UTF-8"] = finalcut::Encoding::UTF8;
  encoding_list["UTF8"]  = finalcut::Encoding::UTF8;
  encoding_list["VT100"] = finalcut::Encoding::VT100;
  encoding_list["PC"]    = finalcut::Encoding::PC;
  encoding_list["ASCII"] = finalcut::Encoding::ASCII;

  data.setTermEncoding(finalcut::Encoding::PC);
  data.setBaudrate(38400);
  data.setTermType("linux");
  data.setTermFileName("/dev/tty2");

#if DEBUG
  data.setFramebufferBpp(32);
#endif

  data.supportShadowCharacter (false);
  data.supportHalfBlockCharacter (false);
  data.supportCursorOptimisation (true);
  data.setCursorHidden (true);
  data.useAlternateScreen (false);
  data.setASCIIConsole (true);
  data.setVT100Console (false);
  data.setUTF8Console (false);
  data.setUTF8 (false);
  data.setNewFont (false);
  data.setVGAFont (false);
  data.setMonochron (false);
  data.setTermResized (false);

  // setupterm is needed for tputs in ncurses >= 6.1
  setupterm (static_cast<char*>(nullptr), 1, static_cast<int*>(nullptr));
  auto& term_detection = finalcut::FTermDetection::getInstance();
  finalcut::FTermLinux linux;
  data.setTermType (finalcut::FTermType::linux_con);

  pid_t pid = forkConEmu();

  if ( isConEmuChildProcess(pid) )
  {
    // (gdb) set follow-fork-mode child
    setenv ("TERM", "linux", 1);
    setenv ("COLUMNS", "90", 1);
    setenv ("LINES", "30", 1);
    unsetenv("TERMCAP");
    unsetenv("COLORTERM");
    unsetenv("COLORFGBG");
    unsetenv("VTE_VERSION");
    unsetenv("XTERM_VERSION");
    unsetenv("ROXTERM_ID");
    unsetenv("KONSOLE_DBUS_SESSION");
    unsetenv("KONSOLE_DCOP");
    unsetenv("TMUX");

    term_detection.detect();
    linux.init();
    const auto& fsystem = finalcut::FSystem::getInstance();
    auto fsystest = static_cast<test::FSystemTest*>(fsystem.get());

    CPPUNIT_ASSERT ( linux.resetColorMap() == true );
    CPPUNIT_ASSERT ( linux.saveColorMap() == true );
    finalcut::FColor index = finalcut::FOptiAttr::vga2ansi(finalcut::FColor::Black);
    test::FSystemTest::RGB& RGB0 = fsystest->getRGB(index);
    CPPUNIT_ASSERT ( RGB0.red   == 0x00 );
    CPPUNIT_ASSERT ( RGB0.green == 0x00 );
    CPPUNIT_ASSERT ( RGB0.blue  == 0x00 );
    CPPUNIT_ASSERT ( linux.setPalette (index, 0x01, 0x02, 0x03) );
    CPPUNIT_ASSERT ( RGB0.red   == 0x01 );
    CPPUNIT_ASSERT ( RGB0.green == 0x02 );
    CPPUNIT_ASSERT ( RGB0.blue  == 0x03 );

    index = finalcut::FOptiAttr::vga2ansi(finalcut::FColor::Blue);
    test::FSystemTest::RGB& RGB1 = fsystest->getRGB(index);
    CPPUNIT_ASSERT ( RGB1.red   == 0x00 );
    CPPUNIT_ASSERT ( RGB1.green == 0x00 );
    CPPUNIT_ASSERT ( RGB1.blue  == 0xaa );
    CPPUNIT_ASSERT ( linux.setPalette (index, 0x04, 0x05, 0x06) );
    CPPUNIT_ASSERT ( RGB1.red   == 0x04 );
    CPPUNIT_ASSERT ( RGB1.green == 0x05 );
    CPPUNIT_ASSERT ( RGB1.blue  == 0x06 );

    index = finalcut::FOptiAttr::vga2ansi(finalcut::FColor::Green);
    test::FSystemTest::RGB& RGB2 = fsystest->getRGB(index);
    CPPUNIT_ASSERT ( RGB2.red   == 0x00 );
    CPPUNIT_ASSERT ( RGB2.green == 0xaa );
    CPPUNIT_ASSERT ( RGB2.blue  == 0x00 );
    CPPUNIT_ASSERT ( linux.setPalette (index, 0x07, 0x08, 0x09) );
    CPPUNIT_ASSERT ( RGB2.red   == 0x07 );
    CPPUNIT_ASSERT ( RGB2.green == 0x08 );
    CPPUNIT_ASSERT ( RGB2.blue  == 0x09 );

    index = finalcut::FOptiAttr::vga2ansi(finalcut::FColor::Cyan);
    test::FSystemTest::RGB& RGB3 = fsystest->getRGB(index);
    CPPUNIT_ASSERT ( RGB3.red   == 0x00 );
    CPPUNIT_ASSERT ( RGB3.green == 0xaa );
    CPPUNIT_ASSERT ( RGB3.blue  == 0xaa );
    CPPUNIT_ASSERT ( linux.setPalette (index, 0x0a, 0x0b, 0x0c) );
    CPPUNIT_ASSERT ( RGB3.red   == 0x0a );
    CPPUNIT_ASSERT ( RGB3.green == 0x0b );
    CPPUNIT_ASSERT ( RGB3.blue  == 0x0c );

    index = finalcut::FOptiAttr::vga2ansi(finalcut::FColor::Red);
    test::FSystemTest::RGB& RGB4 = fsystest->getRGB(index);
    CPPUNIT_ASSERT ( RGB4.red   == 0xaa );
    CPPUNIT_ASSERT ( RGB4.green == 0x00 );
    CPPUNIT_ASSERT ( RGB4.blue  == 0x00 );
    CPPUNIT_ASSERT ( linux.setPalette (index, 0x0d, 0x0e, 0x0f) );
    CPPUNIT_ASSERT ( RGB4.red   == 0x0d );
    CPPUNIT_ASSERT ( RGB4.green == 0x0e );
    CPPUNIT_ASSERT ( RGB4.blue  == 0x0f );

    index = finalcut::FOptiAttr::vga2ansi(finalcut::FColor::Magenta);
    test::FSystemTest::RGB& RGB5 = fsystest->getRGB(index);
    CPPUNIT_ASSERT ( RGB5.red   == 0xaa );
    CPPUNIT_ASSERT ( RGB5.green == 0x00 );
    CPPUNIT_ASSERT ( RGB5.blue  == 0xaa );
    CPPUNIT_ASSERT ( linux.setPalette (index, 0x10, 0x11, 0x12) );
    CPPUNIT_ASSERT ( RGB5.red   == 0x10 );
    CPPUNIT_ASSERT ( RGB5.green == 0x11 );
    CPPUNIT_ASSERT ( RGB5.blue  == 0x12 );

    index = finalcut::FOptiAttr::vga2ansi(finalcut::FColor::Brown);
    test::FSystemTest::RGB& RGB6 = fsystest->getRGB(index);
    CPPUNIT_ASSERT ( RGB6.red   == 0xaa );
    CPPUNIT_ASSERT ( RGB6.green == 0x55 );
    CPPUNIT_ASSERT ( RGB6.blue  == 0x00 );
    CPPUNIT_ASSERT ( linux.setPalette (index, 0x13, 0x14, 0x15) );
    CPPUNIT_ASSERT ( RGB6.red   == 0x13 );
    CPPUNIT_ASSERT ( RGB6.green == 0x14 );
    CPPUNIT_ASSERT ( RGB6.blue  == 0x15 );

    index = finalcut::FOptiAttr::vga2ansi(finalcut::FColor::LightGray);
    test::FSystemTest::RGB& RGB7 = fsystest->getRGB(index);
    CPPUNIT_ASSERT ( RGB7.red   == 0xaa );
    CPPUNIT_ASSERT ( RGB7.green == 0xaa );
    CPPUNIT_ASSERT ( RGB7.blue  == 0xaa );
    CPPUNIT_ASSERT ( linux.setPalette (index, 0x16, 0x17, 0x18) );
    CPPUNIT_ASSERT ( RGB7.red   == 0x16 );
    CPPUNIT_ASSERT ( RGB7.green == 0x17 );
    CPPUNIT_ASSERT ( RGB7.blue  == 0x18 );

    index = finalcut::FOptiAttr::vga2ansi(finalcut::FColor::DarkGray);
    test::FSystemTest::RGB& RGB8 = fsystest->getRGB(index);
    CPPUNIT_ASSERT ( RGB8.red   == 0x55 );
    CPPUNIT_ASSERT ( RGB8.green == 0x55 );
    CPPUNIT_ASSERT ( RGB8.blue  == 0x55 );
    CPPUNIT_ASSERT ( linux.setPalette (index, 0x19, 0x20, 0x21) );
    CPPUNIT_ASSERT ( RGB8.red   == 0x19 );
    CPPUNIT_ASSERT ( RGB8.green == 0x20 );
    CPPUNIT_ASSERT ( RGB8.blue  == 0x21 );

    index = finalcut::FOptiAttr::vga2ansi(finalcut::FColor::LightBlue);
    test::FSystemTest::RGB& RGB9 = fsystest->getRGB(index);
    CPPUNIT_ASSERT ( RGB9.red   == 0x55 );
    CPPUNIT_ASSERT ( RGB9.green == 0x55 );
    CPPUNIT_ASSERT ( RGB9.blue  == 0xff );
    CPPUNIT_ASSERT ( linux.setPalette (index, 0x22, 0x23, 0x24) );
    CPPUNIT_ASSERT ( RGB9.red   == 0x22 );
    CPPUNIT_ASSERT ( RGB9.green == 0x23 );
    CPPUNIT_ASSERT ( RGB9.blue  == 0x24 );

    index = finalcut::FOptiAttr::vga2ansi(finalcut::FColor::LightGreen);
    test::FSystemTest::RGB& RGB10 = fsystest->getRGB(index);
    CPPUNIT_ASSERT ( RGB10.red   == 0x55 );
    CPPUNIT_ASSERT ( RGB10.green == 0xff );
    CPPUNIT_ASSERT ( RGB10.blue  == 0x55 );
    CPPUNIT_ASSERT ( linux.setPalette (index, 0x25, 0x26, 0x27) );
    CPPUNIT_ASSERT ( RGB10.red   == 0x25 );
    CPPUNIT_ASSERT ( RGB10.green == 0x26 );
    CPPUNIT_ASSERT ( RGB10.blue  == 0x27 );

    index = finalcut::FOptiAttr::vga2ansi(finalcut::FColor::LightCyan);
    test::FSystemTest::RGB& RGB11 = fsystest->getRGB(index);
    CPPUNIT_ASSERT ( RGB11.red   == 0x55 );
    CPPUNIT_ASSERT ( RGB11.green == 0xff );
    CPPUNIT_ASSERT ( RGB11.blue  == 0xff );
    CPPUNIT_ASSERT ( linux.setPalette (index, 0x28, 0x29, 0x30) );
    CPPUNIT_ASSERT ( RGB11.red   == 0x28 );
    CPPUNIT_ASSERT ( RGB11.green == 0x29 );
    CPPUNIT_ASSERT ( RGB11.blue  == 0x30 );

    index = finalcut::FOptiAttr::vga2ansi(finalcut::FColor::LightRed);
    test::FSystemTest::RGB& RGB12 = fsystest->getRGB(index);
    CPPUNIT_ASSERT ( RGB12.red   == 0xff );
    CPPUNIT_ASSERT ( RGB12.green == 0x55 );
    CPPUNIT_ASSERT ( RGB12.blue  == 0x55 );
    CPPUNIT_ASSERT ( linux.setPalette (index, 0x31, 0x32, 0x33) );
    CPPUNIT_ASSERT ( RGB12.red   == 0x31 );
    CPPUNIT_ASSERT ( RGB12.green == 0x32 );
    CPPUNIT_ASSERT ( RGB12.blue  == 0x33 );

    index = finalcut::FOptiAttr::vga2ansi(finalcut::FColor::LightMagenta);
    test::FSystemTest::RGB& RGB13 = fsystest->getRGB(index);
    CPPUNIT_ASSERT ( RGB13.red   == 0xff );
    CPPUNIT_ASSERT ( RGB13.green == 0x55 );
    CPPUNIT_ASSERT ( RGB13.blue  == 0xff );
    CPPUNIT_ASSERT ( linux.setPalette (index, 0x34, 0x35, 0x36) );
    CPPUNIT_ASSERT ( RGB13.red   == 0x34 );
    CPPUNIT_ASSERT ( RGB13.green == 0x35 );
    CPPUNIT_ASSERT ( RGB13.blue  == 0x36 );

    index = finalcut::FOptiAttr::vga2ansi(finalcut::FColor::Yellow);
    test::FSystemTest::RGB& RGB14 = fsystest->getRGB(index);
    CPPUNIT_ASSERT ( RGB14.red   == 0xff );
    CPPUNIT_ASSERT ( RGB14.green == 0xff );
    CPPUNIT_ASSERT ( RGB14.blue  == 0x55 );
    CPPUNIT_ASSERT ( linux.setPalette (index, 0x37, 0x38, 0x39) );
    CPPUNIT_ASSERT ( RGB14.red   == 0x37 );
    CPPUNIT_ASSERT ( RGB14.green == 0x38 );
    CPPUNIT_ASSERT ( RGB14.blue  == 0x39 );

    index = finalcut::FOptiAttr::vga2ansi(finalcut::FColor::White);
    test::FSystemTest::RGB& RGB15 = fsystest->getRGB(index);
    CPPUNIT_ASSERT ( RGB15.red   == 0xff );
    CPPUNIT_ASSERT ( RGB15.green == 0xff );
    CPPUNIT_ASSERT ( RGB15.blue  == 0xff );
    CPPUNIT_ASSERT ( linux.setPalette (index, 0x40, 0x41, 0x42) );
    CPPUNIT_ASSERT ( RGB15.red   == 0x40 );
    CPPUNIT_ASSERT ( RGB15.green == 0x41 );
    CPPUNIT_ASSERT ( RGB15.blue  == 0x42 );
    // Out of range -> no change
    CPPUNIT_ASSERT ( linux.setPalette (index, -1, 0, 0) );
    CPPUNIT_ASSERT ( RGB15.red   == 0x40 );
    CPPUNIT_ASSERT ( RGB15.green == 0x41 );
    CPPUNIT_ASSERT ( RGB15.blue  == 0x42 );
    // Out of range -> no change
    CPPUNIT_ASSERT ( linux.setPalette (index, 0, -1, 0) );
    CPPUNIT_ASSERT ( RGB15.red   == 0x40 );
    CPPUNIT_ASSERT ( RGB15.green == 0x41 );
    CPPUNIT_ASSERT ( RGB15.blue  == 0x42 );
    // Out of range -> no change
    CPPUNIT_ASSERT ( linux.setPalette (index, 0, 0, -1) );
    CPPUNIT_ASSERT ( RGB15.red   == 0x40 );
    CPPUNIT_ASSERT ( RGB15.green == 0x41 );
    CPPUNIT_ASSERT ( RGB15.blue  == 0x42 );
    // Out of range -> no change
    CPPUNIT_ASSERT ( linux.setPalette (index, 256, 0, 0) );
    CPPUNIT_ASSERT ( RGB15.red   == 0x40 );
    CPPUNIT_ASSERT ( RGB15.green == 0x41 );
    CPPUNIT_ASSERT ( RGB15.blue  == 0x42 );
    // Out of range -> no change
    CPPUNIT_ASSERT ( linux.setPalette (index, 0, 256, 0) );
    CPPUNIT_ASSERT ( RGB15.red   == 0x40 );
    CPPUNIT_ASSERT ( RGB15.green == 0x41 );
    CPPUNIT_ASSERT ( RGB15.blue  == 0x42 );
    // Out of range -> no change
    CPPUNIT_ASSERT ( linux.setPalette (index, 0, 0, 256) );
    CPPUNIT_ASSERT ( RGB15.red   == 0x40 );
    CPPUNIT_ASSERT ( RGB15.green == 0x41 );
    CPPUNIT_ASSERT ( RGB15.blue  == 0x42 );

    CPPUNIT_ASSERT ( linux.resetColorMap() == true );

    linux.finish();

    closeConEmuStdStreams();
    exit(EXIT_SUCCESS);
  }
  else  // Parent
  {
    // Start the terminal emulation
    startConEmuTerminal (ConEmu::console::linux_con);
    int wstatus;

    if ( waitpid(pid, &wstatus, WUNTRACED) != pid )
      std::cerr << "waitpid error" << std::endl;

    if ( WIFEXITED(wstatus) )
      CPPUNIT_ASSERT ( WEXITSTATUS(wstatus) == 0 );
  }
}

//----------------------------------------------------------------------
void FTermLinuxTest::linuxFontTest()
{
  std::unique_ptr<finalcut::FSystem> fsys = std::make_unique<test::FSystemTest>();
  finalcut::FTerm::setFSystem(fsys);
  std::cout << "\n";
  auto& data = finalcut::FTermData::getInstance();

  auto& encoding_list = data.getEncodingList();
  encoding_list["UTF-8"] = finalcut::Encoding::UTF8;
  encoding_list["UTF8"]  = finalcut::Encoding::UTF8;
  encoding_list["VT100"] = finalcut::Encoding::VT100;
  encoding_list["PC"]    = finalcut::Encoding::PC;
  encoding_list["ASCII"] = finalcut::Encoding::ASCII;

  data.setTermEncoding(finalcut::Encoding::PC);
  data.setBaudrate(38400);
  data.setTermType("linux");
  data.setTermFileName("/dev/tty2");

#if DEBUG
  data.setFramebufferBpp(32);
#endif

  data.supportShadowCharacter (false);
  data.supportHalfBlockCharacter (false);
  data.supportCursorOptimisation (true);
  data.setCursorHidden (true);
  data.useAlternateScreen (false);
  data.setASCIIConsole (true);
  data.setVT100Console (false);
  data.setUTF8Console (false);
  data.setUTF8 (false);
  data.setNewFont (false);
  data.setVGAFont (false);
  data.setMonochron (false);
  data.setTermResized (false);

  // setupterm is needed for tputs in ncurses >= 6.1
  setupterm (static_cast<char*>(nullptr), 1, static_cast<int*>(nullptr));
  auto& term_detection = finalcut::FTermDetection::getInstance();
  finalcut::FTermLinux linux;

  pid_t pid = forkConEmu();

  if ( isConEmuChildProcess(pid) )
  {
    // (gdb) set follow-fork-mode child
    setenv ("TERM", "linux", 1);
    setenv ("COLUMNS", "90", 1);
    setenv ("LINES", "30", 1);
    unsetenv("TERMCAP");
    unsetenv("COLORTERM");
    unsetenv("COLORFGBG");
    unsetenv("VTE_VERSION");
    unsetenv("XTERM_VERSION");
    unsetenv("ROXTERM_ID");
    unsetenv("KONSOLE_DBUS_SESSION");
    unsetenv("KONSOLE_DCOP");
    unsetenv("TMUX");

    term_detection.detect();
    linux.init();
    const auto& fsystem = finalcut::FSystem::getInstance();
    auto fsystest = static_cast<test::FSystemTest*>(fsystem.get());
    console_font_op& font = fsystest->getConsoleFont();
    CPPUNIT_ASSERT ( font.op == KD_FONT_OP_GET );
    CPPUNIT_ASSERT ( ! linux.isVGAFontUsed() );
    CPPUNIT_ASSERT ( ! linux.isNewFontUsed() );
    linux.loadVGAFont();
    CPPUNIT_ASSERT ( data.hasShadowCharacter() );
    CPPUNIT_ASSERT ( data.hasHalfBlockCharacter() );
    CPPUNIT_ASSERT ( font.op == KD_FONT_OP_SET );
    CPPUNIT_ASSERT ( linux.isVGAFontUsed() );
    CPPUNIT_ASSERT ( ! linux.isNewFontUsed() );
    CPPUNIT_ASSERT ( font.data );

    // Full block character test
    for (std::size_t i = 0; i < 16 ; i++)
      CPPUNIT_ASSERT ( font.data && font.data[219 * 32 + i] == 0xff );

    linux.loadNewFont();
    CPPUNIT_ASSERT ( ! linux.isVGAFontUsed() );
    CPPUNIT_ASSERT ( linux.isNewFontUsed() );

    // Full block character test
    for (std::size_t i = 0; i < 16 ; i++)
      CPPUNIT_ASSERT ( font.data[219 * 32 + i] == 0xff );

    // New font bullet
    CPPUNIT_ASSERT ( font.data[249 * 32 + 0] == 0x00 );
    CPPUNIT_ASSERT ( font.data[249 * 32 + 1] == 0x00 );
    CPPUNIT_ASSERT ( font.data[249 * 32 + 2] == 0x00 );
    CPPUNIT_ASSERT ( font.data[249 * 32 + 3] == 0x00 );
    CPPUNIT_ASSERT ( font.data[249 * 32 + 4] == 0x38 );
    CPPUNIT_ASSERT ( font.data[249 * 32 + 5] == 0x7c );
    CPPUNIT_ASSERT ( font.data[249 * 32 + 6] == 0xfe );
    CPPUNIT_ASSERT ( font.data[249 * 32 + 7] == 0xfe );
    CPPUNIT_ASSERT ( font.data[249 * 32 + 8] == 0xfe );
    CPPUNIT_ASSERT ( font.data[249 * 32 + 9] == 0x7c );
    CPPUNIT_ASSERT ( font.data[249 * 32 + 10] == 0x38 );
    CPPUNIT_ASSERT ( font.data[249 * 32 + 11] == 0x00 );
    CPPUNIT_ASSERT ( font.data[249 * 32 + 12] == 0x00 );
    CPPUNIT_ASSERT ( font.data[249 * 32 + 13] == 0x00 );
    CPPUNIT_ASSERT ( font.data[249 * 32 + 14] == 0x00 );
    CPPUNIT_ASSERT ( font.data[249 * 32 + 15] == 0x00 );

    linux.loadOldFont();
    CPPUNIT_ASSERT ( ! linux.isVGAFontUsed() );
    CPPUNIT_ASSERT ( ! linux.isNewFontUsed() );

    // cp437 bullet operator
    CPPUNIT_ASSERT ( font.data[249 * 32 + 0] == 0x00 );
    CPPUNIT_ASSERT ( font.data[249 * 32 + 1] == 0x00 );
    CPPUNIT_ASSERT ( font.data[249 * 32 + 2] == 0x00 );
    CPPUNIT_ASSERT ( font.data[249 * 32 + 3] == 0x00 );
    CPPUNIT_ASSERT ( font.data[249 * 32 + 4] == 0x00 );
    CPPUNIT_ASSERT ( font.data[249 * 32 + 5] == 0x00 );
    CPPUNIT_ASSERT ( font.data[249 * 32 + 6] == 0x00 );
    CPPUNIT_ASSERT ( font.data[249 * 32 + 7] == 0x18 );
    CPPUNIT_ASSERT ( font.data[249 * 32 + 8] == 0x18 );
    CPPUNIT_ASSERT ( font.data[249 * 32 + 9] == 0x00 );
    CPPUNIT_ASSERT ( font.data[249 * 32 + 10] == 0x00 );
    CPPUNIT_ASSERT ( font.data[249 * 32 + 11] == 0x00 );
    CPPUNIT_ASSERT ( font.data[249 * 32 + 12] == 0x00 );
    CPPUNIT_ASSERT ( font.data[249 * 32 + 13] == 0x00 );
    CPPUNIT_ASSERT ( font.data[249 * 32 + 14] == 0x00 );
    CPPUNIT_ASSERT ( font.data[249 * 32 + 15] == 0x00 );

    linux.finish();

    closeConEmuStdStreams();
    exit(EXIT_SUCCESS);
  }
  else  // Parent
  {
    // Start the terminal emulation
    startConEmuTerminal (ConEmu::console::linux_con);
    int wstatus;

    if ( waitpid(pid, &wstatus, WUNTRACED) != pid )
      std::cerr << "waitpid error" << std::endl;

    if ( WIFEXITED(wstatus) )
      CPPUNIT_ASSERT ( WEXITSTATUS(wstatus) == 0 );
  }
}

//----------------------------------------------------------------------
void FTermLinuxTest::modifierKeyTest()
{
  std::unique_ptr<finalcut::FSystem> fsys = std::make_unique<test::FSystemTest>();
  finalcut::FTerm::setFSystem(fsys);

  const auto& fsystem = finalcut::FSystem::getInstance();
  auto fsystest = static_cast<test::FSystemTest*>(fsystem.get());
  test::FSystemTest::ShiftState& mod_key = fsystest->getShiftState();
  finalcut::FTermLinux linux{};
  finalcut::FKey keycode{};
  finalcut::FKey mod_keycode{};
  linux.init();

  // Up key
  keycode = finalcut::FKey::Up;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Up );

  mod_key.shift = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Scroll_backward );

  mod_key.shift = 0;
  mod_key.ctrl = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Ctrl_up );

  mod_key.ctrl = 0;
  mod_key.alt = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Meta_up );

  mod_key.shift = 1;
  mod_key.ctrl = 1;
  mod_key.alt = 0;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Shift_Ctrl_up );

  mod_key.ctrl = 0;
  mod_key.alt = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Shift_Meta_up );

  mod_key.shift = 0;
  mod_key.ctrl = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Ctrl_Meta_up );

  mod_key.shift = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Shift_Ctrl_Meta_up );

  // Down key
  mod_key.shift = 0;
  mod_key.ctrl = 0;
  mod_key.alt = 0;
  keycode = finalcut::FKey::Down;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Down );

  mod_key.shift = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Scroll_forward );

  mod_key.shift = 0;
  mod_key.ctrl = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Ctrl_down );

  mod_key.ctrl = 0;
  mod_key.alt = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Meta_down );

  mod_key.shift = 1;
  mod_key.ctrl = 1;
  mod_key.alt = 0;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Shift_Ctrl_down );

  mod_key.ctrl = 0;
  mod_key.alt = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Shift_Meta_down );

  mod_key.shift = 0;
  mod_key.ctrl = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Ctrl_Meta_down );

  mod_key.shift = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Shift_Ctrl_Meta_down );

  // Left key
  mod_key.shift = 0;
  mod_key.ctrl = 0;
  mod_key.alt = 0;
  keycode = finalcut::FKey::Left;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Left );

  mod_key.shift = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Shift_left );

  mod_key.shift = 0;
  mod_key.ctrl = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Ctrl_left );

  mod_key.ctrl = 0;
  mod_key.alt = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Meta_left );

  mod_key.shift = 1;
  mod_key.ctrl = 1;
  mod_key.alt = 0;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Shift_Ctrl_left );

  mod_key.ctrl = 0;
  mod_key.alt = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Shift_Meta_left );

  mod_key.shift = 0;
  mod_key.ctrl = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Ctrl_Meta_left );

  mod_key.shift = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Shift_Ctrl_Meta_left );

  // Right key
  mod_key.shift = 0;
  mod_key.ctrl = 0;
  mod_key.alt = 0;
  keycode = finalcut::FKey::Right;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Right );

  mod_key.shift = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Shift_right );

  mod_key.shift = 0;
  mod_key.ctrl = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Ctrl_right );

  mod_key.ctrl = 0;
  mod_key.alt = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Meta_right );

  mod_key.shift = 1;
  mod_key.ctrl = 1;
  mod_key.alt = 0;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Shift_Ctrl_right );

  mod_key.ctrl = 0;
  mod_key.alt = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Shift_Meta_right );

  mod_key.shift = 0;
  mod_key.ctrl = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Ctrl_Meta_right );

  mod_key.shift = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Shift_Ctrl_Meta_right );

  // Insert key
  mod_key.shift = 0;
  mod_key.ctrl = 0;
  mod_key.alt = 0;
  keycode = finalcut::FKey::Insert;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Insert );

  mod_key.shift = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Shift_insert );

  mod_key.shift = 0;
  mod_key.ctrl = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Ctrl_insert );

  mod_key.ctrl = 0;
  mod_key.alt = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Meta_insert );

  mod_key.shift = 1;
  mod_key.ctrl = 1;
  mod_key.alt = 0;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Shift_Ctrl_insert );

  mod_key.ctrl = 0;
  mod_key.alt = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Shift_Meta_insert );

  mod_key.shift = 0;
  mod_key.ctrl = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Ctrl_Meta_insert );

  mod_key.shift = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Shift_Ctrl_Meta_insert );

  // Delete key
  mod_key.shift = 0;
  mod_key.ctrl = 0;
  mod_key.alt = 0;
  keycode = finalcut::FKey::Del_char;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Del_char );

  mod_key.shift = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Shift_del_char );

  mod_key.shift = 0;
  mod_key.ctrl = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Ctrl_del_char );

  mod_key.ctrl = 0;
  mod_key.alt = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Meta_del_char );

  mod_key.shift = 1;
  mod_key.ctrl = 1;
  mod_key.alt = 0;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Shift_Ctrl_del_char );

  mod_key.ctrl = 0;
  mod_key.alt = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Shift_Meta_del_char );

  mod_key.shift = 0;
  mod_key.ctrl = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Ctrl_Meta_del_char );

  mod_key.shift = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Shift_Ctrl_Meta_del_char );

  // Home key
  mod_key.shift = 0;
  mod_key.ctrl = 0;
  mod_key.alt = 0;
  keycode = finalcut::FKey::Home;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Home );

  mod_key.shift = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Shift_home );

  mod_key.shift = 0;
  mod_key.ctrl = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Ctrl_home );

  mod_key.ctrl = 0;
  mod_key.alt = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Meta_home );

  mod_key.shift = 1;
  mod_key.ctrl = 1;
  mod_key.alt = 0;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Shift_Ctrl_home );

  mod_key.ctrl = 0;
  mod_key.alt = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Shift_Meta_home );

  mod_key.shift = 0;
  mod_key.ctrl = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Ctrl_Meta_home );

  mod_key.shift = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Shift_Ctrl_Meta_home );

  // End key
  mod_key.shift = 0;
  mod_key.ctrl = 0;
  mod_key.alt = 0;
  keycode = finalcut::FKey::End;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::End );

  mod_key.shift = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Shift_end );

  mod_key.shift = 0;
  mod_key.ctrl = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Ctrl_end );

  mod_key.ctrl = 0;
  mod_key.alt = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Meta_end );

  mod_key.shift = 1;
  mod_key.ctrl = 1;
  mod_key.alt = 0;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Shift_Ctrl_end );

  mod_key.ctrl = 0;
  mod_key.alt = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Shift_Meta_end );

  mod_key.shift = 0;
  mod_key.ctrl = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Ctrl_Meta_end );

  mod_key.shift = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Shift_Ctrl_Meta_end );

  // Page Up key
  mod_key.shift = 0;
  mod_key.ctrl = 0;
  mod_key.alt = 0;
  keycode = finalcut::FKey::Page_up;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Page_up );

  mod_key.shift = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Shift_page_up );

  mod_key.shift = 0;
  mod_key.ctrl = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Ctrl_page_up );

  mod_key.ctrl = 0;
  mod_key.alt = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Meta_page_up );

  mod_key.shift = 1;
  mod_key.ctrl = 1;
  mod_key.alt = 0;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Shift_Ctrl_page_up );

  mod_key.ctrl = 0;
  mod_key.alt = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Shift_Meta_page_up );

  mod_key.shift = 0;
  mod_key.ctrl = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Ctrl_Meta_page_up );

  mod_key.shift = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Shift_Ctrl_Meta_page_up );

  // Page Down key
  mod_key.shift = 0;
  mod_key.ctrl = 0;
  mod_key.alt = 0;
  keycode = finalcut::FKey::Page_down;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Page_down );

  mod_key.shift = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Shift_page_down );

  mod_key.shift = 0;
  mod_key.ctrl = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Ctrl_page_down );

  mod_key.ctrl = 0;
  mod_key.alt = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Meta_page_down );

  mod_key.shift = 1;
  mod_key.ctrl = 1;
  mod_key.alt = 0;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Shift_Ctrl_page_down );

  mod_key.ctrl = 0;
  mod_key.alt = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Shift_Meta_page_down );

  mod_key.shift = 0;
  mod_key.ctrl = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Ctrl_Meta_page_down );

  mod_key.shift = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Shift_Ctrl_Meta_page_down );

  // Space key
  mod_key.shift = 0;
  mod_key.ctrl = 0;
  mod_key.alt = 0;
  keycode = finalcut::FKey::Space;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Space );

  mod_key.shift = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Space );

  mod_key.shift = 0;
  mod_key.ctrl = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Space );

  mod_key.ctrl = 0;
  mod_key.alt = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Space );

  mod_key.shift = 1;
  mod_key.ctrl = 1;
  mod_key.alt = 0;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Space );

  mod_key.ctrl = 0;
  mod_key.alt = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Space );

  mod_key.shift = 0;
  mod_key.ctrl = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Space );

  mod_key.shift = 1;
  mod_keycode = linux.modifierKeyCorrection(keycode);
  CPPUNIT_ASSERT ( mod_keycode == finalcut::FKey::Space );
}

// Put the test suite in the registry
CPPUNIT_TEST_SUITE_REGISTRATION (FTermLinuxTest);

// The general unit test main part
#include <main-test.inc>
