/* $Id: list.h,v 1.8 2003/06/15 10:05:55 sjoerd Exp $ */
#ifndef FFRENZY_LIST_H
#define FFRENZY_LIST_H

/** \defgroup list list
 * Generic list implementation. 
 * Data can be appended, prepended and ofcourse deleted.
 * \warning while the implementation uses reference counting it 
 * is not thread save
 */

typedef struct _List_node List_node;
struct _List_node {
  List_node *next;
  List_node *prev;
  void *data;
  int refcount;
};

typedef struct {
  List_node *head;
  List_node *tail;
  int length;
} List;

typedef struct {
  List_node *current;
  List *list;
} List_ptr;

/** \ingroup list
 * List constructor
 */
List *new_list(void);
/** \ingroup list
 * List destructor. Frees all the lists datatypes
 * \warning don't use list_ptrs of a list after deletion of the list
 */
void del_list(List * list);

/* data modification */
/** \ingroup list
 * Prepend an item with \a data to the \a list
 */
int list_prepend(List * list, void *data);
/** \ingroup list
 * Append an item with \a data to the \a list
 */
int list_append(List * list, void *data);

/** \ingroup list
  *  Removes the first item of the list and return its data.
  *  Returns \c NULL if the list is empty
  */
void *list_pop(List *list);

/** \ingroup list
 * Remove the item with \a data from the \a list.
 * Note that this doesn't free or touch \a data in any way
 */
int list_del(List * list, void *data);

/* utilities */
typedef int (*ListFunc) (void *data, void *user_data);

/** \ingroup list
 * Search the list for an item. 
 * \a cmp_func is an \code int(*func) (void *data, void user_data) \endcode
 * The current items user data is passed with the \a data param and \a
 * user_data is passed directly through from the list_search \p user_data.
 *
 * When cmp_func returns \c TRUE, the right item is found and its data is
 * returned. If the right data isn't found \c NULL will be returned 
 */
void *list_search(List * list, ListFunc cmp_func, void *user_data);
/** \ingroup list
 * Perform a function on all item of a list.
 *
 * \a do_func is an \code int(*func) (void *data, void user_data) \endcode
 * The current items user data is passed with the \a data param and \a
 * user_data is passed directly through from the list_search \p user_data.
 * 
 * All items will be passed one by one to the do_func, untill do_func returns
 * \c FALSE. 
 *
 * Returns \c TRUE when the end of the list was reached or \c FALSE if do_func
 * returned \c FALSE
 */
int list_foreach(List * list, ListFunc do_func, void *user_data);

/* pointer */
/** \ingroup list
 * List pointer constructor.
 */
List_ptr *new_list_ptr(List * list);
/** \ingroup list
 * List pointer destructor.
 */
void del_list_ptr(List_ptr * ptr);

/** \ingroup list
 * Duplicate given list pointer.
 */
List_ptr *list_ptr_dup(List_ptr * ptr);

/** \ingroup list
 * Returns the data at the pointers current position.
 * \warning This can return NULL if the current item has been deleted from the
 * list.
 */
void *list_ptr_get_data(List_ptr * ptr);
/** \ingroup list
 * Remove the item at the pointers position from the list.
 * Note that this doesn't touch the items data in any way.
 */
void list_ptr_del_current(List_ptr * ptr);

/** \ingroup list
 * point the list_ptr to the first item on the list.
 * Returns \c TRUE on success and \c FALSE on failure (empty list)
 */
int list_ptr_first(List_ptr * ptr);
/** \ingroup list
 * point the list_ptr to the last item on the list.
 * Returns \c TRUE on success and \c FALSE on failure (empty list)
 */
int list_ptr_last(List_ptr * ptr);

/** \ingroup list
 * point the list_ptr to the next item on the list.
 * Returns \c TRUE on success and \c FALSE on failure 
 * (reached end of the list or no position in the list)
 */
int list_ptr_next(List_ptr * ptr);
/** \ingroup list
 * point the list_ptr to the next item on the list.
 * Returns \c TRUE on success and \c FALSE on failure 
 * (reached beginning of the list or no position in the list)
 */
int list_ptr_prev(List_ptr * ptr);

int list_length(List *list);

/* sorting */
/* compare function to use with sorting  
 * Returns:
 * <0  - if data0 should be before data1
 *  0  - if data0 and data1 order doesn't matter
 * >0  - if data0 should be after data1
 */
typedef int (*ListSortFunc) (void *data0,void *data1, void *user_data);

int list_sort(List *list,ListSortFunc func, void *user_data);
#endif /* FFRENZY_LIST_H */
