/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file pwd_macos.c
 * \brief MacOS X KeyChain plugin: Store and find of passwords in MacOS keychain
 * \author Louis Lagendijk
 */

#include <ffgtk.h>
#include <CoreFoundation/CoreFoundation.h>
#include <Security/Security.h>
#include <CoreServices/CoreServices.h>


char serviceName[] = "ffgtk";

static gchar *pnSecretPassword = NULL;
static void macKeyringRemovePassword( struct sProfile *psProfile );

static const gchar *macKeyringGetPassword( struct sProfile *psProfile ) {
	OSStatus status;
	UInt32 passwdLength;
	void *pnPassword = NULL;

	if ( pnSecretPassword != NULL ) {
		g_free ( pnSecretPassword );
		pnSecretPassword = NULL;
	}

	if ( psProfile == NULL || psProfile -> pnName == NULL ) {
		return NULL;
	}

	/* Find password */

	status = SecKeychainFindGenericPassword (	
				NULL,
				strlen ( serviceName ),
				serviceName,
				strlen ( psProfile -> pnName ),
				psProfile -> pnName,
				&passwdLength,
				&pnPassword,
				NULL );
				
	/* Check return value, if no error occurred: return password */
	if ( status == noErr ) {
		pnSecretPassword = g_strndup ( (char *) pnPassword, passwdLength );
		SecKeychainItemFreeContent (
                                NULL,
                                pnPassword );
		return pnSecretPassword;
	}

	Debug( KERN_WARNING, "Couldn't find password: %s\n", GetMacOSStatusErrorString ( status ) );
	pnSecretPassword = NULL;

	return NULL;
}

/**
 * \brief Set password to macos keychain
 * \param psProfile profile structure
 * \param pnPassword profile password
 */
static void macKeyringSetPassword( struct sProfile *psProfile, const gchar *pnPassword ) {
	OSStatus status;
	SecKeychainItemRef itemRef = NULL;

	if ( psProfile == NULL || psProfile -> pnName == NULL ) {
		return;
	}

	/* store password */

        status = SecKeychainFindGenericPassword (
                                NULL,
                                strlen ( serviceName ),
                                serviceName,
                                strlen ( psProfile -> pnName ),
                                psProfile -> pnName,
                                NULL,
                                NULL,
                                &itemRef );
	if ( status == noErr ) {
		/* remove existing keychain entry. This is not a clean solution as we remove any access entries as well */
		/* macos however does not have a simple solution to just update a password */

		macKeyringRemovePassword ( psProfile );
	}
		
	/* insert new keychain entry */

	status = SecKeychainAddGenericPassword (
			NULL,
			strlen ( serviceName ),
			serviceName,
			strlen ( psProfile -> pnName ),
                        psProfile -> pnName,
			strlen ( pnPassword ),
			pnPassword,
			NULL );

	/* Check return value, if error occurred: show reason */
	if ( status != noErr ) {
		Debug( KERN_WARNING, "Couldn't store password: %s\n", GetMacOSStatusErrorString ( status ) );
	}
	if ( pnSecretPassword != NULL ) {
		g_free ( pnSecretPassword );
		pnSecretPassword = NULL;
	}
}


/**
 * \brief Remove password from macos keychain
 * \param psProfile profile structure
 * \param pnPassword profile password
 */
static void macKeyringRemovePassword( struct sProfile *psProfile ) {
	OSStatus status;
	SecKeychainItemRef itemRef = NULL;

	if ( psProfile == NULL || psProfile -> pnName == NULL ) {
		return;
	}

        status = SecKeychainFindGenericPassword (
                                NULL,
                                strlen ( serviceName ),
                                serviceName,
				strlen ( psProfile -> pnName ),
				psProfile -> pnName,
				NULL,
				NULL,
                                &itemRef );

	if ( status == noErr ) {
		g_assert ( itemRef != NULL );	
		status = SecKeychainItemDelete ( itemRef );
	}

	/* Check return value, if error occurred: show reason */
	if ( status != noErr ) {
		Debug( KERN_WARNING, "Couldn't remove password: %s\n", GetMacOSStatusErrorString ( status ) );
	}
}

/** password definition */
struct sPassword sKeyring = {
	macKeyringGetPassword,
	macKeyringSetPassword,
	macKeyringRemovePassword
};

MODULE_INIT( PLUGIN_TYPE_PASSWORD, _( "MacOS Keyring" ), &sKeyring, NULL, NULL );
