/*
 * $Id: faum-dev.c,v 1.53 2010-11-19 10:28:48 vrsieh Exp $
 *
 * Copyright (C) 2010 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#define DEBUG	0

#define FAUM_DEV_MAJOR	240
#define FAUM_DEV_MAX	8

#include <linux/version.h>
#include <linux/module.h>

#include <asm/uaccess.h>
#include <linux/delay.h>
#include <linux/fs.h>
#include <linux/interrupt.h>
#include <linux/pci.h>
#include <linux/poll.h>
#include <linux/sched.h>
#include <linux/signal.h>
#include <linux/wait.h>

#include "faum-dev.h"

MODULE_LICENSE("GPL");

static int faum_dev_major;
static struct faum_dev {
	struct pci_dev *pci_dev;
	struct fasync_struct *async_queue;
	wait_queue_head_t wait;
	uint8_t *mem[6];
	uint16_t port[6];

	uint32_t addr[6];
	unsigned long size[6];

	uint8_t config[64];

	int mem_enabled;
	int io_enabled;
	uint32_t base[6];
	uint8_t interrupt_line;

	int irq_test;
	int irq_state;
} faum_dev_device[FAUM_DEV_MAX];

static int
faum_dev_create(struct faum_dev *dev, struct faum_dev_info *info)
{
	unsigned int i;

	for (i = 0; i < 6; i++) {
		info->region[i].base = dev->addr[i];
		info->region[i].size = dev->size[i];
	}
	info->region[6].base = 0; /* FIXME */
	info->region[6].size = 0; /* FIXME */

	return 0;
}

static int
faum_dev_c0r(struct faum_dev *dev, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	uint32_t val32 = 0;
	uint8_t val8;

#if DEBUG
	printk(KERN_INFO "%s: called(addr=0x%x, bs=0x%x, valp=%p)\n", __FUNCTION__,
			addr, bs, valp);
#endif

	addr &= 0xfc;

	switch (addr) {
	case 0x00: /* Device ID, Vendor ID */
	case 0x04: /* Status, Command */
	case 0x08: /* Class Code, Revision ID */
	case 0x0c: /* BIST, Header Type, Latency Timer, Cache Line Size */
	case 0x28: /* Cardbus CIS Pointer */
	case 0x2c: /* Subsystem ID, Subsystem Vendor ID */
	case 0x34: /* Reserved */
	case 0x38: /* Reserved */
	default:
		if ((bs >> 0) & 1) {
			pci_read_config_byte(dev->pci_dev, addr + 0, &val8);
			if (addr == 0x04) {
				val8 &= ~0x03;
				val8 |= dev->mem_enabled << 1;
				val8 |= dev->io_enabled << 0;
			}
			val32 |= val8 << 0;
		}
		if ((bs >> 1) & 1) {
			pci_read_config_byte(dev->pci_dev, addr + 1, &val8);
			val32 |= val8 << 8;
		}
		if ((bs >> 2) & 1) {
			pci_read_config_byte(dev->pci_dev, addr + 2, &val8);
			val32 |= val8 << 16;
		}
		if ((bs >> 3) & 1) {
			pci_read_config_byte(dev->pci_dev, addr + 3, &val8);
			val32 |= val8 << 24;
		}
		break;
	case 0x10: /* Base Address 0 */
	case 0x14: /* Base Address 1 */
	case 0x18: /* Base Address 2 */
	case 0x1c: /* Base Address 3 */
	case 0x20: /* Base Address 4 */
	case 0x24: /* Base Address 5 */
		if (dev->config[addr + 0] & 1) {
			/* I/O */
			val32 = dev->base[(addr - 0x10) / 4]
			      | (dev->config[addr + 0] & 0x3);
		} else {
			/* Mem */
			val32 = dev->base[(addr - 0x10) / 4]
			      | (dev->config[addr + 0] & 0xf);
		}
		break;
	case 0x30: /* Expansion ROM Base Address */
		/* FIXME */
		printk(KERN_INFO "%s: addr=0x%02x\n", __FUNCTION__, addr);
		val32 = (dev->config[addr + 3] << 24)
		      | (dev->config[addr + 2] << 16)
		      | (dev->config[addr + 1] << 8)
		      | (dev->config[addr + 0] << 0);
		break;
	case 0x3c: /* Max_Lat, Min_Gnt, Interrupt Pin, Interrupt Line */
		val32 = (dev->config[addr + 3] << 24)
		      | (dev->config[addr + 2] << 16)
		      | (dev->config[addr + 1] << 8)
		      | (dev->interrupt_line << 0);
		break;
	}

	*valp = val32;

	return 0;
}

static int
faum_dev_c0w(struct faum_dev *dev, uint32_t addr, unsigned int bs, uint32_t val)
{
	uint8_t val8;

#if DEBUG
	printk(KERN_INFO "%s: called(addr=0x%x, bs=0x%x, val=0x%x)\n", __FUNCTION__,
			addr, bs, val);
#endif

	addr &= 0xfc;

	switch (addr) {
	case 0x00: /* Device ID, Vendor ID */
	case 0x04: /* Status, Command */
	case 0x08: /* Class Code, Revision ID */
	case 0x0c: /* BIST, Header Type, Latency Timer, Cache Line Size */
	case 0x28: /* Cardbus CIS Pointer */
	case 0x2c: /* Subsystem ID, Subsystem Vendor ID */
	case 0x34: /* Reserved */
	case 0x38: /* Reserved */
	default:
		if ((bs >> 0) & 1) {
			val8 = (val >> 0) & 0xff;
			if (addr == 0x04) {
				dev->mem_enabled = (val8 >> 1) & 1;
				dev->io_enabled = (val8 >> 0) & 1;
				val8 |= 0x03;
			}
			pci_write_config_byte(dev->pci_dev, addr + 0, val8);
		}
		if ((bs >> 1) & 1) {
			val8 = (val >> 8) & 0xff;
			pci_write_config_byte(dev->pci_dev, addr + 1, val8);
		}
		if ((bs >> 2) & 1) {
			val8 = (val >> 16) & 0xff;
			pci_write_config_byte(dev->pci_dev, addr + 2, val8);
		}
		if ((bs >> 3) & 1) {
			val8 = (val >> 24) & 0xff;
			pci_write_config_byte(dev->pci_dev, addr + 3, val8);
		}
		break;
	case 0x10: /* Base Address 0 */
	case 0x14: /* Base Address 1 */
	case 0x18: /* Base Address 2 */
	case 0x1c: /* Base Address 3 */
	case 0x20: /* Base Address 4 */
	case 0x24: /* Base Address 5 */ {
		uint32_t base;

		base = dev->base[(addr - 0x10) / 4];

		if ((bs >> 0) & 1) {
			base &= ~0x000000ff;
			base |= val & 0x000000ff;
		}
		if ((bs >> 1) & 1) {
			base &= ~0x0000ff00;
			base |= val & 0x0000ff00;
		}
		if ((bs >> 2) & 1) {
			base &= ~0x00ff0000;
			base |= val & 0x00ff0000;
		}
		if ((bs >> 3) & 1) {
			base &= ~0xff000000;
			base |= val & 0xff000000;
		}
		base &= ~(dev->size[(addr - 0x10) / 4] - 1);

		dev->base[(addr - 0x10) / 4] = base;
		break;
	    }
	case 0x30: /* Expansion ROM Base Address */
		/* FIXME */
		printk(KERN_INFO "%s: addr=0x%02x\n", __FUNCTION__, addr);
		break;
	case 0x3c: /* Max_Lat, Min_Gnt, Interrupt Pin, Interrupt Line */
		/* FIXME */
		if ((bs >> 0) & 1) {
			dev->interrupt_line = (val >> 0) & 0xff;
		}
		break;
	}

	return 0;
}

static int
faum_dev_memr(struct faum_dev *dev, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	unsigned int region;
	int ret;

#if DEBUG
	printk(KERN_INFO "%s: called(addr=0x%x, bs=0x%x, valp=%p)\n", __FUNCTION__,
			addr, bs, valp);
#endif

	for (region = 0; ; region++) {
		if (region == 6) {
			ret = -EINVAL;
			break;
		}
		if (dev->mem_enabled
		 && dev->mem[region]
		 && dev->base[region] <= addr
		 && addr < dev->base[region] + dev->size[region]) {
			uint8_t *mem;

			mem = dev->mem[region];
			mem += addr - dev->base[region];

#if DEBUG
			printk(KERN_INFO "reading from %p\n", mem);
#endif
			if (mem < dev->mem[region]
			 || dev->mem[region] + dev->size[region] <= mem) {
				return -EINVAL;
			}

			switch (bs) {
			case 0x1:
				*valp = *(uint8_t *) (mem + 0) << 0;
				break;
			case 0x2:
				*valp = *(uint8_t *) (mem + 1) << 8;
				break;
			case 0x4:
				*valp = *(uint8_t *) (mem + 2) << 16;
				break;
			case 0x8:
				*valp = *(uint8_t *) (mem + 3) << 24;
				break;
			case 0x3:
				*valp = *(uint16_t *) (mem + 0) << 0;
				break;
			case 0xc:
				*valp = *(uint16_t *) (mem + 2) << 16;
				break;
			case 0xf:
				*valp = *(uint32_t *) (mem + 0) << 0;
				break;
			}
			ret = 0;
			break;
		}
	}

	return ret;
}

static int
faum_dev_memw(struct faum_dev *dev, uint32_t addr, unsigned int bs, uint32_t val)
{
	unsigned int region;
	int ret;

#if DEBUG
	printk(KERN_INFO "%s: called(addr=0x%x, bs=0x%x, val=0x%x)\n",
			__FUNCTION__, addr, bs, val);
#endif

	for (region = 0; ; region++) {
		if (region == 6) {
			ret = -EINVAL;
			break;
		}
		if (dev->mem_enabled
		 && dev->mem[region]
		 && dev->base[region] <= addr
		 && addr < dev->base[region] + dev->size[region]) {
			uint8_t *mem;

			mem = dev->mem[region];
			mem += addr - dev->base[region];

#if DEBUG
			printk(KERN_INFO "writing to %p\n", mem);
#endif
			if (mem < dev->mem[region]
			 || dev->mem[region] + dev->size[region] <= mem) {
				return -EINVAL;
			}

			switch (bs) {
			case 0x1:
				*(uint8_t *) (mem + 0) = val >> 0;
				break;
			case 0x2:
				*(uint8_t *) (mem + 1) = val >> 8;
				break;
			case 0x4:
				*(uint8_t *) (mem + 2) = val >> 16;
				break;
			case 0x8:
				*(uint8_t *) (mem + 3) = val >> 24;
				break;
			case 0x3:
				*(uint16_t *) (mem + 0) = val >> 0;
				break;
			case 0xc:
				*(uint16_t *) (mem + 2) = val >> 16;
				break;
			case 0xf:
				*(uint32_t *) (mem + 0) = val >> 0;
				break;
			}
			ret = 0;
			break;
		}
	}

	return ret;
}

static int
faum_dev_ior(struct faum_dev *dev, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	unsigned int region;
	int ret;

#if DEBUG
	printk(KERN_INFO "%s: called(addr=0x%x, bs=0x%x, valp=%p)\n", __FUNCTION__,
			addr, bs, valp);
#endif

	for (region = 0; ; region++) {
		if (region == 6) {
			ret = -EINVAL;
			break;
		}
		if (dev->io_enabled
		 && dev->port[region]
		 && dev->base[region] <= addr
		 && addr < dev->base[region] + dev->size[region]) {
			uint16_t port;

			port = dev->port[region];
			port += addr - dev->base[region];

#if DEBUG
			printk(KERN_INFO "in from 0x%x\n", port);
#endif
			if (port < dev->port[region]
			 || dev->port[region] + dev->size[region] <= port) {
				return -EINVAL;
			}

			switch (bs) {
			case 0x1:
				*valp = inb(port + 0) << 0;
				break;
			case 0x2:
				*valp = inb(port + 1) << 8;
				break;
			case 0x4:
				*valp = inb(port + 2) << 16;
				break;
			case 0x8:
				*valp = inb(port + 3) << 24;
				break;
			case 0x3:
				*valp = inw(port + 0) << 0;
				break;
			case 0xc:
				*valp = inw(port + 2) << 16;
				break;
			case 0xf:
				*valp = inl(port + 0) << 0;
				break;
			}

			ret = 0;
			break;
		}
	}

	return ret;
}

static int
faum_dev_iow(struct faum_dev *dev, uint32_t addr, unsigned int bs, uint32_t val)
{
	unsigned int region;
	int ret;

#if DEBUG
	printk(KERN_INFO "%s: called(addr=0x%x, bs=0x%x, val=0x%x)\n", __FUNCTION__,
			addr, bs, val);
#endif

	for (region = 0; ; region++) {
		if (region == 6) {
			ret = -EINVAL;
			break;
		}
		if (dev->io_enabled
		 && dev->port[region]
		 && dev->base[region] <= addr
		 && addr < dev->base[region] + dev->size[region]) {
			uint16_t port;

			port = dev->port[region];
			port += addr - dev->base[region];

#if DEBUG
			printk(KERN_INFO "out to 0x%x\n", port);
#endif
			if (port < dev->port[region]
			 || dev->port[region] + dev->size[region] <= port) {
				return -EINVAL;
			}

			switch (bs) {
			case 0x1:
				outb(val >> 0, port + 0);
				break;
			case 0x2:
				outb(val >> 8, port + 1);
				break;
			case 0x4:
				outb(val >> 16, port + 2);
				break;
			case 0x8:
				outb(val >> 24, port + 3);
				break;
			case 0x3:
				outw(val >> 0, port + 0);
				break;
			case 0xc:
				outw(val >> 16, port + 2);
				break;
			case 0xf:
				outl(val >> 0, port + 0);
				break;
			}

			ret = 0;
			break;
		}
	}

	return ret;
}

static ssize_t
faum_dev_read(struct file *file, char __user *buf, size_t count, loff_t *ppos)
{
	struct faum_dev *dev = file->private_data;
	uint8_t irq_state;
	int ret;

#if DEBUG
	printk(KERN_INFO "faum_dev: read called\n");
#endif

	irq_state = dev->irq_state;

	ret = put_user(irq_state, (uint8_t __user *) buf);
	if (ret) {
		/* put_user failed. */
		return ret;
	}

	return 1;
}

static int
faum_dev_ioctl(
	struct inode *inode,
	struct file *file,
	unsigned int cmd,
	unsigned long _param
)
{
	struct faum_dev *dev = file->private_data;
	struct faum_dev_info info;
	struct faum_dev_param param;
	int ret;

#if DEBUG
	printk(KERN_INFO "faum_dev: ioctl called\n");
#endif

	/*
	 * Get parameter.
	 */
	switch (cmd) {
	case FAUM_DEV_CREATE:
	case FAUM_DEV_DESTROY:
		if (copy_from_user(&info, (void *) _param, sizeof(info)) != 0) {
			ret = -EFAULT;
			goto bad;
		}
		break;

	case FAUM_DEV_C0R:
	case FAUM_DEV_C0W:
	case FAUM_DEV_MEMR:
	case FAUM_DEV_MEMW:
	case FAUM_DEV_IOR:
	case FAUM_DEV_IOW:
		if (copy_from_user(&param, (void *) _param, sizeof(param)) != 0) {
			ret = -EFAULT;
			goto bad;
		}
		break;

	default:
		ret = -EINVAL;
		goto bad;
	}
	
	/*
	 * Do work.
	 */
	switch (cmd) {
	case FAUM_DEV_CREATE:
		ret = faum_dev_create(dev, &info);
		break;
	case FAUM_DEV_DESTROY:
		/* FIXME */
		ret = 0;
		break;

	case FAUM_DEV_C0R:
		ret = faum_dev_c0r(dev, param.addr, param.bs, &param.data);
		break;
	case FAUM_DEV_C0W:
		ret = faum_dev_c0w(dev, param.addr, param.bs, param.data);
		break;
	case FAUM_DEV_MEMR:
		ret = faum_dev_memr(dev, param.addr, param.bs, &param.data);
		break;
	case FAUM_DEV_MEMW:
		ret = faum_dev_memw(dev, param.addr, param.bs, param.data);
		break;
	case FAUM_DEV_IOR:
		ret = faum_dev_ior(dev, param.addr, param.bs, &param.data);
		break;
	case FAUM_DEV_IOW:
		ret = faum_dev_iow(dev, param.addr, param.bs, param.data);
		break;
	default:
		BUG();
	}

	/*
	 * Check interrupt state.
	 */
	switch (cmd) {
	case FAUM_DEV_CREATE:
	case FAUM_DEV_DESTROY:
		break;

	case FAUM_DEV_C0R:
	case FAUM_DEV_C0W:
	case FAUM_DEV_MEMR:
	case FAUM_DEV_MEMW:
	case FAUM_DEV_IOR:
	case FAUM_DEV_IOW:
		if (0 <= ret
		 && dev->irq_state) {
			dev->irq_test = 1;
			dev->irq_state = 0;
			enable_irq(dev->pci_dev->irq);
			mdelay(1); /* Wait for IRQ to re-occur. */
			dev->irq_test = 0;
		}
		param.irq = dev->irq_state;
		break;

	default:
		BUG();
	}

	/*
	 * Set return values.
	 */
	switch (cmd) {
	case FAUM_DEV_CREATE:
	case FAUM_DEV_DESTROY:
		if (copy_to_user((void *) _param, &info, sizeof(info)) != 0) {
			ret = -EFAULT;
			goto bad;
		}
		break;

	case FAUM_DEV_C0R:
	case FAUM_DEV_C0W:
	case FAUM_DEV_MEMR:
	case FAUM_DEV_MEMW:
	case FAUM_DEV_IOR:
	case FAUM_DEV_IOW:
		if (copy_to_user((void *) _param, &param, sizeof(param)) != 0) {
			ret = -EFAULT;
			goto bad;
		}
		break;

	default:
		BUG();
	}

bad:	;
#if DEBUG
	printk(KERN_INFO "faum_dev: ioctl done\n");
#endif

	return ret;
}

static int
faum_dev_fasync(int fd, struct file *file, int on)
{
	struct faum_dev *dev = file->private_data;

#if DEBUG
	printk(KERN_INFO "faum_dev: fasync called\n");
#endif

        return fasync_helper(fd, file, on, &dev->async_queue);
}

static unsigned int
faum_dev_poll(struct file *file, poll_table *wait)
{
	struct faum_dev *dev = file->private_data;

#if DEBUG
	printk(KERN_INFO "faum_dev: poll called\n");
#endif

	poll_wait(file, &dev->wait, wait);

	return dev->irq_state;
}

static int
faum_dev_open(
	struct inode *inode,
	struct file *file
)
{
	unsigned int minor = iminor(inode);

	if (minor < 0
	 || FAUM_DEV_MAX <= minor
	 || ! faum_dev_device[minor].pci_dev) {
		return -ENODEV;
	}

	file->private_data = &faum_dev_device[minor];

	return 0;
}

static int
faum_dev_release(
	struct inode *inode,
	struct file *file
)
{
	if (file->f_flags & FASYNC)
		faum_dev_fasync(-1, file, 0);

	return 0;
}

static irqreturn_t
faum_dev_isr(int irq, void *_dev)
{
	struct faum_dev *dev = _dev;

#if DEBUG
	printk(KERN_INFO "%s called\n", __FUNCTION__);
#endif

	disable_irq_nosync(irq);

	if (dev->irq_test) {
		/* Just polling... */
		dev->irq_state = 1;
	} else {
		/* New interrupt. */
		dev->irq_state = 1;
		wake_up_interruptible(&dev->wait);
		kill_fasync(&dev->async_queue, SIGIO, POLL_IN);
	}

	return IRQ_HANDLED;
}

static int __devinit
faum_dev_probe(struct pci_dev *dev, const struct pci_device_id *id)
{
	unsigned int minor;
	unsigned int addr;
	unsigned int region;

#if DEBUG
	printk(KERN_INFO "faum_dev: probe called\n");
#endif

	/*
	 * Lookup free state structure.
	 */
	for (minor = 0; ; minor++) {
		if (minor == FAUM_DEV_MAX) {
			return -ENOMEM;
		}
		if (! faum_dev_device[minor].pci_dev) {
			break;
		}
	}
	faum_dev_device[minor].pci_dev = dev;

	/*
	 * Request hardware resources.
	 */
	if (pci_enable_device(dev) < 0) {
		printk(KERN_ERR "faum_dev: unable to enable device!\n");
		faum_dev_device[minor].pci_dev = NULL;
		return -EIO;
	}

	if (pci_request_regions(dev, "faum-dev") < 0) {
		printk(KERN_ERR "faum_dev: unable to request regions!\n");
		pci_disable_device(dev);
		faum_dev_device[minor].pci_dev = NULL;
		return -EBUSY;
	}

	faum_dev_device[minor].irq_test = 0;
	faum_dev_device[minor].irq_state = 0;

	if (request_irq(dev->irq, faum_dev_isr, 0,
			"faum-dev", &faum_dev_device[minor])) {
		printk(KERN_ERR "faum_dev: unable to request IRQ!\n");
		pci_release_regions(dev);
		pci_disable_device(dev);
		faum_dev_device[minor].pci_dev = NULL;
		return -EBUSY;
	}

	pci_set_master(dev); /* Dangerous! FIXME */

	/*
	 * Initialize state.
	 */
	faum_dev_device[minor].async_queue = NULL;
	init_waitqueue_head(&faum_dev_device[minor].wait);

	for (region = 0; region < 6; region++) {
		unsigned long base = pci_resource_start(dev, region);
		unsigned long size = pci_resource_len(dev, region);
		unsigned long flags = pci_resource_flags(dev, region);

		if (size
		 && (flags & (IORESOURCE_MEM | IORESOURCE_IO)) == IORESOURCE_MEM) {
			faum_dev_device[minor].mem[region] = ioremap(base, size);
			faum_dev_device[minor].port[region] = 0;
		} else {
			faum_dev_device[minor].mem[region] = NULL;
			faum_dev_device[minor].port[region] = base;
		}

		pci_read_config_dword(dev, 0x10 + region * 4,
				&faum_dev_device[minor].addr[region]);
		faum_dev_device[minor].size[region] = size;

		faum_dev_device[minor].base[region] = 0x00000000;
	}
	faum_dev_device[minor].mem_enabled = 0;
	faum_dev_device[minor].io_enabled = 0;

	for (addr = 0; addr < 64; addr++) {
		pci_read_config_byte(dev, addr,
				&faum_dev_device[minor].config[addr]);
	}

	/*
	 * Print info.
	 */
	for (region = 0; region < 6; region++) {
		printk(KERN_INFO "addr[%d] = 0x%08x\n",
				region, faum_dev_device[minor].addr[region]);
		printk(KERN_INFO "size[%d] = 0x%08lx\n",
				region, faum_dev_device[minor].size[region]);

		printk(KERN_INFO "mem[%d] = %p\n",
				region, faum_dev_device[minor].mem[region]);
		printk(KERN_INFO "port[%d] = 0x%04x\n",
				region, faum_dev_device[minor].port[region]);
	}

	return 0;
}

static void __devexit
faum_dev_remove(struct pci_dev *dev)
{
	unsigned int minor;
	unsigned int region;

#if DEBUG
	printk(KERN_INFO "faum_dev: remove called\n");
#endif

	for (minor = 0; ; minor++) {
		if (minor == FAUM_DEV_MAX) {
			printk(KERN_ERR "faum_dev: removing not registered device!\n");
			return;
		}
		if (faum_dev_device[minor].pci_dev == dev) {
			break;
		}
	}

	for (region = 0; region < 6; region++) {
		if (faum_dev_device[minor].mem[region]) {
			iounmap(faum_dev_device[minor].mem[region]);
		}
	}
	faum_dev_device[minor].pci_dev = NULL;

	free_irq(dev->irq, &faum_dev_device[minor]);

	pci_release_regions(dev);

	pci_disable_device(dev);
}

static const struct pci_device_id __devinitdata pci_device[] = {
	{
		0x10de, /* Vendor */
		0x05e3, /* Device */
		PCI_ANY_ID, /* Sub-Vendor */
		PCI_ANY_ID, /* Sub-Device */
		0, 0, 0
	},
	{
		0x10de, /* Vendor */
		0x05e7, /* Device */
		PCI_ANY_ID, /* Sub-Vendor */
		PCI_ANY_ID, /* Sub-Device */
		0, 0, 0
	},
	{ 0, 0, 0, 0, 0, 0, 0 }
};

MODULE_DEVICE_TABLE(pci, pci_device);

static struct pci_driver faum_dev_driver = {
	.name = "faum-dev",
	.id_table = pci_device,
	.probe = faum_dev_probe,
	.remove = __devexit_p(faum_dev_remove),
};

static int __init
faum_dev_init(void)
{
	static const struct file_operations fops = {
		.owner = THIS_MODULE,

		.llseek = no_llseek,
		.read = faum_dev_read,
		.ioctl = faum_dev_ioctl,
		.fasync = faum_dev_fasync,
		.poll = faum_dev_poll,

		.open = faum_dev_open,
		.release = faum_dev_release,
	};

#if DEBUG
	printk(KERN_INFO "faum_dev: init called\n");
#endif

	if (register_chrdev(FAUM_DEV_MAJOR, "faum-dev", &fops) != 0) {
		return -EIO;
	}

	if (pci_register_driver(&faum_dev_driver) < 0) {
		unregister_chrdev(faum_dev_major, "faum-dev");
		return -EIO;
	}

	return 0;
}

static void __exit
faum_dev_exit(void)
{
#if DEBUG
	printk(KERN_INFO "faum_dev: exit called\n");
#endif

	pci_unregister_driver(&faum_dev_driver);

	unregister_chrdev(FAUM_DEV_MAJOR, "faum-dev");
}

module_init(faum_dev_init);
module_exit(faum_dev_exit);
