# -*- coding: utf-8 -*-

# Copyright (c) 2017 - 2020 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing the label to show some SSL info.
"""


from PyQt5.QtCore import Qt, pyqtSlot, pyqtSignal, QPoint
from PyQt5.QtWidgets import QLabel


class SafeBrowsingLabel(QLabel):
    """
    Class implementing a label to show some Safe Browsing info.
    
    @signal clicked(pos) emitted to indicate a click of the label (QPoint)
    """
    clicked = pyqtSignal(QPoint)
    
    nokStyle = "QLabel { color : white; background-color : red; }"
    
    def __init__(self, parent=None):
        """
        Constructor
        
        @param parent reference to the parent widget (QWidget)
        """
        super(SafeBrowsingLabel, self).__init__(parent)
        
        self.setFocusPolicy(Qt.NoFocus)
        self.setCursor(Qt.PointingHandCursor)
        
        self.setStyleSheet(SafeBrowsingLabel.nokStyle)
        
        self.__threatType = ""
        self.__threatMessages = ""
        
        self.__deafultText = self.tr("Malicious Site")
        self.__updateLabel()
    
    def mouseReleaseEvent(self, evt):
        """
        Protected method to handle mouse release events.
        
        @param evt reference to the mouse event (QMouseEvent)
        """
        if evt.button() == Qt.LeftButton:
            self.clicked.emit(evt.globalPos())
        else:
            super(SafeBrowsingLabel, self).mouseReleaseEvent(evt)
    
    def mouseDoubleClickEvent(self, evt):
        """
        Protected method to handle mouse double click events.
        
        @param evt reference to the mouse event (QMouseEvent)
        """
        if evt.button() == Qt.LeftButton:
            self.clicked.emit(evt.globalPos())
        else:
            super(SafeBrowsingLabel, self).mouseDoubleClickEvent(evt)
    
    @pyqtSlot()
    def __updateLabel(self):
        """
        Private slot to update the label text.
        """
        if self.__threatType:
            self.setText(self.__threatType)
        else:
            self.setText(self.__deafultText)
    
    @pyqtSlot(str, str)
    def setThreatInfo(self, threatType, threatMessages):
        """
        Public slot to set threat information.
        
        @param threatType threat type
        @type str
        @param threatMessages more verbose info about detected threats
        @type str
        """
        self.__threatType = threatType
        self.__threatMessages = threatMessages
        
        self.__updateLabel()
    
    def getThreatInfo(self):
        """
        Public method to get the threat info text.
        
        @return threat info text
        @rtype str
        """
        return self.__threatMessages
