/*******************************************************************************
 * Copyright (c) 2007, 2008 Wind River Systems, Inc. and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Anton Leherbauer (Wind River Systems) - initial API and implementation
 *     Markus Schorn (Wind River Systems)
 *******************************************************************************/

package org.eclipse.cdt.internal.ui.editor;

import java.util.ResourceBundle;

import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.texteditor.ITextEditor;
import org.eclipse.ui.texteditor.TextEditorAction;

import org.eclipse.cdt.core.model.CModelException;
import org.eclipse.cdt.core.model.ITranslationUnit;
import org.eclipse.cdt.core.model.IWorkingCopy;
import org.eclipse.cdt.ui.CUIPlugin;
import org.eclipse.cdt.ui.IWorkingCopyManager;

import org.eclipse.cdt.internal.ui.util.EditorUtility;

/**
 * Editor action to toggle between source and header files.
 * 
 * @since 4.0
 */
public class ToggleSourceAndHeaderAction extends TextEditorAction {

	private static ITranslationUnit fgLastPartnerUnit;
	private static ITranslationUnit fgLastSourceUnit;

	/**
	 * Create a toggle source/header action for the given editor.
	 * 
	 * @param bundle  the resource bundle to take the label, tooltip and description from.
	 * @param prefix  the prefix to be prepended to the resource bundle keys
	 * @param editor  the text editor this action is associated with
	 * @see TextEditorAction#TextEditorAction(ResourceBundle, String, ITextEditor)
	 */
	public ToggleSourceAndHeaderAction(ResourceBundle bundle, String prefix, ITextEditor editor) {
		super(bundle, prefix, editor);
	}

	/*
	 * @see org.eclipse.jface.action.Action#run()
	 */
	@Override
	public void run() {
		IWorkingCopy currentUnit= getWorkingCopy();
		if (currentUnit == null) {
			return;
		}
		ITranslationUnit partnerUnit= computePartnerFile(currentUnit);
		if (partnerUnit != null) {
			fgLastSourceUnit= currentUnit.getOriginalElement();
			fgLastPartnerUnit= partnerUnit;
			try {
				EditorUtility.openInEditor(partnerUnit);
			} catch (PartInitException exc) {
				CUIPlugin.log(exc.getStatus());
			} catch (CModelException exc) {
				CUIPlugin.log(exc.getStatus());
			}
		}
	}

	private IWorkingCopy getWorkingCopy() {
		IEditorPart editor = getTextEditor();
		if (editor == null) {
			return null;
		}
		IEditorInput input= editor.getEditorInput();
		IWorkingCopyManager manager= CUIPlugin.getDefault().getWorkingCopyManager();				
		return manager.getWorkingCopy(input);
	}

	/*
	 * @see org.eclipse.ui.texteditor.TextEditorAction#update()
	 */
	@Override
	public void update() {
		setEnabled(getWorkingCopy() != null);
	}

	/**
	 * Compute the corresponding translation unit for the given unit.
	 * 
	 * @param tUnit  the current source/header translation unit
	 * @return the partner translation unit
	 */
	private ITranslationUnit computePartnerFile(ITranslationUnit tUnit) {
		// try shortcut for fast toggling
		if (fgLastPartnerUnit != null) {
			final ITranslationUnit originalUnit;
			if (tUnit instanceof IWorkingCopy) {
				originalUnit= ((IWorkingCopy)tUnit).getOriginalElement();
			} else {
				originalUnit= tUnit;
			}
			if (originalUnit.getTranslationUnit().equals(fgLastPartnerUnit)) {
				if (fgLastSourceUnit.exists()) {
					// toggle back
					return fgLastSourceUnit;
				}
			}
		}

		// search partner file based on filename/extension
		return SourceHeaderPartnerFinder.getPartnerTranslationUnit(tUnit);
	}
}
