#ifndef DBA_MSG_VARS_H
#define DBA_MSG_VARS_H

/** @file
 * @ingroup msg
 * Shortcut IDs and functions to quickly refer to commonly used values inside a
 * ::dba_msg.
 *
 * The shortcuts are defined in the file vars.csv, which maps a name to the tuple
 * (preferred C type, BLocal, Level type, L1, L2, P indicator, P1, P2).  The file
 * vars.csv is used to generate code (vars.h and vars.c) that provides a set of
 * getters and setters for every named value.
 */

#include <wreport/varinfo.h>

namespace dballe {

/**
 * Full data corresponding to a shortcut ID
 */
struct MsgVarShortcut {
    /** Variable description code */
    wreport::Varcode code;
    /** Type of the first level.  See @ref level_table. */
    int ltype1;
    /** L1 value of the level.  See @ref level_table. */
    int l1;
    /** Type of the second level.  See @ref level_table. */
    int ltype2;
    /** L2 value of the level.  See @ref level_table. */
    int l2;
    /** Time range type indicator.  See @ref trange_table. */
    int pind;
    /** Time range P1 indicator.  See @ref trange_table. */
    int p1;
    /** Time range P2 indicator.  See @ref trange_table. */
    int p2;
};

/** Array with all the full expansions of the shortcut IDs, indexed by shortcut
  * ID
  */
extern struct MsgVarShortcut shortcutTable[];

/** Resolve a var name to a var ID */
int resolve_var(const char* name);

/** Resolve a var name to a var ID, avoiding to compute the length of name */
int resolve_var_substring(const char* name, int len);

/** WMO block number */
#define DBA_MSG_BLOCK 0
/** WMO station number */
#define DBA_MSG_STATION 1
/** Aircraft registration number or other identification */
#define DBA_MSG_FLIGHT_REG_NO 2
/** Mobile station identifier */
#define DBA_MSG_IDENT 3
/** Station or site name */
#define DBA_MSG_ST_NAME 4
/** ICAO location indicator */
#define DBA_MSG_ST_NAME_ICAO 5
/** Report mnemonic */
#define DBA_MSG_REP_MEMO 6
/** Air quality observing station local code */
#define DBA_MSG_POLL_LCODE 7
/** Airbase air quality observing station code */
#define DBA_MSG_POLL_SCODE 8
/** GEMS air quality observing station code */
#define DBA_MSG_POLL_GEMSCODE 9
/** Air quality observing station dominant emission source */
#define DBA_MSG_POLL_SOURCE 10
/** Air quality observing station area type */
#define DBA_MSG_POLL_ATYPE 11
/** Air quality observing station terrain type */
#define DBA_MSG_POLL_TTYPE 12
/** Type of station */
#define DBA_MSG_ST_TYPE 13
/** Type of instrumentation for wind measurement */
#define DBA_MSG_WIND_INST 14
/** Precision of temperature observation */
#define DBA_MSG_TEMP_PRECISION 15
/** Aircraft navigational system */
#define DBA_MSG_NAVSYS 16
/** Aircraft data relay system */
#define DBA_MSG_DATA_RELAY 17
/** Aircraft roll angle */
#define DBA_MSG_FLIGHT_ROLL 18
/** Original specification of latitude/longitude */
#define DBA_MSG_LATLON_SPEC 19
/** Year of the observation */
#define DBA_MSG_YEAR 20
/** Month of the observation */
#define DBA_MSG_MONTH 21
/** Day of the observation */
#define DBA_MSG_DAY 22
/** Hour of the observation */
#define DBA_MSG_HOUR 23
/** Minute of the observation */
#define DBA_MSG_MINUTE 24
/** Second of the observation */
#define DBA_MSG_SECOND 25
/** Latitude of the station */
#define DBA_MSG_LATITUDE 26
/** Longiture of the station */
#define DBA_MSG_LONGITUDE 27
/** Isobaric surface */
#define DBA_MSG_ISOBARIC_SURFACE 28
/** Height of release of sonde above msl */
#define DBA_MSG_HEIGHT_RELEASE 29
/** Height of station */
#define DBA_MSG_HEIGHT_STATION 30
/** Height of barometer above mean sea level */
#define DBA_MSG_HEIGHT_BARO 31
/** Phase of aircraft flight */
#define DBA_MSG_FLIGHT_PHASE 32
/** Time significance */
#define DBA_MSG_TIMESIG 33
/** Station elevation quality mark (for mobile stations) */
#define DBA_MSG_STATION_HEIGHT_QUALITY 34
/** Total cloud cover (N) */
#define DBA_MSG_CLOUD_N 35
/** Cloud amount (N4) */
#define DBA_MSG_CLOUD_N4 36
/** Cloud amount (C4) */
#define DBA_MSG_CLOUD_C4 37
/** Height of base of cloud (H4) */
#define DBA_MSG_CLOUD_H4 38
/** Cloud amount (N3) */
#define DBA_MSG_CLOUD_N3 39
/** Cloud amount (C3) */
#define DBA_MSG_CLOUD_C3 40
/** Height of base of cloud (H3) */
#define DBA_MSG_CLOUD_H3 41
/** Cloud amount (N2) */
#define DBA_MSG_CLOUD_N2 42
/** Cloud amount (C2) */
#define DBA_MSG_CLOUD_C2 43
/** Height of base of cloud (H2) */
#define DBA_MSG_CLOUD_H2 44
/** Cloud amount (N1) */
#define DBA_MSG_CLOUD_N1 45
/** Cloud amount (C1) */
#define DBA_MSG_CLOUD_C1 46
/** Height of base of cloud (H1) */
#define DBA_MSG_CLOUD_H1 47
/** Cloud type (CH) */
#define DBA_MSG_CLOUD_CH 48
/** Cloud type (CM) */
#define DBA_MSG_CLOUD_CM 49
/** Cloud type (CL) */
#define DBA_MSG_CLOUD_CL 50
/** Cloud amount (NH) */
#define DBA_MSG_CLOUD_NH 51
/** Height of base of cloud (HH) */
#define DBA_MSG_CLOUD_HH 52
/** Wind direction at 10 metres above ground */
#define DBA_MSG_WIND_DIR 53
/** Wind speed at 10 metres above ground */
#define DBA_MSG_WIND_SPEED 54
/** Extreme counterclockwise wind direction of a variable wind at 10 metres above ground */
#define DBA_MSG_EX_CCW_WIND 55
/** Extreme clockwise wind direction of a variable wind at 10 metres above ground */
#define DBA_MSG_EX_CW_WIND 56
/** Maximum wind gust speed at 10 metres above ground */
#define DBA_MSG_WIND_GUST_MAX_SPEED 57
/** Maximum wind gust direction at 10 metres above ground */
#define DBA_MSG_WIND_GUST_MAX_DIR 58
/** Altimeter setting (QNH) */
#define DBA_MSG_QNH 59
/** Temperature at 2 metres above ground */
#define DBA_MSG_TEMP_2M 60
/** Wet bulb temperature at 2 metres above ground */
#define DBA_MSG_WET_TEMP_2M 61
/** Dew point at 2 metres above ground */
#define DBA_MSG_DEWPOINT_2M 62
/** Humidity at 2 metres above ground */
#define DBA_MSG_HUMIDITY 63
/** Pressure reduced to mean sea level */
#define DBA_MSG_PRESS_MSL 64
/** Direction of motion of moving observing platform */
#define DBA_MSG_ST_DIR 65
/** Speed of motion of moving observing platform */
#define DBA_MSG_ST_SPEED 66
/** Type of measuring equipment used */
#define DBA_MSG_MEAS_EQUIP_TYPE 67
/** Radiosonde type */
#define DBA_MSG_SONDE_TYPE 68
/** Radiosonde computational method */
#define DBA_MSG_SONDE_METHOD 69
/** Solar and infrared radiation correction */
#define DBA_MSG_SONDE_CORRECTION 70
/** Tracking technique/status of system used */
#define DBA_MSG_SONDE_TRACKING 71
/** Pressure at ground level */
#define DBA_MSG_PRESS 72
/** Anemometer height */
#define DBA_MSG_HEIGHT_ANEM 73
/** Total show depth */
#define DBA_MSG_TOT_SNOW 74
/** Visibility */
#define DBA_MSG_VISIBILITY 75
/** Present weather */
#define DBA_MSG_PRES_WTR 76
/** General weather indicator (TAF/METAR) */
#define DBA_MSG_METAR_WTR 77
/** State of ground (with or without snow) */
#define DBA_MSG_STATE_GROUND 78
/** Sea/water surface temperature */
#define DBA_MSG_WATER_TEMP 79
/** Past weather (1 - 6h) */
#define DBA_MSG_PAST_WTR1_6H 80
/** Past weather (2 - 6h) */
#define DBA_MSG_PAST_WTR2_6H 81
/** Characteristic of pressure tendency */
#define DBA_MSG_PRESS_TEND 82
/** Past weather (1 - 3h) */
#define DBA_MSG_PAST_WTR1_3H 83
/** Past weather (2 - 3h) */
#define DBA_MSG_PAST_WTR2_3H 84
/** 24 hour pressure change at ground level */
#define DBA_MSG_PRESS_24H 85
/** 3 hour pressure change at ground level */
#define DBA_MSG_PRESS_3H 86
/** Total precipitation in the last 24 hours */
#define DBA_MSG_TOT_PREC24 87
/** Total precipitation in the last 12 hours */
#define DBA_MSG_TOT_PREC12 88
/** Total precipitation in the last 6 hours */
#define DBA_MSG_TOT_PREC6 89
/** Total precipitation in the last 3 hours */
#define DBA_MSG_TOT_PREC3 90
/** Total precipitation in the last hour */
#define DBA_MSG_TOT_PREC1 91
/** Number of items in dba_msg_vartable */
#define DBA_MSG_VARTABLE_SIZE 91

}

#endif

