//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      Img3D/Model/Geometry.cpp
//! @brief     Implements class Geometry.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "Img3D/Model/Geometry.h"
#include "Base/Util/Assert.h"
#include "Img3D/Model/GeometryStore.h"

namespace Img3D {

//------------------------------------------------------------------------------

void Geometry::Vertices::addVertex(const F3& v, int n)
{
    for (int i = 0; i < n; ++i)
        append(v);
}

void Geometry::Vertices::addTriangle(const F3& v1, const F3& v2, const F3& v3)
{
    append(v1);
    append(v2);
    append(v3);
}

void Geometry::Vertices::addQuad(const F3& v1, const F3& v2, const F3& v3, const F3& v4)
{
    addTriangle(v1, v2, v3);
    addTriangle(v3, v4, v1);
}

void Geometry::Vertices::addQuad(const Vertices& vs, unsigned i1, unsigned i2, unsigned i3,
                                 unsigned i4)
{
    addQuad(vs.at(i1), vs.at(i2), vs.at(i3), vs.at(i4));
}

void Geometry::Vertices::addFan(const Vertices& vs, const Indices& is)
{
    ASSERT(is.size() >= 3); // at least one triangle
    const F3& ctr = vs.at(is.at(0));
    for (size_t i = 0; i + 2 < is.size(); ++i)
        addTriangle(ctr, vs.at(is.at(1 + i)), vs.at(is.at(2 + i)));
}

//------------------------------------------------------------------------------

Geometry::Geometry(GeometricID::Key key_, const double2d_t* top, const double2d_t* bottom,
                   bool drawBottom)
    : m_key(key_)
{
    m_mesh = meshRoughBox(top, bottom, drawBottom);
}

Geometry::Geometry(GeometricID::Key key_)
    : m_key(key_)
{
    using namespace GeometricID;

    switch (m_key.id) {
    case BaseShape::Plane:
        m_mesh = meshPlane();
        break;
    case BaseShape::Box:
        m_mesh = meshBox();
        break;
    case BaseShape::Sphere:
        m_mesh = meshSphere(m_key.p1, m_key.p2, m_key.p3);
        break;
    case BaseShape::Column:
        m_mesh = meshColumn(m_key.p1, m_key.p2);
        break;
    case BaseShape::Icosahedron:
        m_mesh = meshIcosahedron();
        break;
    case BaseShape::Dodecahedron:
        m_mesh = meshDodecahedron();
        break;
    case BaseShape::TruncatedBox:
        m_mesh = meshTruncBox(m_key.p1);
        break;
    case BaseShape::Bipyramid4:
        m_mesh = meshBipyramid4(m_key.p1, m_key.p2, m_key.p3);
        break;
    case BaseShape::Ripple:
        m_mesh = meshRipple(m_key.p1, m_key.p2);
        break;
    }
}

Geometry::~Geometry()
{
    // remove self from the store
    geometryStore().geometryDeleted(*this, m_key);
}

Geometry::Mesh Geometry::makeMesh(const Vertices& vs, Vertices const* ns)
{
    int nv = vs.count();
    ASSERT(0 == nv % 3);
    ASSERT(!ns || nv == ns->count()); // if normals not given, will be computed

    Mesh mesh(nv);

    for (int i = 0; i < nv; i += 3) {
        const F3& v0 = vs.at(0 + i);
        const F3& v1 = vs.at(1 + i);
        const F3& v2 = vs.at(2 + i);
        const F3* n0;
        const F3* n1;
        const F3* n2;
        F3 nm;

        if (ns) {
            n0 = &ns->at(0 + i);
            n1 = &ns->at(1 + i);
            n2 = &ns->at(2 + i);
        } else {
            nm = F3::crossProduct((v1 - v0), (v2 - v0));
            n0 = n1 = n2 = &nm;
        }

        mesh.append(VertexAndNormal(v0, *n0));
        mesh.append(VertexAndNormal(v1, *n1));
        mesh.append(VertexAndNormal(v2, *n2));
    }

    return mesh;
}

Geometry::Mesh Geometry::makeMesh(const Vertices& vs, const Vertices& ns)
{
    return makeMesh(vs, &ns);
}

} // namespace Img3D
