"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerDataConnectionsRoute = registerDataConnectionsRoute;

var _configSchema = require("@osd/config-schema");

var _shared = require("../../../common/constants/shared");

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
function registerDataConnectionsRoute(router) {
  router.get({
    path: `${_shared.DATACONNECTIONS_BASE}/{name}`,
    validate: {
      params: _configSchema.schema.object({
        name: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    try {
      const dataConnectionsresponse = await context.observability_plugin.observabilityClient.asScoped(request).callAsCurrentUser('ppl.getDataConnectionById', {
        dataconnection: request.params.name
      });
      return response.ok({
        body: dataConnectionsresponse
      });
    } catch (error) {
      console.error('Issue in fetching data connection:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.delete({
    path: `${_shared.DATACONNECTIONS_BASE}/{name}`,
    validate: {
      params: _configSchema.schema.object({
        name: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    try {
      const dataConnectionsresponse = await context.observability_plugin.observabilityClient.asScoped(request).callAsCurrentUser('ppl.deleteDataConnection', {
        dataconnection: request.params.name
      });
      return response.ok({
        body: dataConnectionsresponse
      });
    } catch (error) {
      console.error('Issue in deleting data connection:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.put({
    path: `${_shared.DATACONNECTIONS_BASE}`,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        connector: _configSchema.schema.string(),
        allowedRoles: _configSchema.schema.arrayOf(_configSchema.schema.string()),
        properties: _configSchema.schema.any()
      })
    }
  }, async (context, request, response) => {
    try {
      const dataConnectionsresponse = await context.observability_plugin.observabilityClient.asScoped(request).callAsCurrentUser('ppl.modifyDataConnection', {
        body: {
          name: request.body.name,
          connector: request.body.connector,
          allowedRoles: request.body.allowedRoles,
          properties: request.body.properties
        }
      });
      return response.ok({
        body: dataConnectionsresponse
      });
    } catch (error) {
      console.error('Issue in modifying data connection:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: `${_shared.DATACONNECTIONS_BASE}`,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        connector: _configSchema.schema.string(),
        allowedRoles: _configSchema.schema.arrayOf(_configSchema.schema.string()),
        properties: _configSchema.schema.any()
      })
    }
  }, async (context, request, response) => {
    try {
      const dataConnectionsresponse = await context.observability_plugin.observabilityClient.asScoped(request).callAsCurrentUser('ppl.createDataSource', {
        body: {
          name: request.body.name,
          connector: request.body.connector,
          allowedRoles: request.body.allowedRoles,
          properties: request.body.properties
        }
      });
      return response.ok({
        body: dataConnectionsresponse
      });
    } catch (error) {
      console.error('Issue in creating data source:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.response
      });
    }
  });
  router.get({
    path: `${_shared.DATACONNECTIONS_BASE}`,
    validate: false
  }, async (context, request, response) => {
    try {
      const dataConnectionsresponse = await context.observability_plugin.observabilityClient.asScoped(request).callAsCurrentUser('ppl.getDataConnections');
      return response.ok({
        body: dataConnectionsresponse
      });
    } catch (error) {
      console.error('Issue in fetching data sources:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.response
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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