"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ZeppelinBackend = void 0;

var _wreck_requests = require("../../common/helpers/notebooks/wreck_requests");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; } /*
                                                                                                                                                                                                                   * Copyright OpenSearch Contributors
                                                                                                                                                                                                                   * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                   */

class ZeppelinBackend {
  constructor() {
    _defineProperty(this, "backend", 'Zeppelin Backend');

    _defineProperty(this, "viewNotes", async function (client, wreckOptions) {
      try {
        let output = [];
        const body = await (0, _wreck_requests.requestor)('GET', 'api/notebook/', wreckOptions);
        output = JSON.parse(body.toString()).body;
        return output;
      } catch (error) {
        throw new Error('View Notebooks Error:' + error);
      }
    });

    _defineProperty(this, "fetchNote", async function (client, noteId, wreckOptions) {
      try {
        const body = await (0, _wreck_requests.requestor)('GET', 'api/notebook/' + noteId, wreckOptions);
        return JSON.parse(body.toString()).body.paragraphs;
      } catch (error) {
        throw new Error('Fetching Notebook Error:' + error);
      }
    });

    _defineProperty(this, "addNote", async function (client, params, wreckOptions) {
      wreckOptions.payload = params;

      try {
        const body = await (0, _wreck_requests.requestor)('POST', 'api/notebook/', wreckOptions);
        return JSON.parse(body.toString());
      } catch (error) {
        throw new Error('Creating New Notebook Error:' + error);
      }
    });

    _defineProperty(this, "renameNote", async function (client, params, wreckOptions) {
      wreckOptions.payload = {
        name: params.name
      };

      try {
        const body = await (0, _wreck_requests.requestor)('PUT', 'api/notebook/' + params.noteId + '/rename/', wreckOptions);
        return JSON.parse(body.toString());
      } catch (error) {
        throw new Error('Renaming Notebook Error:' + error);
      }
    });

    _defineProperty(this, "cloneNote", async function (client, params, wreckOptions) {
      wreckOptions.payload = {
        name: params.name
      };

      try {
        const body = await (0, _wreck_requests.requestor)('POST', 'api/notebook/' + params.noteId, wreckOptions);
        return JSON.parse(body.toString());
      } catch (error) {
        throw new Error('Cloning Notebook Error:' + error);
      }
    });

    _defineProperty(this, "deleteNote", async function (client, noteId, wreckOptions) {
      try {
        const body = await (0, _wreck_requests.requestor)('DELETE', 'api/notebook/' + noteId, wreckOptions);
        return JSON.parse(body.toString());
      } catch (error) {
        throw new Error('Deleting Notebook Error:' + error);
      }
    });

    _defineProperty(this, "exportNote", async function (client, noteId, wreckOptions) {
      try {
        const body = await (0, _wreck_requests.requestor)('GET', 'api/notebook/export/' + noteId, wreckOptions);
        return JSON.parse(body.toString());
      } catch (error) {
        throw new Error('Export Notebook Error:' + error);
      }
    });

    _defineProperty(this, "importNote", async function (client, noteObj, wreckOptions) {
      wreckOptions.payload = noteObj;

      try {
        const body = await (0, _wreck_requests.requestor)('POST', 'api/notebook/import', wreckOptions);
        const respBody = JSON.parse(body.toString());
        return respBody;
      } catch (error) {
        throw new Error('Import Notebook Error:' + error);
      }
    });

    _defineProperty(this, "addParagraph", async function (wreckOptions, params) {
      const visualizationPrefix = '%sh #vizobject:';
      const observabilityVisualizationPrefix = '%sh #observabilityviz:';
      let paragraphText = params.paragraphInput;

      if (params.inputType === 'VISUALIZATION' && params.paragraphInput.substring(0, 15) !== visualizationPrefix) {
        paragraphText = visualizationPrefix + paragraphText;
      }

      if (params.inputType === 'OBSERVABILITY_VISUALIZATION' && params.paragraphInput.substring(0, 22) !== observabilityVisualizationPrefix) {
        paragraphText = visualizationPrefix + paragraphText;
      }

      if (params.paragraphInput === '') {
        paragraphText = '%md\n';
      }

      wreckOptions.payload = {
        title: params.inputType,
        text: paragraphText,
        index: params.paragraphIndex
      };

      try {
        const body = await (0, _wreck_requests.requestor)('POST', 'api/notebook/' + params.noteId + '/paragraph', wreckOptions);
        const respBody = JSON.parse(body.toString());
        return respBody;
      } catch (error) {
        throw new Error('Adding Paragraph Error:' + error);
      }
    });

    _defineProperty(this, "updateParagraph", async function (wreckOptions, params) {
      wreckOptions.payload = {
        text: params.paragraphInput
      };
      if (params.paragraphType !== undefined) wreckOptions.payload.title = params.paragraphType;

      try {
        const body = await (0, _wreck_requests.requestor)('PUT', 'api/notebook/' + params.noteId + '/paragraph/' + params.paragraphId, wreckOptions);
        return JSON.parse(body.toString());
      } catch (error) {
        throw new Error('Updating Paragraph Error:' + error);
      }
    });

    _defineProperty(this, "runPara", async function (wreckOptions, params) {
      wreckOptions.payload = {};

      try {
        const body = await (0, _wreck_requests.requestor)('POST', 'api/notebook/run/' + params.noteId + '/' + params.paragraphId, wreckOptions);
        return JSON.parse(body.toString()).status;
      } catch (error) {
        throw new Error('Running Paragraph Error:' + error);
      }
    });

    _defineProperty(this, "fetchParagraph", async function (wreckOptions, params) {
      try {
        const body = await (0, _wreck_requests.requestor)('GET', 'api/notebook/' + params.noteId + '/paragraph/' + params.paragraphId, wreckOptions);
        return JSON.parse(body.toString()).body;
      } catch (error) {
        throw new Error('Fetching Paragraph Error:' + error);
      }
    });

    _defineProperty(this, "deleteParagraph", async function (wreckOptions, params) {
      wreckOptions.payload = {};

      try {
        const body = await (0, _wreck_requests.requestor)('DELETE', 'api/notebook/' + params.noteId + '/paragraph/' + params.paragraphId, wreckOptions);
        return JSON.parse(body.toString()).status;
      } catch (error) {
        throw new Error('Deleting Paragraph Error:' + error);
      }
    });

    _defineProperty(this, "clearAllParagraphs", async function (wreckOptions, noteid) {
      try {
        const body = await (0, _wreck_requests.requestor)('PUT', 'api/notebook/' + noteid + '/clear', wreckOptions);
        return JSON.parse(body.toString()).status;
      } catch (error) {
        throw new Error('Clearing Paragraph Error:' + error);
      }
    });

    _defineProperty(this, "updateRunFetchParagraph", async function (client, request, wreckOptions) {
      const params = request.params;

      try {
        const updateInfo = await this.updateParagraph(wreckOptions, params);
        const runInfo = await this.runPara(wreckOptions, params);
        const getInfo = await this.fetchParagraph(wreckOptions, params);
        return getInfo;
      } catch (error) {
        throw new Error('Update/Run Paragraph Error:' + error);
      }
    });

    _defineProperty(this, "updateFetchParagraph", async function (client, params, wreckOptions) {
      try {
        const updateInfo = await this.updateParagraph(wreckOptions, params);
        const getInfo = await this.fetchParagraph(wreckOptions, params);
        return getInfo;
      } catch (error) {
        throw new Error('Save Paragraph Error:' + error);
      }
    });

    _defineProperty(this, "addFetchNewParagraph", async function (client, params, wreckOptions) {
      try {
        const respBody = await this.addParagraph(wreckOptions, params);
        const payload = { ...params,
          paragraphId: respBody.body
        };
        const getinfo = await this.fetchParagraph(wreckOptions, payload);
        return getinfo;
      } catch (error) {
        throw new Error('add/Fetch Paragraph Error:' + error);
      }
    });

    _defineProperty(this, "deleteFetchParagraphs", async function (client, params, wreckOptions) {
      try {
        const delinfo = await this.deleteParagraph(wreckOptions, params);
        const notebookinfo = await this.fetchNote(client, params.noteId, wreckOptions);
        return {
          paragraphs: notebookinfo
        };
      } catch (error) {
        throw new Error('Delete Paragraph Error:' + error);
      }
    });

    _defineProperty(this, "clearAllFetchParagraphs", async function (client, params, wreckOptions) {
      try {
        const clearinfo = await this.clearAllParagraphs(wreckOptions, params.noteId);
        const notebookinfo = await this.fetchNote(client, params.noteId, wreckOptions);
        return {
          paragraphs: notebookinfo
        };
      } catch (error) {
        throw new Error('Clear Paragraph Error:' + error);
      }
    });
  } // Gets all the notebooks available from Zeppelin Server
  // ZS Endpoint => http://[zeppelin-server]:[zeppelin-port]/api/notebook

  /* Fetches a notebook by id from Zeppelin Server
   * Param: noteId -> Id of notebook to be fetched
   * ZS Endpoint => http://[zeppelin-server]:[zeppelin-port]/api/notebook/[noteId]
   */

  /* Add a notebook to the Zeppelin Server
   * Param: name -> name of new notebook
   * ZS Endpoint => http://[zeppelin-server]:[zeppelin-port]/api/notebook
   */

  /* Rename a notebook in Zeppelin Server
   * Params: name -> new name for the notebook to be renamed
   *         noteId -> Id of notebook to be fetched
   * ZS Endpoint => http://[zeppelin-server]:[zeppelin-port]/api/notebook/[noteId]/rename
   */

  /* Clone a notebook in Zeppelin Server
   * Params: name -> new name for the cloned notebook
   *         noteId -> Id for the notebook to be cloned
   * ZS Endpoint => http://[zeppelin-server]:[zeppelin-port]/api/notebook/[noteId]
   */

  /* Delete a notebook in Zeppelin Server
   * Param: noteId -> Id for the notebook to be deleted
   * ZS Endpoint => http://[zeppelin-server]:[zeppelin-port]/api/notebook
   */

  /* Export a notebook from Zeppelin Server
   * Param: noteId -> Id for the notebook to be exported
   * ZS Endpoint => http://[zeppelin-server]:[zeppelin-port]/api/notebook/export/{noteid}
   */

  /* Import a notebook in Zeppelin Server
   * Params: noteObj -> note Object to be imported
   * ZS Endpoint => http://[zeppelin-server]:[zeppelin-port]/api/notebook/import
   */

  /* Add a paragraph in notebook
   * Params: noteId -> Id for the notebook
   *         paragraphIndex -> index(position) to add a new paragraph
   *         paragraphInput -> paragraph input code
   * ZS Endpoint => http://[zeppelin-server]:[zeppelin-port]/api/notebook/[noteId]/paragraph
   */

  /* Update a Paragraph in notebook
   * Params: noteId -> Id of the notebook
   *         paragraphId -> Id of the paragraph to be updated
   *         paragraphInput -> paragraph input code
   * ZS Endpoint => http://[zeppelin-server]:[zeppelin-port]/api/notebook/[noteId]/paragraph/[paragraphId]
   */

  /* Run a Paragraph in notebook
   * Params: noteId -> Id of the notebook
   *         paragraphId -> Id of the paragraph to be run
   * ZS Endpoint => http://[zeppelin-server]:[zeppelin-port]/api/notebook/run/[noteId]/[paragraphId]
   */

  /* Fetch a Paragraph from notebook
   * Params: noteId -> Id of the notebook
   *         paragraphId -> Id of the paragraph to be fetched
   * ZS Endpoint => http://[zeppelin-server]:[zeppelin-port]/api/notebook/[noteId]/paragraph/[paragraphId]
   */

  /* Delete a Paragraph in notebook
   * Params: noteId -> Id of the notebook
   *         paragraphId -> Id of the paragraph to be deleted
   * ZS Endpoint => http://[zeppelin-server]:[zeppelin-port]/api/notebook/[noteId]/paragraph/[paragraphId]
   */

  /* Clear all the paragraphs in the notebook
   * Param: noteId -> Id of notebook to be cleared
   * ZS Endpoint => http://[zeppelin-server]:[zeppelin-port]/api/notebook/[noteId]/clear
   */

  /* --> Updates a Paragraph with input content
   * --> Runs it
   * --> Fetches the updated Paragraph (with new input content and output result)
   * Params: noteId -> Id of the notebook
   *         paragraphId -> Id of the paragraph to be updated
   *         paragraphInput -> paragraph input code
   */

  /* --> Updates a Paragraph with input content
   * --> Fetches the updated Paragraph (with new input content)
   * Params: noteId -> Id of the notebook
   *         paragraphId -> Id of the paragraph to be updated
   *         paragraphInput -> paragraph input code
   */

  /* --> Adds a Paragraph with input content
   * --> Fetches the Paragraph
   * Params: noteId -> Id of the notebook
   *         paragraphId -> Id of the paragraph to be fetched
   */

  /* --> Deletes a Paragraph with id
   * --> Fetches the all other Paragraphs as a list
   * Params: noteId -> Id of the notebook
   *         paragraphId -> Id of the paragraph to be deleted
   */

  /* --> Clears output for all the paragraphs
   * --> Fetches the all Paragraphs as a list (with cleared outputs)
   * Param: noteId -> Id of notebook to be cleared
   */


}

exports.ZeppelinBackend = ZeppelinBackend;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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