<?php

declare(strict_types=1);

/*
 * Copyright (c) 2023 François Kooman <fkooman@tuxed.net>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

namespace fkooman\Radius;

use fkooman\Radius\Exception\SocketException;
use RangeException;

class PhpSocket implements SocketInterface
{
    /** @var ?resource */
    private $socketStream = null;

    public function open(string $serverAddress, int $timeOutSec = 3): void
    {
        if (false === $socketStream = stream_socket_client($serverAddress, $errNo, $errStr, $timeOutSec)) {
            throw new SocketException(sprintf('(%d) %s', $errNo, $errStr));
        }
        stream_set_timeout($socketStream, $timeOutSec);
        $this->socketStream = $socketStream;
    }

    public function close(): void
    {
        if (null === $this->socketStream) {
            throw new SocketException('socket not open');
        }
        fclose($this->socketStream);
    }

    public function read(int $noOfBytesToRead): string
    {
        if ($noOfBytesToRead < 0) {
            throw new RangeException('must read >=0 bytes');
        }
        if (null === $this->socketStream) {
            throw new SocketException('socket not open');
        }
        if (false === $bytesRead = fread($this->socketStream, $noOfBytesToRead)) {
            throw new SocketException('unable to read');
        }
        if ($noOfBytesToRead !== strlen($bytesRead)) {
            throw new SocketException(sprintf('unable to read "%d" bytes', $noOfBytesToRead));
        }

        return $bytesRead;
    }

    public function write(string $bytesToWrite): void
    {
        if (null === $this->socketStream) {
            throw new SocketException('socket not open');
        }
        if (false === fwrite($this->socketStream, $bytesToWrite)) {
            throw new SocketException('unable to write');
        }
    }
}
