"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.IntegrationReader = void 0;

var _path = _interopRequireDefault(require("path"));

var _validators = require("../validators");

var _fs_data_adaptor = require("./fs_data_adaptor");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; } /*
                                                                                                                                                                                                                   * Copyright OpenSearch Contributors
                                                                                                                                                                                                                   * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                   */

/**
 * The Integration class represents the data for Integration Templates.
 * It is backed by the repository file system.
 * It includes accessor methods for integration configs, as well as helpers for nested components.
 */
class IntegrationReader {
  constructor(directory, reader) {
    _defineProperty(this, "reader", void 0);

    _defineProperty(this, "directory", void 0);

    _defineProperty(this, "name", void 0);

    this.directory = directory;
    this.name = _path.default.basename(directory);
    this.reader = reader !== null && reader !== void 0 ? reader : new _fs_data_adaptor.FileSystemCatalogDataAdaptor(directory);
  }
  /**
   * Like getConfig(), but thoroughly checks all nested integration dependencies for validity.
   *
   * @returns a Result indicating whether the integration is valid.
   */


  async deepCheck() {
    const configResult = await this.getConfig();

    if (!configResult.ok) {
      return configResult;
    }

    try {
      const schemas = await this.getSchemas();

      if (!schemas.ok || Object.keys(schemas.value.mappings).length === 0) {
        return {
          ok: false,
          error: new Error('The integration has no schemas available')
        };
      }

      const assets = await this.getAssets();

      if (!assets.ok || Object.keys(assets).length === 0) {
        return {
          ok: false,
          error: new Error('An integration must have at least one asset')
        };
      }
    } catch (err) {
      return {
        ok: false,
        error: err
      };
    }

    return configResult;
  }
  /**
   * Get the latest version of the integration available.
   * This method relies on the fact that integration configs have their versions in their name.
   * Any files that don't match the config naming convention will be ignored.
   *
   * @returns A string with the latest version, or null if no versions are available.
   */


  async getLatestVersion() {
    const versions = await this.reader.findIntegrationVersions();

    if (!versions.ok) {
      console.error(versions.error);
      return null;
    }

    return versions.value.length > 0 ? versions.value[0] : null;
  }
  /**
   * Get the configuration of the current integration.
   *
   * @param version The version of the config to retrieve.
   * @returns The config if a valid config matching the version is present, otherwise null.
   */


  async getConfig(version) {
    if ((await this.reader.getDirectoryType()) !== 'integration') {
      return {
        ok: false,
        error: new Error(`${this.directory} is not a valid integration`)
      };
    }

    const maybeVersion = version ? version : await this.getLatestVersion();

    if (maybeVersion === null) {
      return {
        ok: false,
        error: new Error(`No valid config matching version ${version} is available`)
      };
    }

    const configFile = `${this.name}-${maybeVersion}.json`;
    const config = await this.reader.readFile(configFile);

    if (!config.ok) {
      return config;
    }

    return (0, _validators.validateTemplate)(config.value);
  }
  /**
   * Retrieve assets associated with the integration.
   * This method greedily retrieves all assets.
   * If the version is invalid, an error is thrown.
   * If an asset is invalid, it will be skipped.
   *
   * @param version The version of the integration to retrieve assets for.
   * @returns An object containing the different types of assets.
   */


  async getAssets(version) {
    const configResult = await this.getConfig(version);

    if (!configResult.ok) {
      return configResult;
    }

    const config = configResult.value;
    const resultValue = {};

    if (config.assets.savedObjects) {
      const sobjPath = `${config.assets.savedObjects.name}-${config.assets.savedObjects.version}.ndjson`;
      const assets = await this.reader.readFile(sobjPath, 'assets');

      if (!assets.ok) {
        return assets;
      }

      resultValue.savedObjects = assets.value;
    }

    if (config.assets.queries) {
      resultValue.queries = [];
      const queries = await Promise.all(config.assets.queries.map(async item => {
        const queryPath = `${item.name}-${item.version}.${item.language}`;
        const query = await this.reader.readFileRaw(queryPath, 'assets');

        if (!query.ok) {
          return query;
        }

        return {
          ok: true,
          value: {
            language: item.language,
            query: query.value.toString('utf8')
          }
        };
      }));

      for (const query of queries) {
        if (!query.ok) {
          return query;
        }

        resultValue.queries.push(query.value);
      }
    }

    return {
      ok: true,
      value: resultValue
    };
  }
  /**
   * Retrieve sample data associated with the integration.
   * If the version is invalid, an error is thrown.
   * If the sample data is invalid, null will be returned
   *
   * @param version The version of the integration to retrieve assets for.
   * @returns An object containing a list of sample data with adjusted timestamps.
   */


  async getSampleData(version) {
    const configResult = await this.getConfig(version);

    if (!configResult.ok) {
      return configResult;
    }

    const config = configResult.value;
    const resultValue = {
      sampleData: null
    };

    if (config.sampleData) {
      const jsonContent = await this.reader.readFile(config.sampleData.path, 'data');

      if (!jsonContent.ok) {
        return jsonContent;
      }

      for (const value of jsonContent.value) {
        if (!('@timestamp' in value)) {
          continue;
        } // Randomly scatter timestamps across last 10 minutes
        // Assume for now that the ordering of events isn't important, can change to a sequence if needed
        // Also doesn't handle fields like `observedTimestamp` if present


        const newTime = new Date(Date.now() - Math.floor(Math.random() * 1000 * 60 * 10)).toISOString();
        Object.assign(value, {
          '@timestamp': newTime
        });

        if ('observedTimestamp' in value) {
          Object.assign(value, {
            observedTimestamp: newTime
          });
        }
      }

      resultValue.sampleData = jsonContent.value;
    }

    return {
      ok: true,
      value: resultValue
    };
  }
  /**
   * Retrieve schema data associated with the integration.
   * This method greedily retrieves all mappings and schemas.
   * It's assumed that a valid version will be provided.
   * If the version is invalid, an error is thrown.
   * If a schema is invalid, an error will be thrown.
   *
   * @param version The version of the integration to retrieve assets for.
   * @returns An object containing the different types of assets.
   */


  async getSchemas(version) {
    const configResult = await this.getConfig(version);

    if (!configResult.ok) {
      return configResult;
    }

    const config = configResult.value;
    const resultValue = {
      mappings: {}
    };

    for (const component of config.components) {
      const schemaFile = `${component.name}-${component.version}.mapping.json`;
      const schema = await this.reader.readFile(schemaFile, 'schemas');

      if (!schema.ok) {
        return schema;
      }

      resultValue.mappings[component.name] = schema.value;
    }

    return {
      ok: true,
      value: resultValue
    };
  }
  /**
   * Retrieves the data for a static file associated with the integration.
   *
   * @param staticPath The path of the static to retrieve.
   * @returns A buffer with the static's data if present, otherwise null.
   */


  async getStatic(staticPath) {
    return await this.reader.readFileRaw(staticPath, 'static');
  }

}

exports.IntegrationReader = IntegrationReader;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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