/*
 * Copyright (c) 2015 Google, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.google.common.truth;

import static com.google.common.truth.ExpectFailure.expectFailure;
import static com.google.common.truth.FailureAssertions.assertFailureKeys;
import static com.google.common.truth.FailureAssertions.assertFailureValue;
import static com.google.common.truth.Truth.assertThat;
import static java.math.BigDecimal.TEN;

import java.math.BigDecimal;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.JUnit4;

/** Tests for {@link BigDecimalSubject}. */
@RunWith(JUnit4.class)
public class BigDecimalSubjectTest {
  @SuppressWarnings("TruthSelfEquals")
  @Test
  public void isEqualTo() {
    // make sure this still works
    assertThat(TEN).isEqualTo(TEN);
  }

  @Test
  public void isEquivalentAccordingToCompareTo() {
    // make sure this still works
    assertThat(TEN).isEquivalentAccordingToCompareTo(TEN);
  }

  @Test
  public void isEqualToIgnoringScale_bigDecimal() {
    assertThat(TEN).isEqualToIgnoringScale(TEN);
    assertThat(TEN).isEqualToIgnoringScale(new BigDecimal(10));
    AssertionError e =
        expectFailure(
            whenTesting -> whenTesting.that(TEN).isEqualToIgnoringScale(new BigDecimal(3)));
    assertFailureKeys(e, "expected", "but was", "(scale is ignored)");
    assertFailureValue(e, "expected", "3");
    assertFailureValue(e, "but was", "10");
  }

  @Test
  public void isEqualToIgnoringScale_int() {
    assertThat(TEN).isEqualToIgnoringScale(10);
    AssertionError e =
        expectFailure(whenTesting -> whenTesting.that(TEN).isEqualToIgnoringScale(3));
    assertFailureKeys(e, "expected", "but was", "(scale is ignored)");
    assertFailureValue(e, "expected", "3");
    assertFailureValue(e, "but was", "10");
  }

  @Test
  public void isEqualToIgnoringScale_long() {
    assertThat(TEN).isEqualToIgnoringScale(10L);
    AssertionError e =
        expectFailure(whenTesting -> whenTesting.that(TEN).isEqualToIgnoringScale(3L));
    assertFailureKeys(e, "expected", "but was", "(scale is ignored)");
    assertFailureValue(e, "expected", "3");
    assertFailureValue(e, "but was", "10");
  }

  @Test
  public void isEqualToIgnoringScale_string() {
    assertThat(TEN).isEqualToIgnoringScale("10");
    assertThat(TEN).isEqualToIgnoringScale("10.");
    assertThat(TEN).isEqualToIgnoringScale("10.0");
    assertThat(TEN).isEqualToIgnoringScale("10.00");
    AssertionError e =
        expectFailure(whenTesting -> whenTesting.that(TEN).isEqualToIgnoringScale("3"));
    assertFailureKeys(e, "expected", "but was", "(scale is ignored)");
    assertFailureValue(e, "expected", "3");
    assertFailureValue(e, "but was", "10");
  }

  @Test
  public void isEqualToIgnoringScale_stringWithDecimals() {
    BigDecimal tenFour = new BigDecimal("10.4");
    assertThat(tenFour).isEqualToIgnoringScale("10.4");
    assertThat(tenFour).isEqualToIgnoringScale("10.4");
    assertThat(tenFour).isEqualToIgnoringScale("10.40");
    assertThat(tenFour).isEqualToIgnoringScale("10.400");
    AssertionError e =
        expectFailure(whenTesting -> whenTesting.that(tenFour).isEqualToIgnoringScale("3.4"));
    assertFailureKeys(e, "expected", "but was", "(scale is ignored)");
    assertFailureValue(e, "expected", "3.4");
    assertFailureValue(e, "but was", "10.4");
  }

  @Test
  public void isEqualToIgnoringScale_nullActual() {
    AssertionError e =
        expectFailure(
            whenTesting -> whenTesting.that((BigDecimal) null).isEqualToIgnoringScale("3.4"));
    assertFailureKeys(e, "expected", "but was");
    assertFailureValue(e, "expected", "3.4");
    assertFailureValue(e, "but was", "null");
  }

  @Test
  public void isEqualToIgnoringScale_nullExpected() {
    AssertionError e =
        expectFailure(
            whenTesting ->
                whenTesting.that(new BigDecimal("3.4")).isEqualToIgnoringScale((BigDecimal) null));
    assertFailureKeys(e, "expected", "but was");
    assertFailureValue(e, "expected", "null");
    assertFailureValue(e, "but was", "3.4");
  }

  @Test
  public void isEqualToIgnoringScale_bothNull() {
    assertThat((BigDecimal) null).isEqualToIgnoringScale((BigDecimal) null);
  }
}
