searchData={"items":[{"type":"module","title":"asn1ct","doc":"ASN.1 compiler and compile-time support functions\n\nThe ASN.1 compiler takes an ASN.1 module as input and generates a corresponding\nErlang module, which can encode and decode the specified data types.\nAlternatively, the compiler takes a specification module specifying all input\nmodules, and generates a module with encode/decode functions. In addition, some\ngeneric functions can be used during development of applications that handles\nASN.1 data (encoded as `BER` or `PER`).\n\n> #### Note {: .info }\n>\n> By default in Erlang/OTP 17, the representation of the `BIT STRING` and\n> `OCTET STRING` types as Erlang terms were changed. `BIT STRING` values are now\n> Erlang bit strings and `OCTET STRING` values are binaries. Also, an undecoded\n> open type is now wrapped in an `asn1_OPENTYPE` tuple. For details, see\n> [BIT STRING](asn1_getting_started.md#bit-string),\n> [OCTET STRING](asn1_getting_started.md#octet-string), and\n> [ASN.1 Information Objects](asn1_getting_started.md#Information-Object) in the\n> User's Guide.\n>\n> To revert to the old representation of the types, use option\n> `legacy_erlang_types`.","ref":"asn1ct.html"},{"type":"function","title":"asn1ct.compile/1","doc":"Equivalent to  compile(Asn1Module, []) .","ref":"asn1ct.html#compile/1"},{"type":"function","title":"asn1ct.compile/2","doc":"Compiles the ASN.1 module `Asn1Module` and generates an Erlang module\n`Asn1Module.erl` with encode and decode functions for all types defined in\nthe ASN.1 module.\n\nFor each ASN.1 value defined in the module, an Erlang function that\nreturns the value in Erlang representation is generated.\n\nIf `Asn1Module` is a filename without extension, first `\".asn1\"` is assumed,\nthen `\".asn\"`, and finally `\".py\"` (to be compatible with the old ASN.1\ncompiler). `Asn1Module` can be a full pathname (relative or absolute) including\nfilename with (or without) extension.[](){: #asn1set }\n\nIf it is needed to compile a set of `ASN.1` modules into an Erlang\nfile with encode/decode functions, list all involved files in a\nconfiguration file, one line per file. This configuration file must\nhave a double extension `\".set.asn1\"` (`\".asn1\"` can alternatively be\n`\".asn\"` or `\".py\"`). If the input files are `File1.asn1`,\n`File2.asn1`, and `File3.asn1`, the configuration file should look as\nfollows:\n\n```text\nFile1.asn1\nFile2.asn1\nFile3.asn1\n```\n\nThe output files in this case get their names from the configuration file. If\nthe configuration file is named `SetOfFiles.set.asn1`, the names of the output\nfiles are `SetOfFiles.hrl, SetOfFiles.erl, and SetOfFiles.asn1db`.\n\nSometimes in a system of `ASN.1` modules, different modules can have\ndifferent default tag modes, for example, one uses `AUTOMATIC` and\nanother `IMPLICIT`.  The multi-file compilation resolves the default\ntagging as if the modules were compiled separately.\n\nName collisions is an unwanted effect that can occur in multi-file\ncompilation. The compiler solves this problem in one of two ways:\n\n- If the definitions are identical, the output module keeps only one definition\n  with the original name.\n- If the definitions have the same name and differs in the definition, they are\n  renamed. The new names are the definition name and the original module name\n  concatenated.\n\nIf a name collision occurs, the compiler reports a `\"NOTICE: ...\"` message that\ntells if a definition was renamed, and the new name that must be used to\nencode/decode data.\n\n`Options` is a list with options specific for the ASN.1 compiler and options\nthat are applied to the Erlang compiler. The ASN.1 compiler passes on any\nunrecognized options to the Erlang compiler. The available options are as follows:\n\n- **`ber | per | uper | jer`** - The encoding rule to be used. The\n  supported encoding rules are Basic Encoding Rules (`ber`), Packed\n  Encoding Rules (`per`) aligned, PER unaligned (`uper`), and JSON\n  Encoding Rules (`jer`). The `jer` option can be used by itself to\n  generate a module that only supports encoding/decoding of JER, or it\n  can be used as a supplementary option to `ber`, `per`, and `uper`,\n  in which case a module that handles both the main encoding rules and\n  JER will be generated. In that case, the exported functions for JER\n  will be `jer_encode(Type, Value)` and `jer_decode(Type, Bytes)`.\n\n  JER (ITU-T X.697) are experimental in OTP 22. There is support for a\n  subset of the X.697 standard, for example there is no support for:\n\n  - JER encoding instructions\n  - the REAL type\n\n  > #### Change {: .info }\n  >\n  > In Erlang/OTP 27 and later, module `m:json` in STDLIB is used for\n  > encoding and decoding JSON. Before Erlang/OTP 27, it was necessary\n  > to provide an external JSON library.\n\n  If the encoding rule option is omitted, `ber` is the default.\n\n  The generated Erlang module always gets the same name as the ASN.1 module.\n  Therefore, only one encoding rule per ASN.1 module can be used at runtime.\n\n- **`der`** - With this option the Distinguished Encoding Rules (`der`) is\n  chosen. DER is regarded as a specialized variant of the BER encoding rule.\n  Therefore, this option only makes sense together with option `ber`. This\n  option sometimes adds sorting and value checks when encoding, which implies\n  slower encoding. The decoding routines are the same as for `ber`.\n\n- **`maps`** - This option changes the representation of the types `SEQUENCE`\n  and `SET` to use maps (instead of records). This option also suppresses the\n  generation of `.hrl` files.\n\n  For details, see section\n  [Map representation for SEQUENCE and SET](asn1_getting_started.md#MAP_SEQ_SET)\n  in the User's Guide.\n\n- **`compact_bit_string`** - The `BIT STRING` type is decoded to \"compact\n  notation\".\n\n  **This option is not recommended for new code.**\n\n  For details, see section [BIT STRING](asn1_getting_started.md#bit-string) in\n  the User's Guide.\n\n  This option implies option `legacy_erlang_types`, and it cannot be combined\n  with option `maps`.\n\n- **`legacy_bit_string`** - The `BIT STRING` type is decoded to the legacy\n  format, that is, a list of zeroes and ones.\n\n  **This option is not recommended for new code.**\n\n  For details, see section [BIT STRING](asn1_getting_started.md#bit-string) in\n  the User's Guide\n\n  This option implies option `legacy_erlang_types`, and it cannot be combined\n  with option `maps`.\n\n- **`legacy_erlang_types`** - Use the same Erlang types to represent\n  `BIT STRING` and `OCTET STRING` as in Erlang/OTP R16.\n\n  **This option is not recommended for new code.**\n\n  For details, see section [BIT STRING](asn1_getting_started.md#bit-string) and\n  section [OCTET STRING](asn1_getting_started.md#octet-string) in the User's\n  Guide.\n\n  This option cannot be combined with option `maps`.\n\n- **`{n2n, EnumTypeName}`** - Tells the compiler to generate functions for\n  conversion between names (as atoms) and numbers and conversely for the\n  specified `EnumTypeName`. There can be multiple occurrences of this option to\n  specify several type names. The type names must be declared as `ENUMERATIONS`\n  in the ASN.1 specification.\n\n  If `EnumTypeName` does not exist in the ASN.1 specification, the compilation\n  stops with an error code.\n\n  The generated conversion functions are named `name2num_EnumTypeName/1` and\n  `num2name_EnumTypeName/1`.\n\n- **`noobj`** - Do not compile (that is, do not produce object code) the\n  generated `.erl` file. If this option is omitted, the generated Erlang module\n  is compiled.\n\n- **`{i, IncludeDir}`** - Adds `IncludeDir` to the search-path for `.asn1db` and\n  ASN.1 source files. The compiler tries to open an `.asn1db` file when a\n  module imports definitions from another ASN.1 module. If no `.asn1db` file\n  is found, the ASN.1 source file is parsed. Several `{i, IncludeDir}` can be\n  given.\n\n- **`{outdir, Dir}`** - Specifies directory `Dir` where all generated files are\n  to be placed. If this option is omitted, the files are placed in the current\n  directory.\n\n- **`asn1config`** - When using one of the specialized decodes, exclusive or\n  selective decode, instructions must be given in a configuration file. Option\n  `asn1config` enables specialized decodes and takes the configuration file in\n  concern. The configuration file has the same name as the ASN.1 specification,\n  but with extension `.asn1config`.\n\n  For instructions for exclusive decode, see section\n  [Exclusive Decode](asn1_spec.md#Exclusive-Instruction) in the User's Guide.\n\n  For instructions for selective decode, see section\n  [Selective Decode](asn1_spec.md#Selective-Instruction) in the User's Guide.\n\n- **`undec_rest`** - By default when decoding, any bytes following the\n  end of an ASN.1 data structure are discarded. If an ASN.1 module is\n  compiled with option `undec_rest`, the decode function returns a\n  tuple `{ok, Value, Rest}`, where `Rest` is the bytes following the ASN.1\n  data structure. `Rest` can be a list or a binary.\n\n- **`no_ok_wrapper`** - With this option, the generated `encode/2` and\n  `decode/2` functions do not wrap a successful return value in an `{ok,...}`\n  tuple. If any error occurs, an exception will be raised.\n\n- **`{macro_name_prefix, Prefix}`** - All macro names generated by the compiler\n  are prefixed with `Prefix`. This is useful when multiple protocols that\n  contain macros with identical names are included in a single module.\n\n- **`{record_name_prefix, Prefix}`** - All record names generated by the\n  compiler are prefixed with `Prefix`. This is useful when multiple protocols\n  that contain records with identical names are included in a single module.\n\n- **`verbose`** - Causes more verbose information from the compiler describing\n  what it is doing.\n\n- **`warnings_as_errors`** - Causes warnings to be treated as errors.\n\n- **`deterministic`** - Causes all non-deterministic options to be stripped from\n  the `-asn1_info()` attribute.\n\nUnrecognized options are passed on to the Erlang compiler when the generated\n`.erl` file is compiled.\n\nThe compiler generates the following files:\n\n- `Asn1Module.hrl` (if any `SET` or `SEQUENCE` is defined)\n- `Asn1Module.erl` \\- Erlang module with encode, decode, and value functions\n- `Asn1Module.asn1db` \\- Intermediate format used by the compiler when modules\n  `IMPORT` definitions from each other.","ref":"asn1ct.html#compile/2"},{"type":"function","title":"asn1ct.test/1","doc":"Tests encoding and decoding of all types in `Module`.\n\nFor more details, see `test/3`.","ref":"asn1ct.html#test/1"},{"type":"function","title":"asn1ct.test/2","doc":"Tests encoding and decoding of `Module`.\n\nIf the second argument is given as atom `Type`, that type is tested.\n\nIf the second argument is given as list `Options`, that are the options\nthat are used for testing all types in the module.\n\nFor more details, see `test/3`.","ref":"asn1ct.html#test/2"},{"type":"function","title":"asn1ct.test/3","doc":"Performs a test of encode and decode of types in `Module`.\n\nThe generated functions are called by this function. This function is\nuseful for testing to ensure that the generated encode and decode\nfunctions as well as the general runtime support work as expected.\n\n> #### Note {: .info }\n>\n> Currently, the `test` functions have many limitations. Essentially, they will\n> mostly work for old specifications based on the 1997 standard for ASN.1, but\n> not for most modern-style applications. Another limitation is that the `test`\n> functions may not work if options that change code generations strategies such\n> as the options `macro_name_prefix` and `record_name_prefix` have been used.\n\n- [`test/1`](`test/1`) iterates over all types in `Module`.\n- [`test/2`](`test/2`) tests type `Type` with a random value.\n- [`test/3`](`test/3`) tests type `Type` with `Value`.\n\nSchematically, the following occurs for each type in the module:\n\n```erlang\n{ok, Value} = asn1ct:value(Module, Type),\n{ok, Bytes} = Module:encode(Type, Value),\n{ok, Value} = Module:decode(Type, Bytes).\n```\n\nThe `test` functions use the `*.asn1db` files for all included modules. If they\nare located in a different directory than the current working directory, use the\n`include` option to add paths. This is only needed when automatically generating\nvalues. For static values using `Value` no options are needed.","ref":"asn1ct.html#test/3"},{"type":"function","title":"asn1ct.value/2","doc":"Returns an Erlang term that is an example of a valid Erlang representation of a\nvalue of the ASN.1 type `Type`.\n\nThe value is a random value and subsequent calls to this function will\nfor most types return different values.\n\n> #### Note {: .info }\n>\n> Currently, the `value` function has many limitations. Essentially, it will\n> mostly work for old specifications based on the 1997 standard for ASN.1, but\n> not for most modern-style applications. Another limitation is that the `value`\n> function may not work if options that change code generations strategies such\n> as the options `macro_name_prefix` and `record_name_prefix` have been used.","ref":"asn1ct.html#value/2"},{"type":"extras","title":"asn1 Release Notes","doc":"<!--\n%CopyrightBegin%\n\nSPDX-License-Identifier: Apache-2.0\n\nCopyright Ericsson AB 2023-2025. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# asn1 Release Notes\n\nThis document describes the changes made to the asn1 application.","ref":"notes.html"},{"type":"extras","title":"Asn1 5.4.2 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-4-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- Decoding a constrained BIT STRING using JER was broken.\n\n  Own Id: OTP-19681 Aux Id: [PR-9949]\n\n- NIFs and linked-in drivers are now loadable when running in an Erlang source tree on Windows.\n\n  Own Id: OTP-19686 Aux Id: [PR-9969]\n\n[PR-9949]: https://github.com/erlang/otp/pull/9949\n[PR-9969]: https://github.com/erlang/otp/pull/9969","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"Asn1 5.4.1 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-4-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- The ASN.1 compiler could generate code that would cause Dialyzer with the `unmatched_returns` option to emit warnings.\n\n  Own Id: OTP-19638 Aux Id: [GH-9841], [PR-9846]\n\n[GH-9841]: https://github.com/erlang/otp/issues/9841\n[PR-9846]: https://github.com/erlang/otp/pull/9846","ref":"notes.html#fixed-bugs-and-malfunctions-1"},{"type":"extras","title":"Asn1 5.4 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-4"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- The `undec_rest` option would be ignored in generated functions for exclusive decode. The option is now respected, meaning that the return value from such functions are now three-tuples instead of a two-tuples.\n\n  *** POTENTIAL INCOMPATIBILITY ***\n\n  Own Id: OTP-19290 Aux Id: [PR-8798]\n\n[PR-8798]: https://github.com/erlang/otp/pull/8798","ref":"notes.html#fixed-bugs-and-malfunctions-2"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- The license and copyright header has changed format to include an `SPDX-License-Identifier`. At the same time, most files have been updated to follow a uniform standard for license headers.\n\n  Own Id: OTP-19575 Aux Id: [PR-9670]\n\n- The ancient ASN.1 modules used in `public_key` has been replaced with more modern versions, but we have strived to keep the documented Erlang API for the `public_key` application compatible.\n\n  *** POTENTIAL INCOMPATIBILITY ***\n\n  Own Id: OTP-19612 Aux Id: [PR-9774]\n\n[PR-9670]: https://github.com/erlang/otp/pull/9670\n[PR-9774]: https://github.com/erlang/otp/pull/9774","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"Asn1 5.3.4.2 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-3-4-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- Decoding a constrained BIT STRING using JER was broken.\n\n  Own Id: OTP-19681 Aux Id: [PR-9949]\n\n[PR-9949]: https://github.com/erlang/otp/pull/9949","ref":"notes.html#fixed-bugs-and-malfunctions-3"},{"type":"extras","title":"Asn1 5.3.4.1 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-3-4-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- The ASN.1 compiler could generate code that would cause Dialyzer with the `unmatched_returns` option to emit warnings.\n\n  Own Id: OTP-19638 Aux Id: [GH-9841], [PR-9846]\n\n[GH-9841]: https://github.com/erlang/otp/issues/9841\n[PR-9846]: https://github.com/erlang/otp/pull/9846","ref":"notes.html#fixed-bugs-and-malfunctions-4"},{"type":"extras","title":"Asn1 5.3.4 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-3-4"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- Negative REAL numbers greater than -1 would be incorrectly encoded (the minus sign would be lost).\n\n  Own Id: OTP-19567 Aux Id: ERIERL-1214, [PR-9658]\n\n[PR-9658]: https://github.com/erlang/otp/pull/9658","ref":"notes.html#fixed-bugs-and-malfunctions-5"},{"type":"extras","title":"Asn1 5.3.3 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-3-3"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- The JER backend will now include the SIZE constraint in the type info for OCTET STRINGs, and a SIZE constraint with a range will now be included for BIT STRINGs. This does not change the actual encoding or decoding of JER, but can be useful for tools.\n\n  Own Id: OTP-19542 Aux Id: ERIERL-1204, [PR-9588]\n\n[PR-9588]: https://github.com/erlang/otp/pull/9588","ref":"notes.html#fixed-bugs-and-malfunctions-6"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- When using the JSON encoding rules, it is now possible to call the decode/2 function in the following way with data that has already been decoded by json:decode/1:\n  \n  ```\n  SomeModule:decode(Type, {json_decoded, Decoded}).\n  ```\n\n  Own Id: OTP-19547 Aux Id: ERIERL-1206, [PR-9611]\n\n[PR-9611]: https://github.com/erlang/otp/pull/9611","ref":"notes.html#improvements-and-new-features-1"},{"type":"extras","title":"Asn1 5.3.2 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-3-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- Multiple bugs in decoding of the `REAL` type has been eliminated. Also, the documentation for `REAL` has been updated to mention the special values `0`, `PLUS-INFINITY`, and `MINUS-INFINITY`.\n\n  Own Id: OTP-19504 Aux Id: [GH-9096], [PR-9469]\n\n[GH-9096]: https://github.com/erlang/otp/issues/9096\n[PR-9469]: https://github.com/erlang/otp/pull/9469","ref":"notes.html#fixed-bugs-and-malfunctions-7"},{"type":"extras","title":"Asn1 5.3.1 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-3-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- Fixed a cosmetic but harmless issue with the ASN.1 compiler passing on the `undec_rest` option to the Erlang compiler.\n\n  Own Id: OTP-19218 Aux Id: [GH-8779], [PR-8781]\n\n[GH-8779]: https://github.com/erlang/otp/issues/8779\n[PR-8781]: https://github.com/erlang/otp/pull/8781","ref":"notes.html#fixed-bugs-and-malfunctions-8"},{"type":"extras","title":"Asn1 5.3 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-3"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- Multiple bugs has been eliminated in the [specialized decode feature](`e:asn1:asn1_spec.md`).\n\n  Own Id: OTP-18813 Aux Id: [PR-7790]\n\n[PR-7790]: https://github.com/erlang/otp/pull/7790","ref":"notes.html#fixed-bugs-and-malfunctions-9"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- Specs have been added to all `asn1ct` API functions.\n\n  Own Id: OTP-18804 Aux Id: [PR-7738]\n\n- The documentation has been migrated to use Markdown and ExDoc.\n\n  Own Id: OTP-18955 Aux Id: [PR-8026]\n\n- The `jer` (JSON Encoding Rules) for ASN.1 now use the new `m:json` module for encoding and decoding JSON. Thus, there is no longer any need for an external JSON library.\n\n  Own Id: OTP-19018 Aux Id: [PR-8241]\n\n[PR-7738]: https://github.com/erlang/otp/pull/7738\n[PR-8026]: https://github.com/erlang/otp/pull/8026\n[PR-8241]: https://github.com/erlang/otp/pull/8241","ref":"notes.html#improvements-and-new-features-2"},{"type":"extras","title":"Asn1 5.2.2.1 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-2-2-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"* The ASN.1 compiler could generate code that would cause Dialyzer with the `unmatched_returns` option to emit warnings.\n\n  Own Id: OTP-19638 Aux Id: GH-9841, PR-9846","ref":"notes.html#fixed-bugs-and-malfunctions-10"},{"type":"extras","title":"Asn1 5.2.2 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-2-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"* An ASN.1 module that contains named `BIT STRING` values would fail to compiled if both the BER and JER back-ends were enabled.\n\n  Own Id: OTP-19039 Aux Id: GH-8291, PR-8297","ref":"notes.html#fixed-bugs-and-malfunctions-11"},{"type":"extras","title":"Asn1 5.2.1 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-2-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- Fix benign warning from gcc 11 about mismatching call to free().\n\n  Own Id: OTP-18844","ref":"notes.html#fixed-bugs-and-malfunctions-12"},{"type":"extras","title":"Asn1 5.2 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- The ASN.1 compiler would ignore a constraint such as `(SIZE (1..4), ...)`,\n  causing incorrect behavior of the encoding and decoding function for the PER\n  and UPER backends. Corrected to handle the constraint in the same way as\n  `(SIZE (1..4, ...))`.\n\n  Own Id: OTP-18729 Aux Id: PR-7575","ref":"notes.html#fixed-bugs-and-malfunctions-13"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- The JER backend has been internally refactored in a way that is compatible for\n  applications that use the documented API. However, for a group of ASN.1\n  modules that depend on each other (for example, `S1AP-PDU-Descriptions`,\n  S1AP-Contents, and so on), all modules in the group must be recompiled if on\n  of the group members is recompiled.\n\n  Own Id: OTP-18748 Aux Id: ERIERL-957, PR-7637","ref":"notes.html#improvements-and-new-features-3"},{"type":"extras","title":"Asn1 5.1 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- The ASN.1 compiler used to reject correctly specified RELATIVE-OID values\n  containing other RELATIVE-OID values. This is now corrected.\n\n  Own Id: OTP-18534 Aux Id: ERIERL-737, PR-7039","ref":"notes.html#fixed-bugs-and-malfunctions-14"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- Minor code improvements.\n\n  Own Id: OTP-18441\n\n- Handling of `on_load` modules during boot has been improved by adding an extra\n  step in the boot order for embedded mode that runs all `on_load` handlers,\n  instead of relying on explicit invocation of them, later, when the kernel\n  supervision tree starts.\n\n  This is mostly a code improvement and OTP internal simplification to avoid\n  future bugs and to simplify code maintenance.\n\n  Own Id: OTP-18447","ref":"notes.html#improvements-and-new-features-4"},{"type":"extras","title":"Asn1 5.0.21.1 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-0-21-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- Fix benign warning from gcc 11 about mismatching call to free().\n\n  Own Id: OTP-18844","ref":"notes.html#fixed-bugs-and-malfunctions-15"},{"type":"extras","title":"Asn1 5.0.21 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-0-21"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- For the `per` and `uper` ASN.1 encoding rules, encoding and decoding the\n  `SEQUENCE OF` and `SET OF` constructs with 16384 items or more is now\n  supported.\n\n  Own Id: OTP-18245 Aux Id: ERIERL-859","ref":"notes.html#fixed-bugs-and-malfunctions-16"},{"type":"extras","title":"Asn1 5.0.20 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-0-20"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- There is a new configure option, `--enable-deterministic-build`, which will\n  apply the `deterministic` compiler option when building Erlang/OTP. The\n  `deterministic` option has been improved to eliminate more sources of\n  non-determinism in several applications.\n\n  Own Id: OTP-18165 Aux Id: PR-5965","ref":"notes.html#improvements-and-new-features-5"},{"type":"extras","title":"Asn1 5.0.19 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-0-19"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- The atom `maybe` has been quoted in the source code.\n\n  Own Id: OTP-17980","ref":"notes.html#fixed-bugs-and-malfunctions-17"},{"type":"extras","title":"Asn1 5.0.18.2 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-0-18-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- Fix benign warning from gcc 11 about mismatching call to free().\n\n  Own Id: OTP-18844","ref":"notes.html#fixed-bugs-and-malfunctions-18"},{"type":"extras","title":"Asn1 5.0.18.1 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-0-18-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- For the `per` and `uper` ASN.1 encoding rules, encoding and decoding the\n  `SEQUENCE OF` and `SET OF` constructs with 16384 items or more is now\n  supported.\n\n  Own Id: OTP-18245 Aux Id: ERIERL-859","ref":"notes.html#fixed-bugs-and-malfunctions-19"},{"type":"extras","title":"Asn1 5.0.18 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-0-18"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- Add support for the `maps` option in combination with the `jer` backend.\n\n  Own Id: OTP-17959 Aux Id: GH-5757","ref":"notes.html#fixed-bugs-and-malfunctions-20"},{"type":"extras","title":"Asn1 5.0.17 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-0-17"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- A parameterized type with a SEQUENCE with extension (\"...\") made the compiler\n  backend to crash. The previous fix for this in GH-4514 was not complete.\n\n  Own Id: OTP-17522 Aux Id: GH-4902","ref":"notes.html#fixed-bugs-and-malfunctions-21"},{"type":"extras","title":"Asn1 5.0.16 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-0-16"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- Fixed a bug in the `asn1` compiler that potentially could cause it to fail to\n  open a file.\n\n  Own Id: OTP-17387 Aux Id: OTP-17123","ref":"notes.html#fixed-bugs-and-malfunctions-22"},{"type":"extras","title":"Asn1 5.0.15.1 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-0-15-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- A parameterized type with a SEQUENCE with extension (\"...\") made the compiler\n  backend to crash. The previous fix for this in GH-4514 was not complete.\n\n  Own Id: OTP-17522 Aux Id: GH-4902","ref":"notes.html#fixed-bugs-and-malfunctions-23"},{"type":"extras","title":"Asn1 5.0.15 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-0-15"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- A parameterized type with a SEQUENCE with extension (\"...\") made the compiler\n  backend to crash.\n\n  Own Id: OTP-17227 Aux Id: GH-4514\n\n- For JER encoding rules an INTEGER value outside the declared range is now\n  reported as error during decode.\n\n  Own Id: OTP-17306 Aux Id: ERIERL-506","ref":"notes.html#fixed-bugs-and-malfunctions-24"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- For the JER encoding rules, the declared order of the fields in a SEQUENCE is\n  now maintained in the resulting JSON object. Previously a map was used which\n  caused an undefined order of the fields which was not friendly for debugging.\n\n  Own Id: OTP-17297 Aux Id: ERIERL-607","ref":"notes.html#improvements-and-new-features-6"},{"type":"extras","title":"Asn1 5.0.14 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-0-14"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- Changes in order to build on the Haiku operating system.\n\n  Thanks to Calvin Buckley\n\n  Own Id: OTP-16707 Aux Id: PR-2638","ref":"notes.html#improvements-and-new-features-7"},{"type":"extras","title":"Asn1 5.0.13 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-0-13"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- Adhere to the ASN.1 specification for hstring & bstring lexical items. That is\n  they may include white space.\n\n  Own Id: OTP-16490","ref":"notes.html#fixed-bugs-and-malfunctions-25"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- Refactored the internal handling of deprecated and removed functions.\n\n  Own Id: OTP-16469\n\n- Improve handling of ellipsis in a CHOICE\n\n  Own Id: OTP-16554 Aux Id: ERL-1189","ref":"notes.html#improvements-and-new-features-8"},{"type":"extras","title":"Asn1 5.0.12 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-0-12"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- Dialyzer warnings of type `no_match` are now suppressed in the generated\n  files.\n\n  Own Id: OTP-16636 Aux Id: ERIERL-145","ref":"notes.html#improvements-and-new-features-9"},{"type":"extras","title":"Asn1 5.0.11 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-0-11"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- The compiler now has limited support for the JSON encoding rules (ITU-T X.697\n  ASN.1 encoding rules: Specification of JavaScript Object Notation Encoding\n  Rules).\n\n  Own Id: OTP-16030","ref":"notes.html#improvements-and-new-features-10"},{"type":"extras","title":"Asn1 5.0.10 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-0-10"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- Fix 'DEFAULT' with 'OCTET STRING' and 'SEQUENCE OF CHOICE' with extensions.\n\n  Own Id: OTP-16542 Aux Id: PR-2159","ref":"notes.html#improvements-and-new-features-11"},{"type":"extras","title":"Asn1 5.0.9 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-0-9"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- All incorrect (that is, all) uses of \"can not\" has been corrected to \"cannot\"\n  in source code comments, documentation, examples, and so on.\n\n  Own Id: OTP-14282 Aux Id: PR-1891\n\n- Corrected problems with the following value definitions:\n\n  - value of SEQUENCE OF CHOICE with extensions\n  - value of CHOICE with extensions\n  - DEFAULT used with OCTET STRING\n\n  Own Id: OTP-15697 Aux Id: PR-2159","ref":"notes.html#fixed-bugs-and-malfunctions-26"},{"type":"extras","title":"Asn1 5.0.8 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-0-8"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- Handle erroneous length during decode (BER only) without crashing.\n\n  Own Id: OTP-15470 Aux Id: ERIERL-278","ref":"notes.html#fixed-bugs-and-malfunctions-27"},{"type":"extras","title":"Asn1 5.0.7 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-0-7"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- A bug in ASN.1 BER decoding has been fixed. When decoding a recursively\n  enclosed term the length was not propagated to that term decoding, so if the\n  length of the enclosed term was longer than the enclosing that error was not\n  detected.\n\n  A hard coded C stack limitation for decoding recursive ASN.1 terms has been\n  introduced. This is currently set to 8 kWords giving a nesting depth of about\n  1000 levels. Deeper terms can not be decoded, which should not be much of a\n  real world limitation.\n\n  Own Id: OTP-14440 Aux Id: ERIERL-220","ref":"notes.html#fixed-bugs-and-malfunctions-28"},{"type":"extras","title":"Asn1 5.0.6 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-0-6"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- Update to use the new string api instead of the old.\n\n  Own Id: OTP-15036","ref":"notes.html#improvements-and-new-features-12"},{"type":"extras","title":"Asn1 5.0.5.2 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-0-5-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- Handle erroneous length during decode (BER only) without crashing.\n\n  Own Id: OTP-15470 Aux Id: ERIERL-278","ref":"notes.html#fixed-bugs-and-malfunctions-29"},{"type":"extras","title":"Asn1 5.0.5.1 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-0-5-1"},{"type":"extras","title":"Known Bugs and Problems - asn1 Release Notes","doc":"- A bug in ASN.1 BER decoding has been fixed. When decoding a recursively\n  enclosed term the length was not propagated to that term decoding, so if the\n  length of the enclosed term was longer than the enclosing that error was not\n  detected\n\n  A hard coded C stack limitation for decoding recursive ASN.1 terms has been\n  introduced. This is currently set to 8 kWords giving a nesting depth of about\n  1000 levels. Deeper terms can not be decoded, which should not be much of a\n  real world limitation.\n\n  Own Id: OTP-14440 Aux Id: ERIERL-220","ref":"notes.html#known-bugs-and-problems"},{"type":"extras","title":"Asn1 5.0.5 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-0-5"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- Dialyzer suppression has been added for the generated ASN.1 helper function\n  to_bitstring/1 that previously created irrelevant warnings.\n\n  Own Id: OTP-13882 Aux Id: ERIERL-144","ref":"notes.html#fixed-bugs-and-malfunctions-30"},{"type":"extras","title":"Asn1 5.0.4 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-0-4"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- There was a issue with BER encoding and the `undec_rest` option in generated\n  decoders. An exception could be thrown instead of returning an error tuple.\n\n  Own Id: OTP-14786 Aux Id: ERL-518\n\n- The asn1ct:test functions crashed on decoders generated with options\n  `no_ok_wrapper`, `undec_rest`.\n\n  Own Id: OTP-14787 Aux Id: ERL-518","ref":"notes.html#fixed-bugs-and-malfunctions-31"},{"type":"extras","title":"Asn1 5.0.3 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-0-3"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- Compiling an ASN.1 module using the option \\{n2n, EnumTypeName\\} when\n  EnumTypeName contains a hyphen like for example Cause-Misc caused syntax\n  errors when compiling the generated Erlang code. This is now corrected.\n\n  Own Id: OTP-14495 Aux Id: ERL-437","ref":"notes.html#fixed-bugs-and-malfunctions-32"},{"type":"extras","title":"Asn1 5.0.2 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-0-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- Default values now work in extension for PER, so if you give the atom\n  `asn1_DEFAULT` instead of a value it will become the default value.\n\n  Own Id: OTP-13011 Aux Id: ERIERL-60","ref":"notes.html#fixed-bugs-and-malfunctions-33"},{"type":"extras","title":"Asn1 5.0.1 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-0-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- Fixed compilation error of generated code caused by a missing quotation of\n  function names as part of an external call for encoding.\n\n  Own Id: OTP-14519 Aux Id: ERIERL-49","ref":"notes.html#fixed-bugs-and-malfunctions-34"},{"type":"extras","title":"Asn1 5.0 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-5-0"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- Add compile option `-compile(no_native)` in modules with `on_load` directive\n  which is not yet supported by HiPE.\n\n  Own Id: OTP-14316 Aux Id: PR-1390","ref":"notes.html#fixed-bugs-and-malfunctions-35"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- The `error` tuple returned from the `encode` and `decode` functions will now\n  include the stack backtrace to make it easier to understand what went wrong.\n\n  Own Id: OTP-13961\n\n- The deprecated module `asn1rt` has been removed. The deprecated functions\n  `asn1ct:encode/3` and `asn1ct:decode/3` have been removed. The undocumented\n  function `asn1ct:encode/2` has been removed.\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-14146\n\n- The new '`maps`' option changes the representation of the types `SEQUENCE` and\n  `SET` to be maps (instead of records).\n\n  Own Id: OTP-14219","ref":"notes.html#improvements-and-new-features-13"},{"type":"extras","title":"Asn1 4.0.4 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-4-0-4"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- Compiling multiple ASN.1 modules in the same directory with parallel make\n  (make -j) should now be safe.\n\n  Own Id: OTP-13624","ref":"notes.html#fixed-bugs-and-malfunctions-36"},{"type":"extras","title":"Asn1 4.0.3 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-4-0-3"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- Internal changes\n\n  Own Id: OTP-13551","ref":"notes.html#improvements-and-new-features-14"},{"type":"extras","title":"Asn1 4.0.2 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-4-0-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- When compiling to the PER format, the ASN.1 compiler would crash when\n  attempting to compile an ASN.1 module with a constrained INTEGER with more\n  than 65536 values and named values. (Thanks to Ingars for reporting this bug.)\n\n  Own Id: OTP-13257\n\n- The ASN.1 compiler will now emit Dialyzer suppressions for improper lists.\n  Thus, there is no longer any need to use `--Wno_improper_lists` when analyzing\n  modules generated by the ASN.1 compiler.\n\n  Own Id: OTP-13324","ref":"notes.html#fixed-bugs-and-malfunctions-37"},{"type":"extras","title":"Asn1 4.0.1 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-4-0-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- Trying to encode an empty named BIT STRING in BER would fail with a\n  `function_clause` exception. (Thanks to Svilen Ivanov for reporting this bug.)\n\n  Own Id: OTP-13149","ref":"notes.html#fixed-bugs-and-malfunctions-38"},{"type":"extras","title":"Asn1 4.0 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-4-0"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- Many bugs have been eliminated in the ASN.1 compiler so that it can now\n  successfully compile many more ASN.1 specifications. Error messages have also\n  been improved.\n\n  Own Id: OTP-12395","ref":"notes.html#fixed-bugs-and-malfunctions-39"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- The documentation for `asn1ct:test/1,2,3` and `asn1ct:value/2` has been\n  updated with information about the limitations of the functions.\n\n  Own Id: OTP-12765 Aux Id: seq12866, seq12867","ref":"notes.html#improvements-and-new-features-15"},{"type":"extras","title":"Asn1 3.0.4 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-3-0-4"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- The ASN.1 compiler would crash if a SEQUENCE ended with a double set of\n  ellipses (`...`).\n\n  Own Id: OTP-12546 Aux Id: seq12815","ref":"notes.html#fixed-bugs-and-malfunctions-40"},{"type":"extras","title":"Asn1 3.0.3 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-3-0-3"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- When decoding BER, primitives with an indefinite length will be immediately\n  rejected. (Thanks to Simon Cornish for reporting this bug.)\n\n  Own Id: OTP-12205\n\n- BER: A bug with compliance to X.680 (200811) s31.2.7 has been fixed.\n  Basically, when TagDefault is AUTOMATIC then tags are IMPLICIT unless EXPLICIT\n  is given.\n\n  Own Id: OTP-12318\n\n- Usage of the `EXTERNAL` 1994 variant type was broken.\n\n  Own Id: OTP-12326","ref":"notes.html#fixed-bugs-and-malfunctions-41"},{"type":"extras","title":"Asn1 3.0.2 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-3-0-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- Several problems where the ASN.1 compiler would crash when attempting to\n  compile correct specifications have been corrected.\n\n  Own Id: OTP-12125\n\n- Robustness when decoding incorrect BER messages has been improved.\n\n  Own Id: OTP-12145","ref":"notes.html#fixed-bugs-and-malfunctions-42"},{"type":"extras","title":"Asn1 3.0.1 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-3-0-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- The ASN.1 compiler now generates code that don't trigger Dialyzer warnings.\n  Along the way, a few minor bugs were fixed.\n\n  Own Id: OTP-11372 Aux Id: seq12397","ref":"notes.html#fixed-bugs-and-malfunctions-43"},{"type":"extras","title":"Asn1 3.0 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-3-0"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- Subtyping an extensible ENUMERATED would cause an compilation error. (Thanks\n  to Morten Nygaard Åsnes for reporting this bug.)\n\n  Own Id: OTP-11700\n\n- When specifying the value for an OCTET STRING in a specification, the ASN.1\n  standard clearly states that the value must be either a bstring or an hstring,\n  but NOT a cstring. The ASN.1 compiler will now generate a compilation error if\n  the value of an OCTET STRING is given as a character string.\n\n  That is, the following example is now illegal:\n\n  `string OCTET STRING ::= \"Now illegal\"`\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-11727\n\n- Application upgrade (appup) files are corrected for the following\n  applications:\n\n  `asn1, common_test, compiler, crypto, debugger, dialyzer, edoc, eldap, erl_docgen, et, eunit, gs, hipe, inets, observer, odbc, os_mon, otp_mibs, parsetools, percept, public_key, reltool, runtime_tools, ssh, syntax_tools, test_server, tools, typer, webtool, wx, xmerl`\n\n  A new test utility for testing appup files is added to test_server. This is\n  now used by most applications in OTP.\n\n  (Thanks to Tobias Schlager)\n\n  Own Id: OTP-11744","ref":"notes.html#fixed-bugs-and-malfunctions-44"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- By giving --enable-static-\\{nifs,drivers\\} to configure it is now possible to\n  statically linking of nifs and drivers to the main Erlang VM binary. At the\n  moment only the asn1 and crypto nifs of the Erlang/OTP nifs and drivers have\n  been prepared to be statically linked. For more details see the Installation\n  Guide in the System documentation.\n\n  Own Id: OTP-11258\n\n- Code generation for the `per` and `uper` backends has been somewhat improved.\n\n  Own Id: OTP-11573\n\n- The OCTET STRING and BIT STRING types now have a more natural mapping to\n  Erlang types (binary and bitstring, respectively), which is more efficient and\n  will avoid useless conversions between lists and binaries/bitstrings.\n\n  This is an incompatible change. To revert to the old mapping to support\n  existing applications, use the `legacy_erlang_types` option.\n\n  Impact: There is a potential for better performance, as it is now possible to\n  avoid conversions between lists and binaries both in the generated ASN.1\n  encode/decode code and in the application itself.\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-11594\n\n- All functions in the `asn1rt` module, as well as `asn1ct:decode/3` and\n  `asn1ct:encode/3`, are now deprecated.\n\n  Own Id: OTP-11731\n\n- Generated .hrl files are now protected from being included more than once.\n\n  Own Id: OTP-11804","ref":"notes.html#improvements-and-new-features-16"},{"type":"extras","title":"Asn1 2.0.4 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-2-0-4"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- The default value for a `BIT STRING` would not always be recognized, causing\n  the encoding to be incorrect for the DER/PER/UPER encodings.\n\n  Own Id: OTP-11319\n\n- The ASN.1 application would fail to build if the `.erlang` file printed\n  something to standard output.\n\n  Own Id: OTP-11360\n\n- An union of integer ranges in an INTEGER constraint could sometimes be\n  interpreted as the intersection of the range.\n\n  Own Id: OTP-11411 Aux Id: seq12443\n\n- Extensible, multiple single value constraints (such as `INTEGER (1|17, ...)`)\n  would be incorrectly encoded.\n\n  Own Id: OTP-11415\n\n- The ASN.1 compiler would fail to compile a constraint with values given for\n  the extension part (such as `INTEGER (1..10, ..., 11..20)`).\n\n  Own Id: OTP-11504","ref":"notes.html#fixed-bugs-and-malfunctions-45"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- The new option '`no_ok_wrapper`' generates M:encode/2 and M:decode/2 functions\n  that don't wrap the return value in an \\{ok,...\\} tuple.\n\n  Own Id: OTP-11314","ref":"notes.html#improvements-and-new-features-17"},{"type":"extras","title":"Asn1 2.0.3 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-2-0-3"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- Open types greater than 16383 bytes will now be correctly encoded and decoded.\n\n  Own Id: OTP-11262 Aux Id: seq12386, OTP-11223","ref":"notes.html#fixed-bugs-and-malfunctions-46"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- For the PER and UPER formats, code generation especially for encoding has been\n  improved.\n\n  When encoding BIT STRINGs, values longer than the maximum size for the BIT\n  STRING type would be truncated silently - they now cause an exception.\n\n  Open types greater than 16383 bytes will now be correctly encoded and decoded.\n\n  IMPORTANT NOTE: For ASN.1 specifications that depend on each other, such as\n  the S1AP-\\* specifications, it is important to recompile all specifications\n  (compiling some with this version of the compiler and some with an older\n  version will not work).\n\n  Own Id: OTP-11300","ref":"notes.html#improvements-and-new-features-18"},{"type":"extras","title":"Asn1 2.0.2 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-2-0-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- Fix some Makefile rules that didn't support silent rules. Thanks to Anthony\n  Ramine.\n\n  Own Id: OTP-11111\n\n- PER/UPER: A semi-constrained INTEGER with a non-zero lower bound would be\n  incorrectly decoded. This bug was introduced in R16.\n\n  PER/UPER: Given `INTEGER (10..MAX, ...)`, attempting to decode any integer\n  below 10 would cause the encoder to enter an infinite loop.\n\n  PER/UPER: For a type with an extensible SIZE constraint, sizes outside of the\n  root range were incorrectly encoded.\n\n  Given a constraint such as `(SIZE (5, ...))`, encoding a size less than 5\n  would fail (PER/UPER). Similarly, for BER decoding would fail.\n\n  PER: The encoder did not align a known multiplier string (such as IA5String)\n  of length 16 bits (exactly) to an octet boundary.\n\n  In rare circumstances, DEFAULT values for the UPER backend could be wrongly\n  encoded.\n\n  Own Id: OTP-11134\n\n- UPER: The compiler would crash when compiling an ENUMERATED having more than\n  63 extended values.\n\n  PER/UPER: A SEQUENCE with more 64 extended values could not be decoded.\n\n  Own Id: OTP-11153\n\n- When decoding a SEQUENCE defined inline inside a an extension addition group,\n  the record named generated by the decoding code would not match the name in\n  the generated .hrl file.\n\n  Own Id: OTP-11154 Aux Id: seq12339","ref":"notes.html#fixed-bugs-and-malfunctions-47"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- Postscript files no longer needed for the generation of PDF files have been\n  removed.\n\n  Own Id: OTP-11016","ref":"notes.html#improvements-and-new-features-19"},{"type":"extras","title":"Asn1 2.0.1.2 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-2-0-1-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- When an object set is an actual parameter, the extension marker for the object\n  set could get lost (which would cause the decoding of unknown values to fail).\n\n  Own Id: OTP-10995 Aux Id: seq12290","ref":"notes.html#fixed-bugs-and-malfunctions-48"},{"type":"extras","title":"Asn1 2.0.1.1 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-2-0-1-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- The generated decoder for the 'per' and 'uper' backends did not correctly\n  decode ENUMERATEDs with a single value.\n\n  The generated encoder for the 'per' and 'uper' backends generated an empty\n  binary for a top-level type that did not need to be encoded (such as an\n  ENUMERATED with a single value). The correct result should be a binary\n  containing a 0 byte.\n\n  Own Id: OTP-10916 Aux Id: seq12270","ref":"notes.html#fixed-bugs-and-malfunctions-49"},{"type":"extras","title":"Asn1 2.0.1 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-2-0-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- Fixed broken table constraints within a SET OF or SEQUENCE OF for the BER\n  backend.\n\n  Own Id: OTP-10853 Aux Id: seq12245","ref":"notes.html#fixed-bugs-and-malfunctions-50"},{"type":"extras","title":"Asn1 2.0 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-2-0"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- Encoding SEQUENCEs with multiple extension addition groups with optional\n  values could fail (depending both on the specification and whether all values\n  were provided).\n\n  Own Id: OTP-10664","ref":"notes.html#fixed-bugs-and-malfunctions-51"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- The options for the ASN.1 compiler has been drastically simplified. The\n  backend is chosen by using `ber`, `per`, or `uper`. The options `optimize`,\n  `nif`, and `driver` are no longer needed. The old options will still work, but\n  will issue a warning.\n\n  Another change is that generated `encode/2` function will always return a\n  binary (some backends used to return an iolist).\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-10410 Aux Id: kunagi-254 \\[165]\n\n- The ASN.1 compiler generates faster decode functions for PER and UPER. Some\n  minor improvements have also been made for PER/UPER encoding, and to the BER\n  backend.\n\n  Own Id: OTP-10519 Aux Id: kunagi-322 \\[233]\n\n- The ASN.1 compiler will now always include necessary run-time functions in the\n  generated Erlang modules (except for `asn1rt_nif` which is still needed). If\n  the option '`inline`' is used the ASN.1 compiler will generate a warning. But\n  if '`{inline,OutputFile}`' is use, the ASN.1 compiler will refuse to compile\n  the file. (Use a `.set.asn` file if you need to remove the output file.)\n\n  The '`BIT STRING`' type will now be decoded as Erlang bitstrings by default.\n  Use the new `legacy_bit_string` option to encode as lists of ones and zeroes.\n  (The `compact_bit_string` option still works as before.)\n\n  Open types are now always returned as binaries (when there is no information\n  allowing them to be decoded).\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-10588 Aux Id: kunagi-341 \\[252]","ref":"notes.html#improvements-and-new-features-20"},{"type":"extras","title":"Asn1 1.8.1 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-1-8-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- ASN.1 decoders generated with the options `-bber_bin +optimize +nif` would\n  decode open types with a size larger than 511 incorrectly. That bug could\n  cause decoding by `public_key` to fail. The bug was in the NIF library\n  `asn1_erl_nif.so`; therefore there is no need re-compile ASN.1 specifications\n  that had the problem.\n\n  Own Id: OTP-10805 Aux Id: seq12244\n\n- Encoding SEQUENCEs with multiple extension addition groups with optional\n  values could fail (depending both on the specification and whether all values\n  were provided).\n\n  Own Id: OTP-10811 Aux Id: OTP-10664","ref":"notes.html#fixed-bugs-and-malfunctions-52"},{"type":"extras","title":"Asn1 1.8 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-1-8"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- Encoding and decoding of integer ranges can now be done with an upper bound\n  larger than the previous limit of 16^10. The new upper bound in per encoding\n  and decodings for constrained whole numbers is 2^2040 (close to 16^508)\n\n  Own Id: OTP-10128\n\n- Per encoding/decoding now works correctly for single value subtyping of an\n  integer type where a subtype is a predefined value. Previously a predefined\n  value could cause a non-valid range-check in the generated Erlang code for per\n  encoding/decoding due to a bug in the constraint checking.\n\n  Own Id: OTP-10139\n\n- Fix typo error in selected decode function (Thanks to Artem Teslenko)\n\n  Own Id: OTP-10152\n\n- Better error indication when detecting unexpected tags during decoding of BER\n  encoded data.\n\n  Own Id: OTP-10186\n\n- asn1rt_check: Fix transform_to_EXTERNAL1990 for binary input (Thanks to Harald\n  Welte)\n\n  Own Id: OTP-10233","ref":"notes.html#fixed-bugs-and-malfunctions-53"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- Add support for multiple ExtensionAdditionGroups\n\n  Own Id: OTP-10058\n\n- Add support for extensible enumeration types in n2n generated functions.\n\n  Own Id: OTP-10144","ref":"notes.html#improvements-and-new-features-21"},{"type":"extras","title":"Asn1 1.7 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-1-7"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- Some ASN.1 INTEGER type and SEQUENCE constructor variants previously not\n  handled by the ASN.1 compiler are now correctly handled\n\n  Own Id: OTP-9688\n\n- An INTEGER with a value constraint where unions are used e.g. X1 ::= INTEGER\n  (1..4 | 6 | 8 | 10 | 20) is not handled correctly. For PER the value is\n  encoded in wrong number of bits.\n\n  Own Id: OTP-9946","ref":"notes.html#improvements-and-new-features-22"},{"type":"extras","title":"Asn1 1.6.19 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-1-6-19"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- The linked-in driver used for ber decode and per encode has been replaced with\n  nifs. To enable the usage of nifs pass the nif option to erlc or\n  asn1rt:compile when compiling. If you previously used the linked-in driver,\n  you have to recompile your ASN1 modules with the current version of asn1\n  application as the linked-in driver modules have been removed.\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-9419\n\n- A few of the heavy calculations which are done for encoding and decoding\n  operations when dealing with SEQUENCE OF and DEFAULT in runtime have been\n  moved to be done in compile time instead.\n\n  Own Id: OTP-9440\n\n- When compiling an ASN.1 ber module with the +nif option, the module will use a\n  new nif for ber encoding, increasing performance by about 5%.\n\n  Own Id: OTP-9441\n\n- Tuple funs (a two-element tuple with a module name and a function) are now\n  officially deprecated and will be removed in R16. Use '`fun M:F/A`' instead.\n  To make you aware that your system uses tuple funs, the very first time a\n  tuple fun is applied, a warning will be sent to the error logger.\n\n  Own Id: OTP-9649","ref":"notes.html#improvements-and-new-features-23"},{"type":"extras","title":"Asn1 1.6.18 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-1-6-18"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- Implement or fix -Werror option\n\n  If -Werror is enabled and there are warnings no output file is written. Also\n  make sure that error/warning reporting is consistent. (Thanks to Tuncer Ayaz)\n\n  Own Id: OTP-9536","ref":"notes.html#fixed-bugs-and-malfunctions-54"},{"type":"extras","title":"Asn1 1.6.17 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-1-6-17"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- Test cases which started failing when timer:tc was changed to not catch are\n  corrected.\n\n  Own Id: OTP-9286\n\n- The bounds checking in the asn1_erl_driver when the length value of a TLV is a\n  Long Definite Length is corrected. Thanks to Vance Shipley.\n\n  Own Id: OTP-9303","ref":"notes.html#fixed-bugs-and-malfunctions-55"},{"type":"extras","title":"Asn1 1.6.16 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-1-6-16"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- asn1ct: Make formatting of errors and warnings consistent\n\n  Consistently format warning and error reports. Warning and error options from\n  erlc now also work in asnc1ct. (thanks to Tuncer Ayaz)\n\n  Own Id: OTP-9062\n\n- Shut off some dialyzer warnings\n\n  Own Id: OTP-9063","ref":"notes.html#fixed-bugs-and-malfunctions-56"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- Crash in asn1ct_check, componentrelation_leadingattr fixed. (Thanks to\n  Stephane Pamelard for finding the bug)\n\n  Own Id: OTP-9092","ref":"notes.html#improvements-and-new-features-24"},{"type":"extras","title":"Asn1 1.6.15 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-1-6-15"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- The encoding of ExtensionAdditionGroup (for PER and UPER) is corrected.\n\n  Own Id: OTP-8866 Aux Id: OTP-8797, SEQ-11557\n\n- A race condition when several processes in parallel start to do encode/decode\n  using the driver could cause an error log regarding crashing port owner\n  process. This race is now eliminated.\n\n  Own Id: OTP-8948 Aux Id: seq11733","ref":"notes.html#fixed-bugs-and-malfunctions-57"},{"type":"extras","title":"Asn1 1.6.14.1 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-1-6-14-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- Extension Addition Groups are now supported by the parser and in all backends.\n\n  Own Id: OTP-8598 Aux Id: seq-11557\n\n- Extension Addition Groups are now supported in nested types within a SEQUENCE\n  and CHOICE as well (missed that in previous fix)\n\n  Own Id: OTP-8797 Aux Id: seq-11557","ref":"notes.html#fixed-bugs-and-malfunctions-58"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- Bug in UNALIGNED PER regarding encoding and decoding of constrained numbers\n  with a valuerange > 1024. (Thanks to Vincent de Phily)\n\n  Own Id: OTP-8779\n\n- Minor corrections in the User Guide.\n\n  Own Id: OTP-8829","ref":"notes.html#improvements-and-new-features-25"},{"type":"extras","title":"Asn1 1.6.14 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-1-6-14"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- By default, the ASN.1 compiler is now silent in the absence of warnings or\n  errors. The new '`verbose`' option or the '`-v`' option for `erlc` can be\n  given to show extra information (for instance, about the files that are\n  generated). (Thanks to Tuncer Ayaz.)\n\n  Own Id: OTP-8565","ref":"notes.html#improvements-and-new-features-26"},{"type":"extras","title":"Asn1 1.6.13 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-1-6-13"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- Harmless buffer overflow by one byte in asn1 and ram_file_drv.\n\n  Own Id: OTP-8451","ref":"notes.html#fixed-bugs-and-malfunctions-59"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- Cross compilation improvements and other build system improvements.\n\n  Most notable:\n\n  - Lots of cross compilation improvements. The old cross compilation support\n    was more or less non-existing as well as broken. Please, note that the cross\n    compilation support should still be considered as experimental. Also note\n    that old cross compilation configurations cannot be used without\n    modifications. For more information on cross compiling Erlang/OTP see the\n    `$ERL_TOP/INSTALL-CROSS.md` file.\n  - Support for staged install using\n    [DESTDIR](http://www.gnu.org/prep/standards/html_node/DESTDIR.html). The old\n    broken `INSTALL_PREFIX` has also been fixed. For more information see the\n    `$ERL_TOP/INSTALL.md` file.\n  - Documentation of the `release` target of the top `Makefile`. For more\n    information see the `$ERL_TOP/INSTALL.md` file.\n  - `make install` now by default creates relative symbolic links instead of\n    absolute ones. For more information see the `$ERL_TOP/INSTALL.md` file.\n  - `$ERL_TOP/configure --help=recursive` now works and prints help for all\n    applications with `configure` scripts.\n  - Doing `make install`, or `make release` directly after `make all` no longer\n    triggers miscellaneous rebuilds.\n  - Existing bootstrap system is now used when doing `make install`, or\n    `make release` without a preceding `make all`.\n  - The `crypto` and `ssl` applications use the same runtime library path when\n    dynamically linking against `libssl.so` and `libcrypto.so`. The runtime\n    library search path has also been extended.\n  - The `configure` scripts of `erl_interface` and `odbc` now search for thread\n    libraries and thread library quirks the same way as ERTS do.\n  - The `configure` script of the `odbc` application now also looks for odbc\n    libraries in `lib64` and `lib/64` directories when building on a 64-bit\n    system.\n  - The `config.h.in` file in the `erl_interface` application is now\n    automatically generated in instead of statically updated which reduces the\n    risk of `configure` tests without any effect.\n\n  (Thanks to Henrik Riomar for suggestions and testing)\n\n  (Thanks to Winston Smith for the AVR32-Linux cross configuration and testing)\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-8323\n\n- Add support for prefixing macro names generated by the compiler\n\n  This is useful when multiple protocols that contains macros with identical\n  names are included in a single module.\n\n  Add the missing `record_name_prefix` compiler option to the documentation.\n\n  Own Id: OTP-8453\n\n- Cleanups suggested by tidier and modernization of types and specs.\n\n  Own Id: OTP-8455\n\n- Support for `EXTENSIBILITY IMPLIED` and `SET/SEQ OF NamedType` is added.\n\n  Own Id: OTP-8463","ref":"notes.html#improvements-and-new-features-27"},{"type":"extras","title":"Asn1 1.6.12 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-1-6-12"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- The documentation is now built with open source tools (xsltproc and fop) that\n  exists on most platforms. One visible change is that the frames are removed.\n\n  Own Id: OTP-8256","ref":"notes.html#improvements-and-new-features-28"},{"type":"extras","title":"Asn1 1.6.11 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-1-6-11"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- A new option `{n2n,TypeName}` can be used to enable generation of conversion\n  functions from name to number and vice versa for selected ENUMERATION types.\n  The option can be repeated many times in order to specify several types in the\n  same file.  \n  If the `TypeName` specified does not exists or is not an ENUMERATION type, the\n  compilation will be terminated with an error code.  \n  Below follows an example on how to use the option from the command line with\n  `erlc`:  \n  `erlc -bper+\"{n2n,'CauseMisc'}\" +\"{n2n,'CausePcl'}\" MyModyle.asn`\n\n  Own Id: OTP-8136 Aux Id: seq11347\n\n- Range checks added for BIT STRING with fixed SIZE constraint.\n\n  Own Id: OTP-7972 Aux Id: seq11280\n\n- Now support multiple-line comments in asn1-specs as specified in ASN1 X.680\n  (07/2002), section 11.6.4\n\n  Own Id: OTP-8043\n\n- Now parses and adds abstract syntax for PATTERN subtype constraint. No other\n  action is taken on this type of constraint.\n\n  Own Id: OTP-8046\n\n- The ASN1 subtype constraint `CONTAINING Type`,\n  `CONTAINING Type ENCODED BY Value` and `ENCODED BY Value` now is parsed.\n  Abstract syntax is added but no further action in generated code is taken.\n\n  Own Id: OTP-8047","ref":"notes.html#improvements-and-new-features-29"},{"type":"extras","title":"Asn1 1.6.10 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-1-6-10"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- A faulty receive case that catch-ed all messages in the initialization of the\n  driver has been removed, the initialization has been restructured.\n\n  Own Id: OTP-7954 Aux Id: seq11220","ref":"notes.html#fixed-bugs-and-malfunctions-60"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- The anonymous part of the decode that splits the ASN1 TLV into Tag Value\n  tuples has been optimized.\n\n  Own Id: OTP-7953","ref":"notes.html#improvements-and-new-features-30"},{"type":"extras","title":"Asn1 1.6.9 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-1-6-9"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- Error that caused crash when drivers were loaded is now corrected. Parallel\n  driver for asn1 now enabled.\n\n  Own Id: OTP-7904 Aux Id: seq11220","ref":"notes.html#fixed-bugs-and-malfunctions-61"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- Optimized code for ENUMERATION type in encoder/decoder.\n\n  Own Id: OTP-7909","ref":"notes.html#improvements-and-new-features-31"},{"type":"extras","title":"Asn1 1.6.8.1 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-1-6-8-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- Removed parallel-driver functionality due to failure when loading the driver.\n\n  Own Id: OTP-7900 Aux Id: seq11220","ref":"notes.html#fixed-bugs-and-malfunctions-62"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- Generated code now uses guards that is not obsolete, e.g.\n  [`is_integer/1`](`is_integer/1`) instead of `integer/1`.\n\n  Own Id: OTP-7910","ref":"notes.html#improvements-and-new-features-32"},{"type":"extras","title":"Asn1 1.6.8 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-1-6-8"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- A BIT STRING with a size constraint that has a single value and an extension\n  as in `BIT STRING (SIZE (16,...))` was erroneous encoded/decoded. This is now\n  corrected and follows X.691 Section 15.6.\n\n  Own Id: OTP-7876 Aux Id: seq11220","ref":"notes.html#fixed-bugs-and-malfunctions-63"},{"type":"extras","title":"Asn1 1.6.7 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-1-6-7"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- Now asn1 starts multiple drivers to enable simultaneous encode/decode in\n  different processes for the asn1-backends using linked-in driver.\n\n  Own Id: OTP-7801","ref":"notes.html#improvements-and-new-features-33"},{"type":"extras","title":"Asn1 1.6.6 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-1-6-6"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- Decode of an open_type when the value was empty tagged type encoded with\n  indefinite length failed. This is now corrected.\n\n  Own Id: OTP-7759 Aux Id: seq11166\n\n- Encode of BIT STRING with size of exact length, on compact_bit_string format\n  in UNALIGNED PER failed when value had the right size, i.e. no padding needed.\n\n  Own Id: OTP-7763 Aux Id: seq11182","ref":"notes.html#fixed-bugs-and-malfunctions-64"},{"type":"extras","title":"Asn1 1.6.5 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-1-6-5"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- For a BIT STRING with SIZE constraint higher than 255 compiled with\n  `[per_bin,optimize, compact_bit_string]` an improper io-list was created and\n  sent to the c-driver for complete encoding. This error has been resolved.\n\n  Own Id: OTP-7734 Aux Id: seq11170","ref":"notes.html#fixed-bugs-and-malfunctions-65"},{"type":"extras","title":"Asn1 1.6.4 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-1-6-4"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- A a SEQUENCE OF with a type that is a CHOICE with ellipses occurred falsely a\n  compile error. The error causing that is now removed.\n\n  Own Id: OTP-7708 Aux Id: seq11136","ref":"notes.html#fixed-bugs-and-malfunctions-66"},{"type":"extras","title":"Asn1 1.6.3 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-1-6-3"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- constrained number with a value-range greater than 512 now has the proper\n  interpretation of the values that causes shift to the next number of units\n  (bits), According to limit condition `2^m < \"range\" =< 2^(m + 1)` then the\n  number of bits are m + 1.\n\n  Own Id: OTP-7681 Aux Id: seq11114","ref":"notes.html#fixed-bugs-and-malfunctions-67"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- Can now handle default values of simple types that is provided on its own\n  format, i.e. not just as asn1_DEFAULT.\n\n  Own Id: OTP-7678 Aux Id: seq11114","ref":"notes.html#improvements-and-new-features-34"},{"type":"extras","title":"Asn1 1.6.2 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-1-6-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- comparison of two value definitions failed due to new module name field in\n  valuedef record. It is now corrected.\n\n  Own Id: OTP-7608","ref":"notes.html#fixed-bugs-and-malfunctions-68"},{"type":"extras","title":"Asn1 1.6.1 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-1-6-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- Bug regarding propagation of parameters of parameterized type fixed.\n\n  Own Id: OTP-7174 Aux Id: seq10864\n\n- A bug, related to instantiation of a parameterized type with a type definition\n  in the parameter-list, has been removed. The definition of the parameter type\n  was in another module than the instance definition causing limited module\n  info.\n\n  Own Id: OTP-7299 Aux Id: seq10864\n\n- Removed hard-coded name that may cause name collision.\n\n  Own Id: OTP-7322 Aux Id: seq10864\n\n- Object set of a class with id with properties UNIQUE OPTIONAL and the id field\n  is lacking in the object is for now treated as a object without a unique\n  identifier, i.e. no table is generated for this object.\n\n  Own Id: OTP-7332 Aux Id: seq10864\n\n- Compiler crashed when failed to handle a OID as ValueFromObject.\n\n  Own Id: OTP-7476 Aux Id: seq10999\n\n- A corrupted encoding may cause a loop when a buffer of at least two bytes of\n  zero matches tag and length of a SET component. This behavior occurred only\n  with decoder generated with `ber` or `ber_bin` options. Now a control breaks\n  the loop.\n\n  Own Id: OTP-7533\n\n- Encode of BIT STRING longer than 255 bits with a `SIZE(integer())` constraint\n  caused a crash when spec was compiled with `per_bin, optimize` options.\n\n  Own Id: OTP-7602 Aux Id: seq11079","ref":"notes.html#fixed-bugs-and-malfunctions-69"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- Now supports REAL type of base 2 and 10\n\n  Own Id: OTP-7166 Aux Id: seq10864\n\n- By the asn1 compiler option `{record_name_prefix Name}` a prefix is chosen to\n  the name of the record generated in the .hrl and used in the generated .erl\n  files.\n\n  Own Id: OTP-7204 Aux Id: seq10853\n\n- The TypeFromObject production now covered\n\n  Own Id: OTP-7295 Aux Id: seq10468\n\n- Extended support for ObjectSetFromObjects. Production occurred as a part of\n  the RootElementSetSpec of the ObjectSetSpec. Added also support for Exclusion\n  of Element in ObjectSetSpec.\n\n  Own Id: OTP-7306 Aux Id: seq10864\n\n- Now implements RELATIVE-OID\n\n  Own Id: OTP-7334 Aux Id: seq10864","ref":"notes.html#improvements-and-new-features-35"},{"type":"extras","title":"Asn1 1.6 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-1-6"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- Now is ordering, according to the canonical order, of components in a SET\n  added. Canonical order is described in X.691 9.2 and X.680 8.6\n\n  Own Id: OTP-7375 Aux Id: unaligned PER\n\n- The precedence rules for extended constraints have been misinterpreted. The\n  rule says for instance that if there are more than one constraint on a type\n  that have extension-mark, only the last of the extension-marks would be kept.\n  This affects the encoding of PER and is now corrected.\n\n  Own Id: OTP-7400 Aux Id: OTP-7335\n\n- A constrained number with a single-value constraint that is extensible was\n  falsely encoded/decoded in aligned/unaligned PER. This is now corrected.\n\n  Own Id: OTP-7403","ref":"notes.html#fixed-bugs-and-malfunctions-70"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- The ASN.1 compiler has got a new backend supporting PER UNALIGNED. Previously\n  it was only support for PER ALIGNED.\n\n  Own Id: OTP-7335\n\n- Now the asn1-compiler handles unions and intersections of PermittedAlphabet\n  constraints.\n\n  Own Id: OTP-7374 Aux Id: unaligned PER\n\n- With the undocumented option `no_final_padding` the whole encoded message is\n  not padded to a border of a byte. Thus the returned encoded message is a\n  `bitstring`.\n\n  Own Id: OTP-7407","ref":"notes.html#improvements-and-new-features-36"},{"type":"extras","title":"Asn1 1.5.2 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-1-5-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- When duplicates of object fields were removed only one table access function\n  for each unique identifier value was generated. This can occur when several\n  object sets are merged by use of ObjectSetFromObjects.\n\n  Own Id: OTP-7263 Aux Id: seq10864\n\n- DER: For some complex types and components with reference to type in several\n  steps the default value check function was not generated. This is now fixed.\n\n  Own Id: OTP-7268 Aux Id: seq10684\n\n- Now is the tag in a tagged type as parameter propagated to the instance.\n\n  Own Id: OTP-7273 Aux Id: seq10864","ref":"notes.html#fixed-bugs-and-malfunctions-71"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- Added type T61String that is similar to TeletexString\n\n  Own Id: OTP-7264 Aux Id: seq10864","ref":"notes.html#improvements-and-new-features-37"},{"type":"extras","title":"Asn1 1.5.1 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-1-5-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- A bug related to renaming of types has been fixed.This occurred using the\n  .set.asn functionality.\n\n  Own Id: OTP-7149 Aux Id: seq10853\n\n- syntax error in ASN1 value now correctly shown\n\n  Own Id: OTP-7154 Aux Id: seq10864\n\n- Now a COMPONENTS OF construct in a parameterized type is expanded correctly\n\n  Own Id: OTP-7155 Aux Id: seq10864\n\n- Now the asn1-compiler also handles empty SEQUENCE DEFAULT values as `{}`.\n\n  Own Id: OTP-7169 Aux Id: seq10864\n\n- Now SelectionType gets the tag of the selected type.\n\n  Own Id: OTP-7171 Aux Id: seq10864\n\n- Correction of generated code for decode of an open type in a SEQUECNE OF/ SET\n  OF\n\n  Own Id: OTP-7193 Aux Id: seq10875","ref":"notes.html#fixed-bugs-and-malfunctions-72"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- Misc improvements and bug corrections regarding default values.\n\n  Own Id: OTP-7199 Aux Id: seq10864","ref":"notes.html#improvements-and-new-features-38"},{"type":"extras","title":"Asn1 1.5 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-1-5"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- Now generating records in .hrl file for instances of parameterized SEQUENCE or\n  SET.\n\n  Own Id: OTP-6835\n\n- Optimization using bitstr in encode/decode functions. Active with\n  `[per_bin, optimize]` options.\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-6882","ref":"notes.html#improvements-and-new-features-39"},{"type":"extras","title":"Asn1 1.4.6 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-1-4-6"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- Parsing and encoding/decoding of type constrained with SIZE with extension is\n  now recovered.\n\n  Own Id: OTP-6763\n\n- `inline` failed because trying to use a removed module.\n\n  Own Id: OTP-6769\n\n- Fixed problem with a reference to a type from an object. The failure was\n  caused bye change of type name when using `inline` option.\n\n  Own Id: OTP-6770\n\n- Handling of decode pattern for exclusive decode was false in the case when an\n  un-decoded component had more than one following elements that should be\n  decoded.\n\n  Own Id: OTP-6786","ref":"notes.html#fixed-bugs-and-malfunctions-73"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- Now the asn1-compiler supports two root lists in SEQUENCE and SET according to\n  alternative three in ComponentTypeLists (X.680 07/2002 section 24.1), i.e.\n  with an extension list between two ellipses.\n\n  Own Id: OTP-5067 Aux Id: seq8452","ref":"notes.html#improvements-and-new-features-40"},{"type":"extras","title":"Asn1 1.4.5 - asn1 Release Notes","doc":"","ref":"notes.html#asn1-1-4-5"},{"type":"extras","title":"Fixed Bugs and Malfunctions - asn1 Release Notes","doc":"- Merging modules by `inline` earlier disabled the driver (used in modules\n  generated with \\[optimized]/\\[optimized,driver] options). Now this is\n  repaired.\n\n  Own Id: OTP-6601\n\n- Checking phase now aware of which module an INSTANCE OF is declared in.\n\n  Own Id: OTP-6702","ref":"notes.html#fixed-bugs-and-malfunctions-74"},{"type":"extras","title":"Improvements and New Features - asn1 Release Notes","doc":"- The compiler now handle all forms of ObjectSetSpec according to ITU-T\n  recommendation X.681 (ISO/IEC 8824-2:2002).\n\n  Own Id: OTP-6698\n\n- Enhanced support of referencing object sets by ObjectSetFromObjects.\n\n  Own Id: OTP-6707\n\n- Support for parameterized object in an object set.\n\n  Own Id: OTP-6717","ref":"notes.html#improvements-and-new-features-41"},{"type":"extras","title":"Introduction","doc":"<!--\n%CopyrightBegin%\n\nSPDX-License-Identifier: Apache-2.0\n\nCopyright Ericsson AB 2023-2025. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# Introduction\n\nThe `asn1` application provides the following:\n\n- An ASN.1 compiler for Erlang, which generates encode and decode functions to\n  be used by Erlang programs sending and receiving ASN.1-specified data.\n- Runtime functions used by the generated code.\n- Support for the following encoding rules:\n  - Basic Encoding Rules (BER)\n  - Distinguished Encoding Rules (DER), a specialized form of BER that is used\n    in security-conscious applications\n  - Packed Encoding Rules (PER), both the aligned and unaligned variant\n  - JSON Encoding Rules (JER)","ref":"asn1_introduction.html"},{"type":"extras","title":"Scope - Introduction","doc":"This application covers all features of ASN.1 up to the 1997 edition of the\nspecification. In the 2002 edition, new features were introduced. The following\nfeatures of the 2002 edition are fully or partly supported:\n\n- Decimal notation (for example, `\"1.5e3`) for REAL values. The NR1, NR2, and\n  NR3 formats as explained in ISO 6093 are supported.\n- The `RELATIVE-OID` type for relative object identifiers is fully supported.\n- The subtype constraint (`CONTAINING`/`ENCODED BY`) to constrain the content of\n  an octet string or a bit string is parsed when compiling, but no further\n  action is taken. This constraint is not a PER-visible constraint.\n- The subtype constraint by regular expressions (`PATTERN`) for character string\n  types is parsed when compiling, but no further action is taken. This\n  constraint is not a PER-visible constraint.\n- Multiple-line comments as in C, `/* ... */`, are supported.","ref":"asn1_introduction.html#scope"},{"type":"extras","title":"Prerequisites - Introduction","doc":"It is assumed that the reader is familiar with the Erlang programming language,\nconcepts of OTP, and is familiar with the ASN.1 notation. The ASN.1 notation is\ndocumented in the standard definition X.680, which is the primary text. It can\nalso be helpful, but not necessary, to read the standard definitions X.681,\nX.682, X.683, X.690, and X.691.\n\nA good book explaining those reference texts is Dubuisson: ASN.1 - Communication\nBetween Heterogeneous Systems, is free to download at\n[http://www.oss.com/asn1/dubuisson.html](http://www.oss.com/asn1/dubuisson.html).","ref":"asn1_introduction.html#prerequisites"},{"type":"extras","title":"ASN.1","doc":"<!--\n%CopyrightBegin%\n\nSPDX-License-Identifier: Apache-2.0\n\nCopyright Ericsson AB 2023-2025. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# ASN.1","ref":"asn1_overview.html"},{"type":"extras","title":"Introduction - ASN.1","doc":"ASN.1 is a formal language for describing data structures to be exchanged\nbetween distributed computer systems. The purpose of ASN.1 is to have a platform\nand programming language independent notation to express types using a\nstandardized set of rules for the transformation of values of a defined type\ninto a stream of bytes. This stream of bytes can then be sent on any type of\ncommunication channel. This way, two applications written in different\nprogramming languages running on different computers, and with different\ninternal representation of data, can exchange instances of structured data\ntypes.","ref":"asn1_overview.html#introduction"},{"type":"extras","title":"Getting Started","doc":"<!--\n%CopyrightBegin%\n\nSPDX-License-Identifier: Apache-2.0\n\nCopyright Ericsson AB 2023-2025. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# Getting Started","ref":"asn1_getting_started.html"},{"type":"extras","title":"Example - Getting Started","doc":"The following example demonstrates the basic functionality used to run the\nErlang ASN.1 compiler.\n\nCreate a file named `People.asn` containing the following:\n\n```text\nPeople DEFINITIONS AUTOMATIC TAGS ::=\nBEGIN\n  Person ::= SEQUENCE {\n    name PrintableString,\n    location INTEGER {home(0),field(1),roving(2)},\n    age INTEGER OPTIONAL\n  }\nEND\n```\n\nThis file must be compiled before it can be used. The ASN.1 compiler checks that\nthe syntax is correct and that the text represents proper ASN.1 code before\ngenerating an abstract syntax tree. The code generator then uses the abstract\nsyntax tree to generate code.\n\nThe generated Erlang files are placed in the current directory or in the\ndirectory specified with option `{outdir,Dir}`.\n\nThe compiler can be called from the Erlang shell like this:\n\n```erlang\n1> asn1ct:compile(\"People\", [ber]).\nok\n```\n\nOption `verbose` can be added to get information about the generated files:\n\n```erlang\n2> asn1ct:compile(\"People\", [ber,verbose]).\nErlang ASN.1 compiling \"People.asn\"\n--{generated,\"People.asn1db\"}--\n--{generated,\"People.hrl\"}--\n--{generated,\"People.erl\"}--\nok\n```\n\nASN.1 module `People` is now accepted and the abstract syntax tree is saved in\nfile `People.asn1db`. The generated Erlang code is compiled using the Erlang\ncompiler and loaded into the Erlang runtime system. There is now an API for\n`encode/2` and `decode/2` in module `People`, which is called like this:\n\n```\n'People':encode( ,  )\n```\n\nor:\n\n```\n'People':decode( ,  )\n```\n\nAssume that there is a network application that receives instances of the ASN.1\ndefined type `Person`, modifies, and sends them back again:\n\n```erlang\nreceive\n   {Port,{data,Bytes}} ->\n       case 'People':decode('Person',Bytes) of\n           {ok,P} ->\n               {ok,Answer} = 'People':encode('Person',mk_answer(P)),\n               Port ! {self(),{command,Answer}};\n           {error,Reason} ->\n               exit({error,Reason})\n       end\n    end,\n```\n\nIn this example, a series of bytes is received from an external source and the\nbytes are then decoded into a valid Erlang term. This was achieved with the call\n`'People':decode('Person',Bytes)`, which returned an Erlang value of the ASN.1\ntype `Person`. Then an answer was constructed and encoded using\n`'People':encode('Person',Answer)`, which takes an instance of a defined ASN.1\ntype and transforms it to a binary according to the BER or PER encoding rules.\n\nThe encoder and decoder can also be run from the shell:\n\n```erlang\n2> Rockstar = {'Person',\"Some Name\",roving,50}.\n{'Person',\"Some Name\",roving,50}\n3> {ok,Bin} = 'People':encode('Person',Rockstar).\n{ok,<<243,17,19,9,83,111,109,101,32,78,97,109,101,2,1,2,\n      2,1,50>>}\n4> {ok,Person} = 'People':decode('Person',Bin).\n{ok,{'Person',\"Some Name\",roving,50}}\n```","ref":"asn1_getting_started.html#example"},{"type":"extras","title":"Module Dependencies - Getting Started","doc":"It is common that ASN.1 modules import defined types, values, and other entities\nfrom another ASN.1 module.\n\nEarlier versions of the ASN.1 compiler required that modules that were imported\nfrom had to be compiled before the module that imported. This caused problems\nwhen ASN.1 modules had circular dependencies.\n\nReferenced modules are now parsed when the compiler finds an entity that is\nimported. No code is generated for the referenced module. However, the compiled\nmodules rely on that the referenced modules are also compiled.","ref":"asn1_getting_started.html#module-dependencies"},{"type":"extras","title":"ASN.1 Application User Interface - Getting Started","doc":"The `ASN.1` application provides the following two separate user interfaces:\n\n- The module `asn1ct`, which provides the compile-time functions (including the\n  compiler)\n- The module `asn1rt_nif`, which provides the runtime functions for the ASN.1\n  decoder for the BER back end\n\nThe reason for this division of the interfaces into compile-time and runtime is\nthat only runtime modules (`asn1rt_nif`) need to be loaded in an embedded system.","ref":"asn1_getting_started.html#asn-1-application-user-interface"},{"type":"extras","title":"Compile-Time Functions - Getting Started","doc":"The ASN.1 compiler can be started directly from the command line by the `erlc`\nprogram. This is convenient when compiling many ASN.1 files from the command\nline or when using Makefiles. Here some examples showing of how `erlc` can\ncompile ASN.1 modules:\n\n```text\nerlc Person.asn\nerlc -bper Person.asn\nerlc -bber ../Example.asn\nerlc -o ../asnfiles -I ../asnfiles -I /usr/local/standards/asn1 Person.asn\n```\n\nUseful options for the ASN.1 compiler:\n\n- **`-b[ber | per | uper | jer]`** - Choice of encoding rules. If omitted, `ber`\n  is the default.\n\n- **`-o OutDirectory`** - Where to put the generated files. Default is the\n  current directory.\n\n- **`-I IncludeDir`** - Where to search for `.asn1db` files and ASN.1 source\n  specs to resolve references to other modules. This option can be repeated many\n  times if there are several places to search in. The compiler searches the\n  current directory first.\n\n- **`+der`** - DER encoding rule. Only when using option `-bber`.\n\n- **`+jer`** - Functions `jer_encode/2` and `jer_decode/2` for JSON encoding\n  rules are generated together with functions for `ber` or `per`. Only to be\n  used when the main encoding option is `-bber`, `-bper` or `-buper`.\n\n- **`+maps`** - Use maps instead of records to represent the `SEQUENCE` and\n  `SET` types. No `.hrl` files will be generated. See the section\n  [Map representation for SEQUENCE and SET](asn1_getting_started.md#MAP_SEQ_SET)\n  for more information.\n\n- **`+asn1config`** - This functionality works together with option `ber`. It\n  enables the specialized decodes, see section\n  [Specialized Decode](asn1_spec.md).\n\n- **`+undec_rest`** - A buffer that holds a message being decoded can also have\n  trailing bytes. If those trailing bytes are important, they can be returned\n  along with the decoded value by compiling the ASN.1 specification with option\n  `+undec_rest`. The return value from the decoder is `{ok,Value,Rest}` where\n  `Rest` is a binary containing the trailing bytes.\n\n- **`+'Any Erlc Option'`** - Any option can be added to the Erlang compiler when\n  compiling the generated Erlang files. Any option unrecognized by the ASN.1\n  compiler is passed to the Erlang compiler.\n\nFor a complete description of `erlc`, see ERTS Reference Manual.\n\nThe compiler and other compile-time functions can also be started from the\nErlang shell. Here follows a brief description of the primary functions. For a\ncomplete description of each function, see module `asn1ct` in the\n[ASN.1 Reference Manual](`m:asn1ct`).\n\nThe compiler is started by `asn1ct:compile/1` with default options, or\n`asn1ct:compile/2` if explicit options are given.\n\nExample:\n\n```text\nasn1ct:compile(\"H323-MESSAGES\").\n```\n\nThis is equivalent to:\n\n```text\nasn1ct:compile(\"H323-MESSAGES\", [ber]).\n```\n\nIf PER encoding is wanted:\n\n```text\nasn1ct:compile(\"H323-MESSAGES\", [per]).\n```\n\nThe generic encode and decode functions can be called as follows:\n\n```text\n'H323-MESSAGES':encode('SomeChoiceType', {call,<<\"octetstring\">>}).\n'H323-MESSAGES':decode('SomeChoiceType', Bytes).\n```","ref":"asn1_getting_started.html#compile-time-functions"},{"type":"extras","title":"Runtime Functions - Getting Started","doc":"When an ASN.1 specification is compiled with option `ber`, the `asn1rt_nif`\nmodule and the NIF library in `asn1/priv_dir` are needed at runtime.\n\nBy calling function `info/0` in a generated module, you get information about\nwhich compiler options were used.","ref":"asn1_getting_started.html#runtime-functions"},{"type":"extras","title":"Special Decode Functionality for JSON Encoding Rules (JER) - Getting Started","doc":"When using the JSON encoding rules, it is possible to call the\n`decode/2` function in the following way with data that has already\nbeen decoded by `json:decode/1`:\n\n```erlang\nSomeModule:decode(Type, {json_decoded, Decoded}).\n```\n\nExample:\n\n```erlang\n1> asn1ct:compile(\"People\", [jer]).\nok\n2> Rockstar = {'Person',\"Vince Eclipse\",roving,50}.\n{'Person',\"Vince Eclipse\",roving,50}\n3> {ok,Bin} = 'People':encode('Person', Rockstar).\n{ok,<<\"{\\\"name\\\":\\\"Vince Eclipse\\\",\\\"location\\\":2,\\\"age\\\":50}\">>}\n4> 'People':decode('Person', Bin).\n{ok,{'Person',\"Vince Eclipse\",roving,50}}\n5> 'People':decode('Person', {json_decoded,json:decode(Bin)}).\n{ok,{'Person',\"Vince Eclipse\",roving,50}}\n\n```","ref":"asn1_getting_started.html#special-decode-functionality-for-json-encoding-rules-jer"},{"type":"extras","title":"Errors - Getting Started","doc":"Errors detected at compile-time are displayed on the screen together with line\nnumbers indicating where in the source file the respective error was detected.\nIf no errors are found, an Erlang ASN.1 module is created.\n\nThe runtime encoders and decoders execute within a `catch` and return `{ok, Data}`\nor `{error, {asn1, Description}}` where `Description` is an Erlang term\ndescribing the error.\n\nCurrently, `Description` looks like this: `{ErrorDescription, StackTrace}`.\nApplications should not depend on the exact contents of `Description` as it\ncould change in the future.","ref":"asn1_getting_started.html#errors"},{"type":"extras","title":"Multi-File Compilation - Getting Started","doc":"There are various reasons for using multi-file compilation:\n\n- To choose the name for the generated module, for example, because you need to\n  compile the same specs for different encoding rules.\n- You want only one resulting module.\n\nSpecify which ASN.1 specs to compile in a module with extension `.set.asn`.\nChoose a module name and provide the names of the ASN.1 specs. For example, if\nyou have the specs `File1.asn`, `File2.asn`, and `File3.asn`, your module\n`MyModule.set.asn` looks as follows:\n\n```text\nFile1.asn\nFile2.asn\nFile3.asn\n```\n\nIf you compile with the following, the result is one merged module\n`MyModule.erl` with the generated code from the three ASN.1 specs:\n\n```text\n% erlc MyModule.set.asn\n```","ref":"asn1_getting_started.html#multi-file-compilation"},{"type":"extras","title":"Note about tags - Getting Started","doc":"Tags used to be important for all users of ASN.1, because it was necessary to\nto manually add tags to certain constructs in order for the ASN.1 specification to\nbe valid. Example of an old-style specification:\n\n```erlang\nTags DEFINITIONS ::=\nBEGIN\n  Afters ::= CHOICE { cheese [0] IA5String,\n                      dessert [1] IA5String }\nEND\n```\n\nWithout the tags (the numbers in square brackets) the ASN.1 compiler refused to\ncompile the file.\n\nIn 1994 the global tagging mode `AUTOMATIC TAGS` was introduced. By putting\n`AUTOMATIC TAGS` in the module header, the ASN.1 compiler automatically adds\ntags when needed. The following is the same specification in `AUTOMATIC TAGS`\nmode:\n\n```erlang\nTags DEFINITIONS AUTOMATIC TAGS ::=\nBEGIN\n  Afters ::= CHOICE { cheese IA5String,\n                      dessert IA5String }\nEND\n```\n\n[](){: #ASN1Types }","ref":"asn1_getting_started.html#note-about-tags"},{"type":"extras","title":"ASN.1 Types - Getting Started","doc":"This section describes the ASN.1 types including their functionality, purpose,\nand how values are assigned in Erlang.\n\nASN.1 has both primitive and constructed types:\n\n| Primitive Types                                                | Constructed Types                                            |\n| -------------------------------------------------------------- | ------------------------------------------------------------ |\n| [BOOLEAN](asn1_getting_started.md#boolean)                     | [SEQUENCE](asn1_getting_started.md#sequence)                 |\n| [INTEGER](asn1_getting_started.md#integer)                     | [SET](asn1_getting_started.md#set)                           |\n| [REAL](asn1_getting_started.md#real)                           | [CHOICE](asn1_getting_started.md#choice)                     |\n| [NULL](asn1_getting_started.md#null)                           | [SET OF and SEQUENCE OF](asn1_getting_started.md#SOF)        |\n| [ENUMERATED](asn1_getting_started.md#enumerated)               | [ANY](asn1_getting_started.md#ANY)                           |\n| [BIT STRING](asn1_getting_started.md#bit-string)               | [ANY DEFINED BY](asn1_getting_started.md#ANY)                |\n| [OCTET STRING](asn1_getting_started.md#octet-string)           | [EXTERNAL](asn1_getting_started.md#NegotiationTypes)         |\n| [Character Strings](asn1_getting_started.md#character-strings) | [EMBEDDED PDV](asn1_getting_started.md#NegotiationTypes)     |\n| [OBJECT IDENTIFIER](asn1_getting_started.md#object-identifier) | [CHARACTER STRING](asn1_getting_started.md#NegotiationTypes) |\n| [Object Descriptor](asn1_getting_started.md#object-descriptor) |                                                              |\n| [TIME Types](asn1_getting_started.md#The-TIME-types)           |                                                              |\n\n_Table: Supported ASN.1 Types_\n\n[](){: #TypeNameValue }\n\n> #### Note {: .info }\n>\n> The values of each ASN.1 type have their own representation in Erlang, as\n> described in the following sections. Users must provide these values for\n> encoding according to the representation, as shown in the following example:\n\n```text\nOperational ::= BOOLEAN --ASN.1 definition\n```\n\nIn Erlang code it can look as follows:\n\n```erlang\nVal = true,\n{ok,Bytes} = MyModule:encode('Operational', Val),\n```","ref":"asn1_getting_started.html#asn-1-types"},{"type":"extras","title":"BOOLEAN - Getting Started","doc":"Booleans in ASN.1 express values that can be either `TRUE` or `FALSE`. The\nmeanings assigned to `TRUE` and `FALSE` are outside the scope of this text.\n\nIn ASN.1 it is possible to have:\n\n```text\nOperational ::= BOOLEAN\n```\n\nAssigning a value to type `Operational` in Erlang is possible by using the\nfollowing Erlang code:\n\n```erlang\nMyvar1 = true,\n```\n\nThus, in Erlang the atoms `true` and `false` are used to encode a boolean value.","ref":"asn1_getting_started.html#boolean"},{"type":"extras","title":"INTEGER - Getting Started","doc":"An ASN.1 INTEGER is represented by an integer in Erlang.\n\nThe concept of subtyping can be applied to integers and to other ASN.1 types.\nThe details of subtyping are not explained here; for more information, see\nX.680. Various syntaxes are allowed when defining a type as an integer:\n\n```erlang\nT1 ::= INTEGER\nT2 ::= INTEGER (-2..7)\nT3 ::= INTEGER (0..MAX)\nT4 ::= INTEGER (0<..MAX)\nT5 ::= INTEGER (MIN<..-99)\nT6 ::= INTEGER {red(0),blue(1),white(2)}\n```\n\nThe Erlang representation of an ASN.1 `INTEGER` is an integer or an atom if a\n`Named Number List` (see `T6` in the previous list) is specified.\n\nThe following is an example of Erlang code that assigns values for the types in\nthe previous list:\n\n```erlang\nT1value = 0,\nT2value = 6,\nT6value1 = blue,\nT6value2 = 0,\nT6value3 = white\n```\n\nThese Erlang variables are now bound to valid instances of ASN.1 defined types.\nThis style of value can be passed directly to the encoder for transformation\ninto a series of bytes.\n\nThe decoder returns an atom if the value corresponds to a symbol in the\n`Named Number List`.","ref":"asn1_getting_started.html#integer"},{"type":"extras","title":"REAL - Getting Started","doc":"The following ASN.1 type is used for real numbers:\n\n```text\nR1 ::= REAL\n```\n\nIt is assigned a value in Erlang as follows:\n\n```text\nR1value1 = \"2.14\",\nR1value2 = {256,10,-2}\n```\n\nIn the last line, notice that the tuple \\{256,10,-2\\} is the real number 2.56 in\na special notation, which encodes faster than simply stating the number as\n`\"2.56\"`. The arity three tuple is `{Mantissa,Base,Exponent}`, that is,\n`Mantissa * Base^Exponent`.\n\nThe following special values are also recognized:\n\n```text\nR1value3 = 0,\nR1value4 = 'PLUS-INFINITY',\nR1value5 = 'MINUS-INFINITY'\n```","ref":"asn1_getting_started.html#real"},{"type":"extras","title":"NULL - Getting Started","doc":"The type `NULL` is suitable where supply and recognition of a value is important\nbut the actual value is not.\n\n```text\nNotype ::= NULL\n```\n\nThis type is assigned in Erlang as follows:\n\n```text\nN1 = 'NULL',\n```\n\nThe actual value is the quoted atom `'NULL'`.","ref":"asn1_getting_started.html#null"},{"type":"extras","title":"ENUMERATED - Getting Started","doc":"The type `ENUMERATED` can be used when the value you want to describe can only\ntake one of a set of predefined values. Example:\n\n```text\nDaysOfTheWeek ::= ENUMERATED {\n    sunday(1),monday(2),tuesday(3),\n    wednesday(4),thursday(5),friday(6),saturday(7) }\n```\n\nFor example, to assign a weekday value in Erlang, use the same atom as in the\n`Enumerations` of the type definition:\n\n```text\nDay1 = saturday,\n```\n\nThe enumerated type is similar to an integer type, when defined with a set of\npredefined values. The difference is that an enumerated type can only have\nspecified values, whereas an integer can have any value.","ref":"asn1_getting_started.html#enumerated"},{"type":"extras","title":"BIT STRING - Getting Started","doc":"The type `BIT STRING` can be used to model information that is made up of\narbitrary length series of bits. It is intended to be used for selection of\nflags, not for binary files.\n\nIn ASN.1, `BIT STRING` definitions can look as follows:\n\n```erlang\nBits1 ::= BIT STRING\nBits2 ::= BIT STRING {foo(0),bar(1),gnu(2),gnome(3),punk(14)}\n```\n\nThe following two notations are available for representation of `BIT STRING`\nvalues in Erlang and as input to the encode functions:\n\n1. A bitstring. By default, a `BIT STRING` with no symbolic names is decoded to\n   an Erlang bitstring.\n1. A list of atoms corresponding to atoms in the `NamedBitList` in the\n   `BIT STRING` definition. A `BIT STRING` with symbolic names is always decoded\n   to the format shown in the following example:\n\n```text\nBits1Val1 = <<0:1,1:1,0:1,1:1,1:1>>,\nBits2Val1 = [gnu,punk],\nBits2Val2 = <<2#1110:4>>,\nBits2Val3 = [bar,gnu,gnome],\n```\n\n`Bits2Val2` and `Bits2Val3` denote the same value.\n\n`Bits2Val1` is assigned symbolic values. The assignment means that the bits\ncorresponding to `gnu` and `punk`, that is, bits 2 and 14 are set to 1, and the\nrest are set to 0. The symbolic values are shown as a list of values. If a named\nvalue, which is not specified in the type definition, is shown, a runtime error\noccurs.\n\n`BIT STRING`s can also be subtyped with, for example, a `SIZE` specification:\n\n```text\nBits3 ::= BIT STRING (SIZE(0..31))\n```\n\nThis means that no bit higher than 31 can be set.\n\n#### Deprecated Representations for BIT STRING\n\nIn addition to the representations described earlier, the following deprecated\nrepresentations are available if the specification has been compiled with option\n`legacy_erlang_types`:\n\n1. Aa a list of binary digits (0 or 1). This format is accepted as input to the\n   encode functions, and a `BIT STRING` is decoded to this format if option\n   `legacy_bit_string` is given.\n1. As `{Unused,Binary}` where `Unused` denotes how many trailing zero-bits 0-7\n   that are unused in the least significant byte in `Binary`. This format is\n   accepted as input to the encode functions, and a `BIT STRING` is decoded to\n   this format if `compact_bit_string` has been given.\n1. As a hexadecimal number (or an integer). Avoid this as it is easy to\n   misinterpret a `BIT STRING` value in this format.","ref":"asn1_getting_started.html#bit-string"},{"type":"extras","title":"OCTET STRING - Getting Started","doc":"`OCTET STRING` is the simplest of all ASN.1 types. `OCTET STRING` only moves or\ntransfers, for example, binary files or other unstructured information complying\nwith two rules: the bytes consist of octets and encoding is not required.\n\nIt is possible to have the following ASN.1 type definitions:\n\n```erlang\nO1 ::= OCTET STRING\nO2 ::= OCTET STRING (SIZE(28))\n```\n\nWith the following example assignments in Erlang:\n\n```text\nO1Val = <<17,13,19,20,0,0,255,254>>,\nO2Val = <<\"must be exactly 28 chars....\">>,\n```\n\nBy default, an `OCTET STRING` is always represented as an Erlang binary. If the\nspecification has been compiled with option `legacy_erlang_types`, the encode\nfunctions accept both lists and binaries, and the decode functions decode an\n`OCTET STRING` to a list.","ref":"asn1_getting_started.html#octet-string"},{"type":"extras","title":"Character Strings - Getting Started","doc":"ASN.1 supports a wide variety of character sets. The main difference between an\n`OCTET STRING` and a character string is that the `OCTET STRING` has no imposed\nsemantics on the bytes delivered.\n\nHowever, when using, for example, `IA5String` (which closely resembles ASCII),\nbyte 65 (in decimal notation) _means_ character 'A'.\n\nFor example, if a defined type is to be a VideotexString and an octet is\nreceived with the unsigned integer value `X`, the octet is to be interpreted as\nspecified in standard ITU-T T.100, T.101.\n\nThe ASN.1 compiler does not determine the correct interpretation of\neach BER string octet value with different character strings. The application is\nresponsible for interpretation of octets. Therefore, from the BER string point\nof view, octets are very similar to character strings and are compiled in the\nsame way.\n\nWhen PER is used, there is a significant difference in the encoding scheme\nfor `OCTET STRING`s and other strings. The constraints specified for a type\nare especially important for PER, because they affect the encoding.\n\nExamples:\n\n```erlang\nDigs ::= NumericString (SIZE(1..3))\nTextFile ::= IA5String (SIZE(0..64000))\n```\n\nThe corresponding Erlang assignments:\n\n```c\nDigsVal1 = \"456\",\nDigsVal2 = \"123\",\nTextFileVal1 = \"abc...xyz...\",\nTextFileVal2 = [88,76,55,44,99,121 .......... a lot of characters here ....]\n```\n\nThe Erlang representation for `BMPString` and `UniversalString` is either a list\nof ASCII values or a list of quadruples. The quadruple representation associates\nto the Unicode standard representation of characters. The ASCII characters are\nall represented by quadruples beginning with three zeros like `{0,0,0,65}` for\ncharacter 'A'. When decoding a value for these strings, the result is a list of\nquadruples, or integers when the value is an ASCII character.\n\nThe following example shows how it works. Assume the following specification is\nin file `PrimStrings.asn1`:\n\n```text\nPrimStrings DEFINITIONS AUTOMATIC TAGS ::=\nBEGIN\n   BMP ::= BMPString\nEND\n```\n\nEncoding and decoding some strings:\n\n```erlang\n1> asn1ct:compile('PrimStrings', [ber]).\nok\n2> {ok,Bytes1} = 'PrimStrings':encode('BMP', [{0,0,53,53},{0,0,45,56}]).\n{ok,<<30,4,53,54,45,56>>}\n3> 'PrimStrings':decode('BMP', Bytes1).\n{ok,[{0,0,53,53},{0,0,45,56}]}\n4> {ok,Bytes2} = 'PrimStrings':encode('BMP', [{0,0,53,53},{0,0,0,65}]).\n{ok,<<30,4,53,53,0,65>>}\n5> 'PrimStrings':decode('BMP', Bytes2).\n{ok,[{0,0,53,53},65]}\n6> {ok,Bytes3} = 'PrimStrings':encode('BMP', \"BMP string\").\n{ok,<<30,20,0,66,0,77,0,80,0,32,0,115,0,116,0,114,0,105,0,110,0,103>>}\n7> 'PrimStrings':decode('BMP', Bytes3).\n{ok,\"BMP string\"}\n```\n\nType `UTF8String` is represented as a UTF-8 encoded binary in Erlang. Such\nbinaries can be created directly using the binary syntax or by converting from a\nlist of Unicode code points using function `unicode:characters_to_binary/1`.\n\nThe following shows examples of how UTF-8 encoded binaries can be created and\nmanipulated:\n\n```erlang\n1> Gs = \"Мой маленький Гном\".\n[1052,1086,1081,32,1084,1072,1083,1077,1085,1100,1082,1080,\n 1081,32,1043,1085,1086,1084]\n2> Gbin = unicode:characters_to_binary(Gs).\n<<208,156,208,190,208,185,32,208,188,208,176,208,187,208,\n  181,208,189,209,140,208,186,208,184,208,185,32,208,147,\n  208,...>>\n3> Gbin = <<\"Мой маленький Гном\"/utf8>>.\n<<208,156,208,190,208,185,32,208,188,208,176,208,187,208,\n  181,208,189,209,140,208,186,208,184,208,185,32,208,147,\n  208,...>>\n4> Gs = unicode:characters_to_list(Gbin).\n[1052,1086,1081,32,1084,1072,1083,1077,1085,1100,1082,1080,\n 1081,32,1043,1085,1086,1084]\n```\n\nFor details, see the `m:unicode` module in STDLIB.\n\nIn the following example, this ASN.1 specification is used:\n\n```text\nUTF DEFINITIONS AUTOMATIC TAGS ::=\nBEGIN\n   UTF ::= UTF8String\nEND\n```\n\nEncoding and decoding a string with Unicode characters:\n\n```erlang\n5> asn1ct:compile('UTF', [ber]).\nok\n6> {ok,Bytes1} = 'UTF':encode('UTF', <<\"Гном\"/utf8>>).\n{ok,<<12,8,208,147,208,189,208,190,208,188>>}\n7> {ok,Bin1} = 'UTF':decode('UTF', Bytes1).\n{ok,<<208,147,208,189,208,190,208,188>>}\n8> io:format(\"~ts\\n\", [Bin1]).\nГном\nok\n9> unicode:characters_to_list(Bin1).\n[1043,1085,1086,1084]\n```","ref":"asn1_getting_started.html#character-strings"},{"type":"extras","title":"OBJECT IDENTIFIER - Getting Started","doc":"The type `OBJECT IDENTIFIER` is used whenever a unique identity is required. An\nASN.1 module, a transfer syntax, and so on, is identified with an\n`OBJECT IDENTIFIER`. Assume the following example:\n\n```erlang\nOid ::= OBJECT IDENTIFIER\n```\n\nTherefore, the following example is a valid Erlang instance of type `Oid`:\n\n```text\nOidVal1 = {1,2,55},\n```\n\nThe `OBJECT IDENTIFIER` value is a tuple with the consecutive integer values.\n\nThe first value is limited to the values 0, 1, or 2. The second value must be in\nthe range 0 through 39 when the first value is 0 or 1.\n\nThe `OBJECT IDENTIFIER` is an important type and it is widely used within\ndifferent standards to identify various objects uniquely. Dubuisson: _ASN.1 -\nCommunication Between Heterogeneous Systems_ includes an easy-to-understand\ndescription of the use of `OBJECT IDENTIFIER`.","ref":"asn1_getting_started.html#object-identifier"},{"type":"extras","title":"Object Descriptor - Getting Started","doc":"Values of this type can be assigned a value as an ordinary string as follows:\n\n```text\n\"This is the value of an Object descriptor\"\n```\n\n[](){: #The-TIME-types }","ref":"asn1_getting_started.html#object-descriptor"},{"type":"extras","title":"TIME Types - Getting Started","doc":"Two time types are defined within ASN.1: Generalized Time and Universal Time\nCoordinated (UTC). Both are assigned a value as an ordinary string within double\nquotes, for example, `\"19820102070533.8\"`.\n\nFor DER encoding, the compiler does not check the validity of the time values.\nThe DER requirements upon those strings are regarded as a matter for the\napplication to fulfill.","ref":"asn1_getting_started.html#time-types"},{"type":"extras","title":"SEQUENCE - Getting Started","doc":"The structured types of ASN.1 are constructed from other types in a manner\nsimilar to the concepts of arrays and structs in C.\n\nA `SEQUENCE` in ASN.1 is comparable with a struct in C and a record in Erlang. A\n`SEQUENCE` can be defined as follows:\n\n```erlang\nPdu ::= SEQUENCE {\n   a INTEGER,\n   b REAL,\n   c OBJECT IDENTIFIER,\n   d NULL }\n```\n\nThis is a 4-component structure called `Pdu`. By default, a `SEQUENCE` is\nrepresented by a record in Erlang. It can also be represented as a map; see\n[Map representation for SEQUENCE and SET](asn1_getting_started.md#MAP_SEQ_SET).\nFor each `SEQUENCE` and `SET` in an ASN.1 module an Erlang record declaration is\ngenerated. For `Pdu`, a record like the following is defined:\n\n```erlang\n-record('Pdu', {a, b, c, d}).\n```\n\nThe record declarations for a module `M` are placed in a separate `M.hrl` file.\n\nValues can be assigned in Erlang as follows:\n\n```erlang\nMyPdu = #'Pdu'{a=22,b=77.99,c={0,1,2,3,4},d='NULL'}.\n```\n\nThe decode functions return a record as result when decoding a `SEQUENCE` or a\n`SET`.\n\nA `SEQUENCE` and a `SET` can contain a component with a `DEFAULT` keyword\nfollowed by the actual value, which is the default value. The `DEFAULT` keyword\nmeans that the application doing the encoding can omit encoding of the value,\nwhich results in fewer bytes to send to the receiving application.\n\nAn application can use the atom `asn1_DEFAULT` to indicate that the encoding is\nto be omitted for that position in the `SEQUENCE`.\n\nDepending on the encoding rules, the encoder can also compare the given value to\nthe default value and automatically omit the encoding if the values are equal.\nHow much effort the encoder makes to compare the values depends on the encoding\nrules. The DER encoding rules forbid encoding a value equal to the default\nvalue, so it has a more thorough and time-consuming comparison than the encoders\nfor the other encoding rules.\n\nIn the following example, this ASN.1 specification is used:\n\n```text\nFile DEFINITIONS AUTOMATIC TAGS ::=\nBEGIN\nSeq1 ::= SEQUENCE {\n    a INTEGER DEFAULT 1,\n    b Seq2 DEFAULT {aa TRUE, bb 15}\n}\n\nSeq2 ::= SEQUENCE {\n    aa BOOLEAN,\n    bb INTEGER\n}\n\nSeq3 ::= SEQUENCE {\n    bs BIT STRING {a(0), b(1), c(2)} DEFAULT {a, c}\n}\nEND\n```\n\nExample where the BER encoder is able to omit encoding of the default values:\n\n```erlang\n1> asn1ct:compile('File', [ber]).\nok\n2> 'File':encode('Seq1', {'Seq1',asn1_DEFAULT,asn1_DEFAULT}).\n{ok,<<48,0>>}\n3> 'File':encode('Seq1', {'Seq1',1,{'Seq2',true,15}}).\n{ok,<<48,0>>}\n```\n\nExample with a named `BIT STRING` where the BER encoder does not omit the\nencoding:\n\n```erlang\n4> 'File':encode('Seq3', {'Seq3',asn1_DEFAULT).\n{ok,<<48,0>>}\n5> 'File':encode('Seq3', {'Seq3',<<16#101:3>>).\n{ok,<<48,4,128,2,5,160>>}\n```\n\nThe DER encoder omits the encoding for the same `BIT STRING`:\n\n```erlang\n6> asn1ct:compile('File', [ber,der]).\nok\n7> 'File':encode('Seq3', {'Seq3',asn1_DEFAULT).\n{ok,<<48,0>>}\n8> 'File':encode('Seq3', {'Seq3',<<16#101:3>>).\n{ok,<<48,0>>}\n```","ref":"asn1_getting_started.html#sequence"},{"type":"extras","title":"SET - Getting Started","doc":"In Erlang, the `SET` type is used exactly as `SEQUENCE`. Notice that if BER or\nDER encoding rules are used, decoding a `SET` is slower than decoding a\n`SEQUENCE` because the components must be sorted.","ref":"asn1_getting_started.html#set"},{"type":"extras","title":"Extensibility for SEQUENCE and SET - Getting Started","doc":"When a `SEQUENCE` or `SET` contains an extension marker and extension components\nas the following, the type can get more components in newer versions of the\nASN.1 spec:\n\n```text\nSExt ::= SEQUENCE {\n           a INTEGER,\n           ...,\n           b BOOLEAN }\n```\n\nIn this case it has got a new component `b`. Thus, incoming messages that are\ndecoded can have more or fever components than this one.\n\nThe component `b` is treated as an original component when encoding a message.\nIn this case, as it is not an optional element, it must be encoded.\n\nDuring decoding, the `b` field of the record gets the decoded value of the `b`\ncomponent, if present, otherwise the value `asn1_NOVALUE`.\n\n[](){: #MAP_SEQ_SET }","ref":"asn1_getting_started.html#extensibility-for-sequence-and-set"},{"type":"extras","title":"Map representation for SEQUENCE and SET - Getting Started","doc":"If the ASN.1 module has been compiled with option `maps`, the types `SEQUENCE`\nand `SET` are represented as maps.\n\nIn the following example, this ASN.1 specification is used:\n\n```text\nFile DEFINITIONS AUTOMATIC TAGS ::=\nBEGIN\nSeq1 ::= SEQUENCE {\n    a INTEGER DEFAULT 42,\n    b BOOLEAN OPTIONAL,\n    c IA5String\n}\nEND\n```\n\nOptional fields are to be omitted from the map if they have no value:\n\n```erlang\n1> asn1ct:compile('File', [per,maps]).\nok\n2> {ok,E} = 'File':encode('Seq1', #{a=>0,c=>\"string\"}).\n{ok,<<128,1,0,6,115,116,114,105,110,103>>}\n```\n\nWhen decoding, optional fields will be omitted from the map:\n\n```erlang\n3> 'File':decode('Seq1', E).\n{ok,#{a => 0,c => \"string\"}}\n```\n\nDefault values can be omitted from the map:\n\n```erlang\n4> {ok,E2} = 'File':encode('Seq1', #{c=>\"string\"}).\n{ok,<<0,6,115,116,114,105,110,103>>}\n5> 'File':decode('Seq1', E2).\n{ok,#{a => 42,c => \"string\"}}\n```\n\n> #### Note {: .info }\n>\n> It is not allowed to use the atoms `asn1_VALUE` and `asn1_DEFAULT` with maps.","ref":"asn1_getting_started.html#map-representation-for-sequence-and-set"},{"type":"extras","title":"CHOICE - Getting Started","doc":"The type `CHOICE` is a space saver and is similar to the concept of union in\nC.\n\nAssume the following:\n\n```text\nSomeModuleName DEFINITIONS AUTOMATIC TAGS ::=\nBEGIN\nT ::= CHOICE {\n        x REAL,\n        y INTEGER,\n        z OBJECT IDENTIFIER }\nEND\n```\n\nIt is then possible to assign values as follows:\n\n```erlang\nTVal1 = {y,17},\nTVal2 = {z,{0,1,2}},\n```\n\nA `CHOICE` value is always represented as the tuple `{ChoiceAlternative, Val}`\nwhere `ChoiceAlternative` is an atom denoting the selected choice alternative.\n\n#### Extensible CHOICE\n\nWhen a `CHOICE` contains an extension marker and the decoder detects an unknown\nalternative of the `CHOICE`, the value is represented as follows:\n\n```text\n{asn1_ExtAlt, BytesForOpenType}\n```\n\nHere `BytesForOpenType` is a list of bytes constituting the encoding of the\n\"unknown\" `CHOICE` alternative.\n\n[](){: #SOF }","ref":"asn1_getting_started.html#choice"},{"type":"extras","title":"SET OF and SEQUENCE OF - Getting Started","doc":"The types `SET OF` and `SEQUENCE OF` correspond to the concept of an array in\nseveral programming languages. The Erlang syntax for both types is\nstraightforward, for example:\n\n```text\nArr1 ::= SET SIZE (5) OF INTEGER (4..9)\nArr2 ::= SEQUENCE OF OCTET STRING\n```\n\nIn Erlang the following can apply:\n\n```text\nArr1Val = [4,5,6,7,8],\nArr2Val = [\"abc\",[14,34,54],\"Octets\"],\n```\n\nNotice that the definition of type `SET OF` implies that the order of the\ncomponents is undefined, but in practice there is no difference between `SET OF`\nand `SEQUENCE OF`. The ASN.1 compiler for Erlang does not randomize the order of\nthe `SET OF` components before encoding.\n\nHowever, for a value of type `SET OF`, the DER encoding format requires the\nelements to be sent in ascending order of their encoding, which implies an\nexpensive sorting procedure in runtime. Therefore it is recommended to use\n`SEQUENCE OF` instead of `SET OF` if possible.\n\n[](){: #ANY }","ref":"asn1_getting_started.html#set-of-and-sequence-of"},{"type":"extras","title":"ANY and ANY DEFINED BY - Getting Started","doc":"The types `ANY` and `ANY DEFINED BY` have been removed from the standard\nsince 1994. It is recommended not to use these types any more. They can,\nhowever, exist in some old ASN.1 modules. The idea with this type was to leave a\n\"hole\" in a definition where it was possible to put unspecified data of any\nkind, even non-ASN.1 data.\n\nA value of this type is encoded as an `open type`.\n\nInstead of `ANY` and `ANY DEFINED BY`, it is recommended to use\ninformation object classes, table constraints, and parameterization. In\nparticular the construct `TYPE-IDENTIFIER.@Type` accomplishes the same as the\ndeprecated `ANY`.\n\nAlso see [Information objects](asn1_getting_started.md#Information-Object).\n\n[](){: #NegotiationTypes }","ref":"asn1_getting_started.html#any-and-any-defined-by"},{"type":"extras","title":"EXTERNAL, EMBEDDED PDV, and CHARACTER STRING - Getting Started","doc":"The types `EXTERNAL`, `EMBEDDED PDV`, and `CHARACTER STRING` are used in\npresentation layer negotiation. They are encoded according to their associated\ntype, see X.680.\n\nThe type `EXTERNAL` had a slightly different associated type before 1994. X.691\nstates that encoding must follow the older associated type. So, generated\nencode/decode functions convert values of the newer format to the older format\nbefore encoding. This implies that it is allowed to use `EXTERNAL` type values\nof either format for encoding. Decoded values are always returned in the newer\nformat.","ref":"asn1_getting_started.html#external-embedded-pdv-and-character-string"},{"type":"extras","title":"Embedded Named Types - Getting Started","doc":"The structured types previously described can have other named types as their\ncomponents. The general syntax to assign a value to component `C` of a named\nASN.1 type `T` in Erlang is the record syntax `#'T'{'C'=Value}`. Here `Value`\ncan be a value of yet another type `T2`, for example:\n\n```text\nEmbeddedExample DEFINITIONS AUTOMATIC TAGS ::=\nBEGIN\nB ::= SEQUENCE {\n        a Arr1,\n        b T }\n\nArr1 ::= SET SIZE (5) OF INTEGER (4..9)\n\nT ::= CHOICE {\n        x REAL,\n        y INTEGER,\n        z OBJECT IDENTIFIER }\n        END\n```\n\n`SEQUENCE` `b` can be encoded as follows in Erlang:\n\n```erlang\n1> 'EmbeddedExample':encode('B', {'B',[4,5,6,7,8],{x,\"7.77\"}}).\n{ok,<<5,56,0,8,3,55,55,55,46,69,45,50>>}\n```","ref":"asn1_getting_started.html#embedded-named-types"},{"type":"extras","title":"Naming of Records in .hrl Files - Getting Started","doc":"When the option `maps` is given, no `.hrl` files will be generated. The rest of\nthis section describes the behavior of the compiler when `maps` is not used.\n\nWhen an ASN.1 specification is compiled, all defined types of type `SET` or\n`SEQUENCE` result in a corresponding record in the generated `.hrl` file. This\nis because the values for `SET` and `SEQUENCE` are represented as records by\ndefault.\n\nSome special cases of this functionality are presented in the next section.","ref":"asn1_getting_started.html#naming-of-records-in-hrl-files"},{"type":"extras","title":"Embedded Structured Types - Getting Started","doc":"In ASN.1 it is also possible to have components that are themselves structured\ntypes. For example, it is possible to have the following:\n\n```text\nEmb ::= SEQUENCE {\n    a SEQUENCE OF OCTET STRING,\n    b SET {\n       a INTEGER,\n       b INTEGER DEFAULT 66},\n    c CHOICE {\n       a INTEGER,\n       b FooType } }\n\nFooType ::= [3] VisibleString\n```\n\nThe following records are generated because of type `Emb`:\n\n```erlang\n-record('Emb,{a, b, c}).\n-record('Emb_b',{a, b = asn1_DEFAULT}). % the embedded SET type\n```\n\nValues of type `Emb` can be assigned as follows:\n\n```erlang\nV = #'Emb'{a=[\"qqqq\",[1,2,255]],\n           b = #'Emb_b'{a=99},\n           c ={b,\"Can you see this\"}}.\n```\n\nFor an embedded type of type `SEQUENCE`/`SET` in a `SEQUENCE`/`SET`, the record\nname is extended with an underscore and the component name. If the embedded\nstructure is deeper with the `SEQUENCE`, `SET`, or `CHOICE` types in the line,\neach component name/alternative name is added to the record name.\n\nExample:\n\n```text\nSeq ::= SEQUENCE{\n    a CHOICE{\n        b SEQUENCE {\n           c  INTEGER\n        }\n    }\n}\n```\n\nThis results in the following record:\n\n```erlang\n-record('Seq_a_b',{c}).\n```\n\nIf the structured type has a component with an embedded `SEQUENCE OF`/`SET OF`\nwhich embedded type in turn is a `SEQUENCE`/`SET`, it gives a record with the\n`SEQUENCE OF`/`SET OF` addition as in the following example:\n\n```text\nSeq ::= SEQUENCE {\n    a SEQUENCE OF SEQUENCE {\n           b\n               }\n    c SET OF SEQUENCE {\n           d\n               }\n}\n```\n\nThis results in the following records:\n\n```erlang\n-record('Seq_a_SEQOF'{b}).\n-record('Seq_c_SETOF'{d}).\n```\n\nA parameterized type is to be considered as an embedded type. Each time such a\ntype is referenced, an instance of it is defined. Thus, in the following example\na record with name `'Seq_b'` is generated in the `.hrl` file and is used to hold\nvalues:\n\n```text\nSeq ::= SEQUENCE {\n    b PType{INTEGER}\n}\n\nPType{T} ::= SEQUENCE{\n    id T\n}\n```","ref":"asn1_getting_started.html#embedded-structured-types"},{"type":"extras","title":"Recursive Types - Getting Started","doc":"Types that refer to themselves are called recursive types. Example:\n\n```erlang\nRec ::= CHOICE {\n     nothing NULL,\n     something SEQUENCE {\n          a INTEGER,\n          b OCTET STRING,\n          c Rec }}\n```\n\nThis is allowed in ASN.1 and the ASN.1-to-Erlang compiler supports this\nrecursive type. A value for this type is assigned in Erlang as follows:\n\n```erlang\nV = {something,#'Rec_something'{a = 77,\n                                b = \"some octets here\",\n                                c = {nothing,'NULL'}}}.\n```","ref":"asn1_getting_started.html#recursive-types"},{"type":"extras","title":"ASN.1 Values - Getting Started","doc":"Values can be assigned to an ASN.1 type within the ASN.1 code itself, as opposed\nto the actions in the previous section where a value was assigned to an ASN.1\ntype in Erlang. The full value syntax of ASN.1 is supported and X.680 describes\nin detail how to assign values in ASN.1. A short example:\n\n```erlang\nTT ::= SEQUENCE {\n   a INTEGER,\n   b SET OF OCTET STRING }\n\ntt TT ::= {a 77,b {\"kalle\",\"kula\"}}\n```\n\nThe value defined here can be used in several ways. It can, for example, be used\nas the value in some `DEFAULT` component:\n\n```text\nSS ::= SET {\n    s OBJECT IDENTIFIER,\n    val TT DEFAULT tt }\n```\n\nIt can also be used from inside an Erlang program. If this ASN.1 code is defined\nin ASN.1 module `Values`, the ASN.1 value `tt` can be reached from Erlang as a\nfunction call to `'Values':tt()` as in the following example:\n\n```erlang\n1> Val = 'Values':tt().\n{'TT',77,[\"kalle\",\"kula\"]}\n2> {ok,Bytes} = 'Values':encode('TT',Val).\n{ok,<<48,18,128,1,77,161,13,4,5,107,97,108,108,101,4,4,\n      107,117,108,97>>}\n4> 'Values':decode('TT',Bytes).\n{ok,{'TT',77,[\"kalle\",\"kula\"]}}\n5>\n```\n\nThis example shows that a function is generated by the compiler that returns a\nvalid Erlang representation of the value, although the value is of a complex\ntype.\n\nFurthermore, if the option `maps` is not used, a macro is generated for each\nvalue in the `.hrl` file. So, the defined value `tt` can also be extracted by\n`?tt` in application code.","ref":"asn1_getting_started.html#asn-1-values"},{"type":"extras","title":"Macros - Getting Started","doc":"The type `MACRO` is not supported. It is no longer part of the ASN.1 standard.\n\n[](){: #Information-Object }","ref":"asn1_getting_started.html#macros"},{"type":"extras","title":"ASN.1 Information Objects (X.681) - Getting Started","doc":"Information Object Classes, Information Objects, and Information Object Sets (in\nthe following called classes, objects, and object sets, respectively) are\ndefined in the standard definition X.681. Only a brief explanation is given\nhere.\n\nThese constructs makes it possible to define open types, that is, values of that\ntype can be of any ASN.1 type. Also, relationships can be defined between\ndifferent types and values, as classes can hold types, values, objects, object\nsets, and other classes in their fields. A class can be defined in ASN.1 as\nfollows:\n\n```text\nGENERAL-PROCEDURE ::= CLASS {\n      &Message,\n      &Reply               OPTIONAL,\n      &Error               OPTIONAL,\n      &id          PrintableString UNIQUE\n}\nWITH SYNTAX {\n      NEW MESSAGE     &Message\n      [REPLY           &Reply]\n      [ERROR           &Error]\n      ADDRESS          &id\n}\n```\n\nAn object is an instance of a class. An object set is a set containing objects\nof a specified class. A definition can look as follows:\n\n```text\nobject1 GENERAL-PROCEDURE ::= {\n    NEW MESSAGE      PrintableString\n    ADDRESS          \"home\"\n}\n\nobject2 GENERAL-PROCEDURE ::= {\n    NEW MESSAGE INTEGER\n    ERROR INTEGER\n    ADDRESS \"remote\"\n}\n```\n\nThe object `object1` is an instance of the class `GENERAL-PROCEDURE` and has one\ntype field and one fixed type value field. The object `object2` has also an\noptional field `ERROR`, which is a type field. The field `ADDRESS` is a `UNIQUE`\nfield. Objects in an object set must have unique values in their `UNIQUE` field,\nas in `GENERAL-PROCEDURES`:\n\n```text\nGENERAL-PROCEDURES GENERAL-PROCEDURE ::= {\n    object1 | object2}\n```\n\nYou cannot encode a class, object, or object set, only refer to it when defining\nother ASN.1 entities. Typically you refer to a class as well as to object sets\nby table constraints and component relation constraints (X.682) in ASN.1 types,\nas in the following:\n\n```erlang\nStartMessage  ::= SEQUENCE {\n    msgId  GENERAL-PROCEDURE.&id  ({GENERAL-PROCEDURES}),\n    content GENERAL-PROCEDURE.&Message ({GENERAL-PROCEDURES}{@msgId}),\n    }\n```\n\nIn type `StartMessage`, the constraint following field `content` tells that in a\nvalue of type `StartMessage` the value in field `content` must come from the\nsame object that is chosen by field `msgId`.\n\nSo, the value `#'StartMessage'{msgId=\"home\",content=\"Any Printable String\"}` is\nlegal to encode as a `StartMessage` value. However, the value\n`#'StartMessage'{msgId=\"remote\", content=\"Some String\"}` is illegal as the\nconstraint in `StartMessage` tells that when you have chosen a value from a\nspecific object in object set `GENERAL-PROCEDURES` in field `msgId`, you must\nchoose a value from that same object in the content field too. In this second\ncase, it is to be any `INTEGER` value.\n\n`StartMessage` can in field `content` be encoded with a value of any type that\nan object in object set `GENERAL-PROCEDURES` has in its `NEW MESSAGE` field.\nThis field refers to a type field `&Message` in the class. Field `msgId` is\nalways encoded as a `PrintableString`, as the field refers to a fixed type in\nthe class.\n\nIn practice, object sets are usually declared to be extensible so that more\nobjects can be added to the set later. Extensibility is indicated as follows:\n\n```text\nGENERAL-PROCEDURES GENERAL-PROCEDURE ::= {\n    object1 | object2, ...}\n```\n\nWhen decoding a type that uses an extensible set constraint, it is always\npossible that the value in field `UNIQUE` is unknown (that is, the type has been\nencoded with a later version of the ASN.1 specification). The unencoded data is\nthen returned wrapped in a tuple as follows:\n\n```text\n{asn1_OPENTYPE,Binary}\n```\n\nHere `Binary` is an Erlang binary that contains the encoded data. (If option\n`legacy_erlang_types` has been given, only the binary is returned.)","ref":"asn1_getting_started.html#asn-1-information-objects-x-681"},{"type":"extras","title":"Parameterization (X.683) - Getting Started","doc":"Parameterization, which is defined in X.683, can be used when defining types,\nvalues, value sets, classes, objects, or object sets. A part of a definition can\nbe supplied as a parameter. For example, if a `Type` is used in a definition\nwith a certain purpose, you want the type name to express the intention. This\ncan be done with parameterization.\n\nWhen many types (or another ASN.1 entity) only differ in some minor cases, but\nthe structure of the types is similar, only one general type can be defined and\nthe differences can be supplied through parameters.\n\nExample of use of parameterization:\n\n```text\nGeneral{Type} ::= SEQUENCE\n{\n     number     INTEGER,\n     string     Type\n}\n\nT1 ::= General{PrintableString}\n\nT2 ::= General{BIT STRING}\n```\n\nAn example of a value that can be encoded as type `T1` is `{12,\"hello\"}`.\n\nNotice that the compiler does not generate encode/decode functions for\nparameterized types, only for the instances of the parameterized types.\nTherefore, if a file contains the types `General{}`, `T1`, and `T2` as in the\nprevious example, encode/decode functions are only generated for `T1` and `T2`.","ref":"asn1_getting_started.html#parameterization-x-683"},{"type":"extras","title":"Specialized Decodes","doc":"<!--\n%CopyrightBegin%\n\nSPDX-License-Identifier: Apache-2.0\n\nCopyright Ericsson AB 2023-2025. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# Specialized Decodes\n\n[](){: #SpecializedDecodes }\n\nWhen performance is of highest priority and one is interested in a limited part\nof the ASN.1 encoded message before deciding what to do with the rest of it, an\noption is to decode only a part of the message. This situation can be a server\nthat has to decide the addressee of a message. The addressee can be interested\nin the entire message, but the server can be a bottleneck that you want to spare\nany unnecessary load.\n\nInstead of making two _complete decodes_ (the normal case of decode), one in the\nserver and one in the addressee, it is only necessary to make one _specialized\ndecode_ (in the server) and another complete decode (in the addressee). This\nsection describes the following specialized decode functionality:\n\n- _Exclusive decode_\n- _Selected decode_\n\nThis functionality is only provided when using `BER` (option `ber`).","ref":"asn1_spec.html"},{"type":"extras","title":"Exclusive Decode - Specialized Decodes","doc":"The basic idea with exclusive decode is to specify which parts of the message\nyou want to exclude from being decoded. These parts remain encoded and are\nreturned in the value structure as binaries. The undecoded parts can be decoded\nlater by calling the `decode_part/2` function.","ref":"asn1_spec.html#exclusive-decode"},{"type":"extras","title":"Procedure - Specialized Decodes","doc":"To perform an exclusive decode:\n\n- _Step 1:_ Decide the name of the function for the exclusive decode.\n- _Step 2:_ Include the following instructions in a configuration file:\n\n  - The name of the exclusive decode function\n  - The name of the ASN.1 specification\n  - A notation that tells which parts of the message structure to be excluded\n    from decode\n\n- _Step 3_ Compile with the additional option `asn1config`. The compiler\n  searches for a configuration file with the same name as the ASN.1\n  specification but with extension `.asn1config`. This configuration file is not\n  the same as used for compilation of a set of files. See section\n  [Writing an Exclusive Decode Instruction.](asn1_spec.md#UndecodedPart)","ref":"asn1_spec.html#procedure"},{"type":"extras","title":"User Interface - Specialized Decodes","doc":"The runtime user interface for exclusive decode comprises the following two\nfunctions:\n\n- A function for an exclusive decode, whose name the user decides in the\n  configuration file\n- A `decode_part/2` function generated by the ASN.1 compiler when\n  exclusive decode is enabled. This function decodes the parts that\n  were left undecoded during the exclusive decode.\n\nBoth functions are described in the following.\n\nIf the exclusive decode function has, for example, the name `decode_exclusive`\nand an ASN.1 encoded message `Bin` is to be exclusive decoded, the call is as\nfollows:\n\n```erlang\n{ok,ExclMessage} = 'MyModule':decode_exclusive(Bin)\n```\n\n[](){: #UndecodedPart } The result `ExclMessage` has the same structure as a\ncomplete decode would have, except for the parts of the top type that were not\ndecoded. The undecoded parts are on their places in the structure on format\n`{TypeKey,UndecodedValue}`.\n\nEach undecoded part that is to be decoded must be fed into function\n`decode_part/2` as follows:\n\n```erlang\n{ok,PartMessage} = 'MyModule':decode_part(TypeKey, UndecodedValue)\n```\n\n[](){: #Exclusive-Instruction }","ref":"asn1_spec.html#user-interface"},{"type":"extras","title":"Writing an Exclusive Decode Instruction - Specialized Decodes","doc":"This instruction is written in the configuration file in the following format:\n\n```erlang\nExclusiveDecodeInstruction = {exclusive_decode,{ModuleName,DecodeInstructions}}.\n\nModuleName = atom()\n\nDecodeInstructions = [DecodeInstruction]+\n\nDecodeInstruction = {ExclusiveDecodeFunctionName,TypeList}\n\nExclusiveDecodeFunctionName = atom()\n\nTypeList = [TopType,ElementList]\n\nElementList = [Element]+\n\nElement = {Name,parts} |\n          {Name,undecoded} |\n          {Name,ElementList}\n\nTopType = atom()\n\nName = atom()\n```\n\nThe instruction must be a valid Erlang term terminated by a dot.\n\nIn `TypeList` the path from the top type to each undecoded subcomponent is\ndescribed. `TopType` is the name of a top-level type in the ASN.1 specification.\nThe action for each component in `ElementList` is described by one of:\n\n- `{Name,parts}`\n- `{Name,undecoded}`\n- `{Name,ElementList}`\n\nThe use and effect of the actions are as follows:\n\n- **`{Name,undecoded}`** - Leaves the element undecoded. The type of `Name` can\n  be any ASN.1 type. The value of element `Name` is returned as a tuple (as\n  mentioned in the previous section) in the value structure of the top type.\n\n- **`{Name,parts}`** - The type of `Name` must be either `SEQUENCE OF` or\n  `SET OF`. The action implies that the different components of `Name` are left\n  undecoded. The value of `Name` is returned as a tuple (as mentioned in the\n  previous section) where the second element is a list of binaries. This is\n  because the representation of a `SEQUENCE OF` or a `SET OF` in Erlang is a\n  list of its internal type. Any of the elements in this list or the entire list\n  can be decoded by function `decodepart`.\n\n- **`{Name,ElementList}`** - This action is used when one or more of the\n  subtypes of `Name` is exclusively decoded.\n\n`Name` in these actions can be a component name of a `SEQUENCE OF` or a\n`SET OF`, or a name of an alternative in a `CHOICE`.","ref":"asn1_spec.html#writing-an-exclusive-decode-instruction"},{"type":"extras","title":"Example - Specialized Decodes","doc":"In this examples, the definitions from the following ASN.1 specification are\nused:\n\n```text\n\nGUI DEFINITIONS AUTOMATIC TAGS ::= BEGIN\n\n  Action ::= SEQUENCE {\n    number  INTEGER DEFAULT 15,\n    handle  Handle DEFAULT {number 12, on TRUE}\n  }\n\n  Key ::= Button\n  Handle ::= Key\n\n  Button ::= SEQUENCE {\n    number INTEGER,\n    on     BOOLEAN\n  }\n\n  Window ::= CHOICE {\n    vsn INTEGER,\n    status Status\n  }\n\n  Status ::= SEQUENCE {\n    state INTEGER,\n    buttonList SEQUENCE OF Button,\n    enabled BOOLEAN OPTIONAL,\n    actions CHOICE {\n      possibleActions SEQUENCE OF Action,\n      noOfActions INTEGER\n    }\n  }\n\nEND\n```\n\n{: #Asn1spec }\n\nIf `Button` is a top type and it is needed to exclude component `number` from\ndecode, `TypeList` in the instruction in the configuration file is\n`['Button',[{number,undecoded}]]`. If you call the decode function\n`decode_Button_exclusive`, `DecodeInstruction` is\n`{decode_Button_exclusive,['Button',[{number,undecoded}]]}`.\n\nAnother top type is `Window` whose subcomponent actions in type `Status` and the\nparts of component `buttonList` are to be left undecoded. For this type, the\nfunction is named `decode__Window_exclusive`. The complete\n`Exclusive_Decode_Instruction` configuration is as follows:\n\n```text\n\n{exclusive_decode,\n {'GUI',\n  [{decode_Window_exclusive,\n    ['Window',[{status,[{buttonList,parts},{actions,undecoded}]}]]},\n   {decode_Button_exclusive,\n    ['Button',[{number,undecoded}]]}]}}.\n```\n\nThe following figure shows the bytes of a `Window:status` message. The\ncomponents `buttonList` and `actions` are excluded from decode. Only `state` and\n`enabled` are decoded when `decode__Window_exclusive` is called.\n\n![Bytes of a Window:status Message](assets/exclusive_Win_But.gif \"Bytes of a Window:status Message\")\n\nHere follows an example of how the module. Note that option `no_ok_wrapper` is\nused to make the example more concise.\n\n```erlang\n1> asn1ct:compile('GUI', [ber,asn1config,no_ok_wrapper]).\nok\n2> rr('GUI').\n['Action','Button','Status']\n3> ButtonMsg = #'Button'{number=123,on=true}.\n#'Button'{number = 123,on = true}\n4> ButtonBytes = 'GUI':encode('Button', ButtonMsg).\n<<48,6,128,1,123,129,1,255>>\n5> ExclusiveMsgButton = 'GUI':decode_Button_exclusive(ButtonBytes).\n#'Button'{number = {'Button_number',<<128,1,123>>},\n          on = true}\n6> {UndecKey,UndecBytes} = ExclusiveMsgButton#'Button'.number.\n{'Button_number',<<128,1,123>>}\n7> 'GUI':decode_part(UndecKey, UndecBytes).\n123\n8> WindowMsg =\n{status,{'Status',35,\n   [{'Button',3,true},\n    {'Button',4,false},\n    {'Button',5,true},\n    {'Button',6,true},\n    {'Button',7,false}],\n   false,\n   {possibleActions,[{'Action',16,{'Button',17,true}}]}}}.\n{status,#'Status'{state = 35,\n        buttonList = [#'Button'{number = 3,on = true},\n                      #'Button'{number = 4,on = false},\n                      #'Button'{number = 5,on = true},\n                      #'Button'{number = 6,on = true},\n                      #'Button'{number = 7,on = false}],\n        enabled = false,\n        actions = {possibleActions,[#'Action'{number = 16,\n                                              handle = #'Button'{number = 17,on = true}}]}}}\n9> WindowBytes = 'GUI':encode('Window', WindowMsg).\n<<161,65,128,1,35,161,40,48,6,128,1,3,129,1,255,48,6,128,\n  1,4,129,1,0,48,6,128,1,5,129,...>>\n10> {status,#'Status'{buttonList={UndecWindowKey,UndecWindowParts}}} =\n'GUI':decode_Window_exclusive(WindowBytes).\n{status,#'Status'{state = 35,\n                  buttonList = {'Status_buttonList',[<<48,6,128,1,3,129,1,\n                                                       255>>,\n                                                     <<48,6,128,1,4,129,1,0>>,\n                                                     <<48,6,128,1,5,129,1,255>>,\n                                                     <<48,6,128,1,6,129,1,255>>,\n                                                     <<48,6,128,1,7,129,1,0>>]},\n                  enabled = false,\n                  actions = {'Status_actions',<<163,15,160,13,48,11,128,\n                                                1,16,161,6,128,1,17,129,\n                                                1,255>>}}}\n11> 'GUI':decode_part(UndecWindowKey, UndecWindowParts).\n[#'Button'{number = 3,on = true},\n #'Button'{number = 4,on = false},\n #'Button'{number = 5,on = true},\n #'Button'{number = 6,on = true},\n #'Button'{number = 7,on = false}]\n12> 'GUI':decode_part(UndecWindowKey, hd(UndecWindowParts)).\n#'Button'{number = 3,on = true}\n13> {status,#'Status'{actions={ChoiceKey,ChoiceUndec}}} = v(10).\n{status,#'Status'{state = 35,\n                  buttonList = {'Status_buttonList',[<<48,6,128,1,3,129,1,\n                                                       255>>,\n                                                     <<48,6,128,1,4,129,1,0>>,\n                                                     <<48,6,128,1,5,129,1,255>>,\n                                                     <<48,6,128,1,6,129,1,255>>,\n                                                     <<48,6,128,1,7,129,1,0>>]},\n                  enabled = false,\n                  actions = {'Status_actions',<<163,15,160,13,48,11,128,\n                                                1,16,161,6,128,1,17,129,\n                                                1,255>>}}}\n14> 'GUI':decode_part(ChoiceKey, ChoiceUndec).\n{possibleActions,[#'Action'{number = 16,\n                            handle = #'Button'{number = 17,on = true}}]}\n```","ref":"asn1_spec.html#example"},{"type":"extras","title":"Selective Decode - Specialized Decodes","doc":"Selective decode decodes a single subtype of a constructed value. This is the\nfastest method to extract a subvalue. Selective decode is typically used when\none want to inspect, for example, a version number to be able to decide how to\nhandle the entire value.","ref":"asn1_spec.html#selective-decode"},{"type":"extras","title":"Procedure - Specialized Decodes","doc":"To perform a selective decode:\n\n- _Step 1:_ Include the following instructions in the configuration file:\n\n  - The name of the user function\n  - The name of the ASN.1 specification\n  - A notation that tells which part of the type to be decoded\n\n- _Step 2:_ Compile with the additional option `asn1config`. The compiler\n  searches for a configuration file with the same name as the ASN.1\n  specification, but with extension `.asn1config`. In the same file you can also\n  provide configuration specifications for exclusive decode. The generated\n  Erlang module has the usual functionality for encode/decode preserved and the\n  specialized decode functionality added.","ref":"asn1_spec.html#procedure-1"},{"type":"extras","title":"User Interface - Specialized Decodes","doc":"The only new user interface function is the one provided by the user in the\nconfiguration file.\n\nFor example, if the configuration file includes the specification\n`{selective_decode,{'ModuleName',[{selected_decode_Window,TypeList}]}}` do the\nselective decode by\n`{ok,Result} = 'ModuleName':selected_decode_Window(EncodedBinary).`\n\n[](){: #Selective-Instruction }","ref":"asn1_spec.html#user-interface-1"},{"type":"extras","title":"Writing a Selective Decode Instruction - Specialized Decodes","doc":"One or more selective decode functions can be described in a configuration file.\nUse the following notation:\n\n```erlang\nSelectiveDecodeInstruction = {selective_decode,{ModuleName,DecodeInstructions}}.\n\nModuleName = atom()\n\nDecodeInstructions = [DecodeInstruction]+\n\nDecodeInstruction = {SelectiveDecodeFunctionName,TypeList}\n\nSelectiveDecodeFunctionName = atom()\n\nTypeList = [TopType|ElementList]\n\nElementList = Name|ListSelector\n\nName = atom()\n\nListSelector = [integer()]\n```\n\nThe instruction must be a valid Erlang term terminated by a dot.\n\n- `ModuleName` is the same as the name of the ASN.1 specification, but without\n  the extension.\n- `DecodeInstruction` is a tuple with your chosen function name and the\n  components from the top type that leads to the single type you want to decode.\n  Make sure to choose a name of your function that is not the same as any of the\n  generated functions.\n- The first element of `TypeList` is the top type of the encoded message. In\n  `ElementList`, it is followed by each of the component names that leads to\n  selected type.\n- Each name in `ElementList` must be a constructed type except the last name,\n  which can be any type.\n- `ListSelector` makes it possible to choose one of the encoded components in a\n  `SEQUENCE OF` or a `SET OF`. It is also possible to go further in that\n  component and pick a subtype of that to decode. So, in the `TypeList`:\n  `['Window',status,buttonList,[1],number]`, component `buttonList` must be of\n  type `SEQUENCE OF` or `SET OF`.\n\nIn the example, component `number` of the first of the encoded elements in the\n`SEQUENCE OF` `buttonList` is selected. This applies on the ASN.1 specification\nin section [Writing an Exclusive Decode Instruction](asn1_spec.md#Asn1spec).","ref":"asn1_spec.html#writing-a-selective-decode-instruction"},{"type":"extras","title":"Example - Specialized Decodes","doc":"In this example, the same ASN.1 specification as in section\n[Writing an Exclusive Decode Instruction](asn1_spec.md#Asn1spec) is used. The\nfollowing is a valid selective decode instruction:\n\n```erlang\n{selective_decode,\n    {'GUI',\n        [{selected_decode_Window1,\n            ['Window',status,buttonList,\n             [1],\n             number]},\n {selected_decode_Action,\n     ['Action',handle,number]},\n {selected_decode_Window2,\n     ['Window',\n      status,\n      actions,\n      possibleActions,\n      [1],\n      handle,number]}]}}.\n```\n\nThe first instruction,\n`{selected_decode_Window1,['Window',status,buttonList,[1],number]}` is described\nin the previous section.\n\nThe second instruction, `{selected_decode_Action,['Action',handle,number]}`,\ntakes component `number` in the `handle` component of type `Action`. If the\nvalue is `ValAction = {'Action',17,{'Button',4711,false}}`, the internal value\n4711 is to be picked by `selected_decode_Action`. In an Erlang terminal it looks\nas follows:\n\n```erlang\n1> asn1ct:compile('GUI', [ber,asn1config,no_ok_wrapper]).\nok\n2> ValAction = {'Action',17,{'Button',4711,false}}.\n{'Action',17,{'Button',4711,false}}\n3> Bytes = 'GUI':encode('Action',ValAction).\n<<48,18,2,1,17,160,13,172,11,171,9,48,7,128,2,18,103,129,1,0>>\n4> 'GUI':selected_decode_Action(Bytes).\n4711\n```\n\nThe third instruction,\n`['Window',status,actions,possibleActions,[1],handle,number]`, works as follows:\n\n- _Step 1:_ Starts with type `Window`.\n- _Step 2:_ Takes component `status` of `Window` that is of type `Status`.\n- _Step 3:_ Takes _actions_ of type `Status`.\n- _Step 4:_ Takes `possibleActions` of the internally defined `CHOICE` type.\n- _Step 5:_ Goes into the first component of `SEQUENCE OF` by `[1]`. That\n  component is of type `Action`.\n- _Step 6:_ Takes component `handle`.\n- _Step 7:_ Takes component `number` of type `Button`.\n\nThe following figure shows which components are in `TypeList`\n`['Window',status,actions,possibleActions,[1],handle,number]`:\n\n![Elements Specified in Configuration File for Selective Decode of a Subvalue in a Window Message](assets/selective_TypeList.gif \"Elements Specified in Configuration File for Selective Decode of a Subvalue in a Window Message\")\n\nIn the following figure, only the marked element is decoded by\n`selected_decode_Window2`:\n\n![Bytes of a Window:status Message](assets/selective_Window2.gif \"Bytes of a Window:status Message\")\n\nWith the following example, you can examine that both `selected_decode_Window2`\nand `selected_decode_Window1` decodes the intended subvalue of value `Val`:\n\n```erlang\n1> Val = {status,{'Status',12,\n                    [{'Button',13,true},\n                     {'Button',14,false},\n                     {'Button',15,true},\n                     {'Button',16,false}],\n                    true,\n                    {possibleActions,[{'Action',17,{'Button',18,false}},\n                                      {'Action',19,{'Button',20,true}},\n                                      {'Action',21,{'Button',22,false}}]}}}.\n2> Bin = 'GUI':encode('Window',Val).\n<<161,89,128,1,12,161,32,48,6,128,1,13,129,1,255,48,6,128,\n  1,14,129,1,0,48,6,128,1,15,129,...>>\n4> 'GUI':selected_decode_Window1(Bin).\n13\n5> 'GUI':selected_decode_Window2(Bin).\n18\n```","ref":"asn1_spec.html#example-1"}],"proglang":"erlang","content_type":"text/markdown","producer":{"name":"ex_doc","version":"0.38.1"}}