# -*- coding: utf-8 -*- #
# frozen_string_literal: true

require 'open-uri'

MATHEMATICA_SYNTAX_URI = "https://reference.wolfram.com/language/guide/AlphabeticalListing.html"
MATHEMATICA_KEYWORDS_FILE = "./lib/rouge/lexers/mathematica/keywords.rb"

namespace :builtins do
  task :mathematica do
    generator = Rouge::Tasks::Builtins::Mathematica.new

    input    = URI.open(MATHEMATICA_SYNTAX_URI) { |f| f.read }
    keywords = generator.extract_keywords(input)

    output = generator.render_output(keywords)

    File.write(MATHEMATICA_KEYWORDS_FILE, output)
  end
end

module Rouge
  module Tasks
    module Builtins
      class Mathematica
        def extract_keywords(input)
          keywords = Array.new

          input.scrub.scan %r(<span class="IFSans"><a href="/language/ref/(\w+)[.]html">)m do |m|
            keywords.push m[0]
          end

          keywords
        end

        def render_output(keywords, &b)
          return enum_for(:render_output, keywords).to_a.join("\n") unless b

          yield   "# -*- coding: utf-8 -*- #"
          yield   "# frozen_string_literal: true"
          yield   ""
          yield   "# DO NOT EDIT"
          yield   ""
          yield   "# This file is automatically generated by `rake builtins:mathematica`."
          yield   "# See tasks/builtins/mathematica.rake for more info."
          yield   ""
          yield   "module Rouge"
          yield   "  module Lexers"
          yield   "    class Mathematica"
          yield   "      def self.builtins"
          yield   "        @builtins ||= Set.new #{keywords.inspect}"
          yield   "      end"
          yield   "    end"
          yield   "  end"
          yield   "end"
        end
      end
    end
  end
end
