/*
Copyright 2023 The OpenVEX Authors
SPDX-License-Identifier: Apache-2.0
*/

package vex

// Vulnerability is a struct that captures the vulnerability identifier and
// its aliases. When defined, the ID field should be an IRI.
type Vulnerability struct {
	//  ID is an IRI to reference the vulnerability in the statement.
	ID string `json:"@id,omitempty"`

	// Name is the main vulnerability identifier.
	Name VulnerabilityID `json:"name,omitempty"`

	// Description is a short free form text description of the vulnerability.
	Description string `json:"description,omitempty"`

	// Aliases is a list of other vulnerability identifier strings that
	// locate the vulnerability in other tracking systems.
	Aliases []VulnerabilityID `json:"aliases,omitempty"`
}

// VulnerabilityID is a string that captures a vulnerability identifier. It is
// a free form string but it is intended to capture the identifiers used by
// tracking systems.
type VulnerabilityID string

// Matches returns true if the vulnerability's name or aliases matches the
// identifier string.
func (v *Vulnerability) Matches(identifier string) bool {
	// Identifier is an IRI, not te vuln ID. To match against the vuln.ID
	// we should normalize the identifier to the full IRI (ie de-localize)
	if v.ID == identifier {
		return true
	}
	if string(v.Name) == identifier {
		return true
	}
	for _, id := range v.Aliases {
		if id == VulnerabilityID(identifier) {
			return true
		}
	}
	return false
}
