//go:build !windows

package updater

import (
	"context"
	"fmt"
	"os"
	"path/filepath"
	"testing"

	"github.com/gopasspw/gopass/internal/config"
	"github.com/stretchr/testify/require"
)

//nolint:wrapcheck
func TestIsUpdateable(t *testing.T) {
	ctx := config.NewContextInMemory()
	oldExec := executable

	defer func() {
		executable = oldExec
	}()

	td := t.TempDir()

	for _, tc := range []struct {
		name string
		pre  func() error
		exec func(context.Context) (string, error)
		post func() error
		ok   bool
	}{
		{
			name: "executable error",
			exec: func(context.Context) (string, error) {
				return "", fmt.Errorf("failed") //nolint:goerr113
			},
		},
		{
			name: "test binary",
			exec: func(context.Context) (string, error) {
				return "action.test", nil
			},
			ok: true,
		},
		{
			name: "force update",
			pre: func() error {
				t.Setenv("GOPASS_FORCE_UPDATE", "true")

				return nil
			},
			exec: func(context.Context) (string, error) {
				return "", nil
			},
			post: func() error {
				return os.Unsetenv("GOPASS_FORCE_UPDATE")
			},
			ok: true,
		},
		{
			name: "update in gopath",
			pre: func() error {
				t.Setenv("GOPATH", "/tmp/foo")

				return nil
			},
			exec: func(context.Context) (string, error) {
				return "/tmp/foo/gopass", nil
			},
		},
		{
			name: "stat error",
			exec: func(context.Context) (string, error) {
				return "/tmp/foo/gopass", nil
			},
		},
		{
			name: "no regular file",
			exec: func(context.Context) (string, error) {
				return td, nil
			},
		},
		{
			name: "no write access to file",
			pre: func() error {
				return os.WriteFile(filepath.Join(td, "gopass"), []byte("foobar"), 0o555)
			},
			exec: func(context.Context) (string, error) {
				return filepath.Join(td, "gopass"), nil
			},
		},
		{
			name: "no write access to dir",
			pre: func() error {
				dir := filepath.Join(td, "bin")

				return os.Mkdir(dir, 0o555)
			},
			exec: func(context.Context) (string, error) {
				return filepath.Join(td, "bin"), nil
			},
		},
	} {
		if tc.pre != nil {
			require.NoError(t, tc.pre(), tc.name)
		}

		executable = tc.exec

		err := IsUpdateable(ctx)
		if tc.ok {
			require.NoError(t, err, tc.name)
		} else {
			require.Error(t, err, tc.name)
		}

		if tc.post != nil {
			require.NoError(t, tc.post(), tc.name)
		}
	}
}
