// SPDX-FileCopyrightText: Copyright 2015-2025 go-swagger maintainers
// SPDX-License-Identifier: Apache-2.0

package runtime

import (
	"bytes"
	"errors"
	"fmt"
	"net/http/httptest"
	"testing"

	"github.com/go-openapi/testify/v2/assert"
	"github.com/go-openapi/testify/v2/require"
)

var consProdText = `The quick brown fox jumped over the lazy dog.`

func TestTextConsumer(t *testing.T) {
	cons := TextConsumer()

	// can consume as a string
	var str string
	err1 := cons.Consume(bytes.NewBufferString(consProdText), &str)
	require.NoError(t, err1)
	assert.Equal(t, consProdText, str)

	var tu textUnmarshalDummy

	// can consume as a TextUnmarshaler
	err3 := cons.Consume(bytes.NewBufferString(consProdText), &tu)
	require.NoError(t, err3)
	assert.Equal(t, consProdText, tu.str)

	// text unmarshal objects can return an error as well, this will be propagated
	require.NoError(t, cons.Consume(bytes.NewBuffer(nil), &tu))

	// when readers can't be read, those errors will be propogated as well
	require.Error(t, cons.Consume(new(nopReader), &tu))

	// readers can also not be nil
	require.Error(t, cons.Consume(nil, &tu))

	// can't consume nil ptr's or unsupported types
	require.Error(t, cons.Consume(bytes.NewBufferString(consProdText), nil))
	require.Error(t, cons.Consume(bytes.NewBufferString(consProdText), 42))
	require.Error(t, cons.Consume(bytes.NewBufferString(consProdText), &struct{}{}))
}

type textUnmarshalDummy struct {
	str string
}

func (t *textUnmarshalDummy) UnmarshalText(b []byte) error {
	if len(b) == 0 {
		return errors.New("no text given")
	}

	t.str = string(b)
	return nil
}

type nopReader struct{}

func (n *nopReader) Read(_ []byte) (int, error) {
	return 0, errors.New("nop")
}

func TestTextProducer(t *testing.T) {
	prod := TextProducer()
	rw := httptest.NewRecorder()
	err := prod.Produce(rw, consProdText)
	require.NoError(t, err)
	assert.Equal(t, consProdText, rw.Body.String())
	rw2 := httptest.NewRecorder()
	err2 := prod.Produce(rw2, &consProdText)
	require.NoError(t, err2)
	assert.Equal(t, consProdText, rw2.Body.String())

	// should always work with type aliases
	// as an alias is sometimes given by generated go-swagger code
	type alias string
	aliasProdText := alias(consProdText)
	rw3 := httptest.NewRecorder()
	err3 := prod.Produce(rw3, aliasProdText)
	require.NoError(t, err3)
	assert.Equal(t, consProdText, rw3.Body.String())
	rw4 := httptest.NewRecorder()
	err4 := prod.Produce(rw4, &aliasProdText)
	require.NoError(t, err4)
	assert.Equal(t, consProdText, rw4.Body.String())

	const answer = "42"

	// Should always work with objects implementing Stringer interface
	rw5 := httptest.NewRecorder()
	err5 := prod.Produce(rw5, &stringerDummy{answer})
	require.NoError(t, err5)
	assert.Equal(t, answer, rw5.Body.String())

	// Should always work with objects implementing TextMarshaler interface
	rw6 := httptest.NewRecorder()
	err6 := prod.Produce(rw6, &textMarshalDummy{answer})
	require.NoError(t, err6)
	assert.Equal(t, answer, rw6.Body.String())

	rw10 := httptest.NewRecorder()
	err10 := prod.Produce(rw10, errors.New(answer))
	require.NoError(t, err10)
	assert.Equal(t, answer, rw10.Body.String())

	rw11 := httptest.NewRecorder()
	err11 := prod.Produce(rw11, Error{Message: answer})
	require.NoError(t, err11)
	assert.Equal(t, fmt.Sprintf(`{"message":%q}`, answer), rw11.Body.String())

	rw12 := httptest.NewRecorder()
	err12 := prod.Produce(rw12, &Error{Message: answer})
	require.NoError(t, err12)
	assert.Equal(t, fmt.Sprintf(`{"message":%q}`, answer), rw12.Body.String())

	rw13 := httptest.NewRecorder()
	err13 := prod.Produce(rw13, []string{answer})
	require.NoError(t, err13)
	assert.Equal(t, fmt.Sprintf(`[%q]`, answer), rw13.Body.String())

	// should not work with anything that's not (indirectly) a string
	rw7 := httptest.NewRecorder()
	err7 := prod.Produce(rw7, 42)
	require.Error(t, err7)
	// nil values should also be safely caught with an error
	rw8 := httptest.NewRecorder()
	err8 := prod.Produce(rw8, nil)
	require.Error(t, err8)

	// writer can not be nil
	require.Error(t, prod.Produce(nil, &textMarshalDummy{answer}))

	// should not work for a textMarshaler that returns an error during marshalling
	rw9 := httptest.NewRecorder()
	err9 := prod.Produce(rw9, new(textMarshalDummy))
	require.Error(t, err9)
}

type Error struct {
	Message string `json:"message"`
}

type stringerDummy struct {
	str string
}

func (t *stringerDummy) String() string {
	return t.str
}

type textMarshalDummy struct {
	str string
}

func (t *textMarshalDummy) MarshalText() ([]byte, error) {
	if t.str == "" {
		return nil, errors.New("no text set")
	}
	return []byte(t.str), nil
}
