package bind9config

// Checks if the view contains no-parse directives.
func (v *View) HasNoParse() bool {
	for _, clause := range v.Clauses {
		switch {
		case clause.NoParse != nil:
			return true
		case clause.Zone != nil:
			return clause.Zone.HasNoParse()
		}
	}
	return false
}

// Returns the allow-transfer clause for the view or nil if it is not found.
func (v *View) GetAllowTransfer() *AllowTransfer {
	for _, clause := range v.Clauses {
		if clause.AllowTransfer != nil {
			return clause.AllowTransfer
		}
	}
	return nil
}

// Returns the match-clients clause for the view or nil if it is not found.
func (v *View) GetMatchClients() *MatchClients {
	for _, clause := range v.Clauses {
		if clause.MatchClients != nil {
			return clause.MatchClients
		}
	}
	return nil
}

// Returns the response-policy clause for the view or nil if it is not found.
// The result of calling this function is cached because it can be accessed
// frequently (for each zone returned to the server).
func (v *View) GetResponsePolicy() *ResponsePolicy {
	v.responsePolicyOnce.Do(func() {
		// Return the cached response-policy clause if it was already accessed.
		if v.responsePolicy != nil {
			return
		}
		for _, clause := range v.Clauses {
			if clause.ResponsePolicy != nil {
				v.responsePolicy = clause.ResponsePolicy
				return
			}
		}
	})
	return v.responsePolicy
}

// Returns the zone with the specified name or nil if the zone is not found.
func (v *View) GetZone(zoneName string) *Zone {
	for _, clause := range v.Clauses {
		if clause.Zone != nil && clause.Zone.Name == zoneName {
			return clause.Zone
		}
	}
	return nil
}
