/*
  This file is part of TALER
  (C) 2025 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify it under the
  terms of the GNU Affero General Public License as published by the Free Software
  Foundation; either version 3, or (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

  You should have received a copy of the GNU General Public License along with
  TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
*/
/**
 * @file taler-merchant-httpd_private-post-units.c
 * @brief implement POST /private/units
 * @author Bohdan Potuzhnyi
 */
#include "platform.h"
#include "taler-merchant-httpd_private-post-units.h"
#include "taler-merchant-httpd_helper.h"
#include <taler/taler_json_lib.h>


MHD_RESULT
TMH_private_post_units (const struct TMH_RequestHandler *rh,
                        struct MHD_Connection *connection,
                        struct TMH_HandlerContext *hc)
{
  struct TMH_MerchantInstance *mi = hc->instance;
  struct TALER_MERCHANTDB_UnitDetails nud = { 0 };
  bool allow_fraction_missing = true;
  bool unit_precision_missing = true;
  bool unit_active_missing = true;
  enum GNUNET_GenericReturnValue res;
  enum GNUNET_DB_QueryStatus qs;
  MHD_RESULT ret;
  struct GNUNET_JSON_Specification spec[] = {
    GNUNET_JSON_spec_string ("unit",
                             (const char **) &nud.unit),
    GNUNET_JSON_spec_string ("unit_name_long",
                             (const char **) &nud.unit_name_long),
    GNUNET_JSON_spec_string ("unit_name_short",
                             (const char **) &nud.unit_name_short),
    GNUNET_JSON_spec_mark_optional (
      GNUNET_JSON_spec_json ("unit_name_long_i18n",
                             &nud.unit_name_long_i18n),
      NULL),
    GNUNET_JSON_spec_mark_optional (
      GNUNET_JSON_spec_json ("unit_name_short_i18n",
                             &nud.unit_name_short_i18n),
      NULL),
    GNUNET_JSON_spec_mark_optional (
      GNUNET_JSON_spec_bool ("unit_allow_fraction",
                             &nud.unit_allow_fraction),
      &allow_fraction_missing),
    GNUNET_JSON_spec_mark_optional (
      GNUNET_JSON_spec_uint32 ("unit_precision_level",
                               &nud.unit_precision_level),
      &unit_precision_missing),
    GNUNET_JSON_spec_mark_optional (
      GNUNET_JSON_spec_bool ("unit_active",
                             &nud.unit_active),
      &unit_active_missing),
    GNUNET_JSON_spec_end ()
  };


  GNUNET_assert (NULL != mi);
  res = TALER_MHD_parse_json_data (connection,
                                   hc->request_body,
                                   spec);
  (void) rh;

  if (GNUNET_OK != res)
  {
    GNUNET_break_op (0);
    return (GNUNET_NO == res)
           ? MHD_YES
           : MHD_NO;
  }

  if (allow_fraction_missing)
  {
    nud.unit_allow_fraction = false;
    nud.unit_precision_level = 0;
  }
  else
  {
    if (! nud.unit_allow_fraction)
    {
      nud.unit_precision_level = 0;
      unit_precision_missing = false;
    }
    else if (unit_precision_missing)
    {
      nud.unit_precision_level = 0;
    }
  }
  if (nud.unit_precision_level > TMH_MAX_FRACTIONAL_PRECISION_LEVEL)
  {
    GNUNET_break_op (0);
    ret = TALER_MHD_reply_with_error (connection,
                                      MHD_HTTP_BAD_REQUEST,
                                      TALER_EC_GENERIC_PARAMETER_MALFORMED,
                                      "unit_precision_level");
    goto cleanup;
  }
  if (unit_active_missing)
    nud.unit_active = true;

  if (NULL == nud.unit_name_long_i18n)
    nud.unit_name_long_i18n = json_object ();
  if (NULL == nud.unit_name_short_i18n)
    nud.unit_name_short_i18n = json_object ();

  if (! TALER_JSON_check_i18n (nud.unit_name_long_i18n))
  {
    GNUNET_break_op (0);
    ret = TALER_MHD_reply_with_error (connection,
                                      MHD_HTTP_BAD_REQUEST,
                                      TALER_EC_GENERIC_PARAMETER_MALFORMED,
                                      "unit_name_long_i18n");
    goto cleanup;
  }
  if (! TALER_JSON_check_i18n (nud.unit_name_short_i18n))
  {
    GNUNET_break_op (0);
    ret = TALER_MHD_reply_with_error (connection,
                                      MHD_HTTP_BAD_REQUEST,
                                      TALER_EC_GENERIC_PARAMETER_MALFORMED,
                                      "unit_name_short_i18n");
    goto cleanup;
  }

  nud.unit_builtin = false;

  {
    bool no_instance = false;
    bool conflict = false;
    uint64_t unit_serial = 0;

    qs = TMH_db->insert_unit (TMH_db->cls,
                              mi->settings.id,
                              &nud,
                              &no_instance,
                              &conflict,
                              &unit_serial);

    switch (qs)
    {
    case GNUNET_DB_STATUS_HARD_ERROR:
      GNUNET_break (0);
      ret = TALER_MHD_reply_with_error (connection,
                                        MHD_HTTP_INTERNAL_SERVER_ERROR,
                                        TALER_EC_GENERIC_DB_STORE_FAILED,
                                        NULL);
      goto cleanup;
    case GNUNET_DB_STATUS_SOFT_ERROR:
      GNUNET_break (0);
      ret = TALER_MHD_reply_with_error (connection,
                                        MHD_HTTP_INTERNAL_SERVER_ERROR,
                                        TALER_EC_GENERIC_DB_SOFT_FAILURE,
                                        NULL);
      goto cleanup;
    case GNUNET_DB_STATUS_SUCCESS_NO_RESULTS:
      GNUNET_break (0);
      ret = TALER_MHD_reply_with_error (connection,
                                        MHD_HTTP_INTERNAL_SERVER_ERROR,
                                        TALER_EC_GENERIC_DB_INVARIANT_FAILURE,
                                        "insert_unit");
      goto cleanup;
    case GNUNET_DB_STATUS_SUCCESS_ONE_RESULT:
      break;
    }

    if (no_instance)
    {
      ret = TALER_MHD_reply_with_error (connection,
                                        MHD_HTTP_NOT_FOUND,
                                        TALER_EC_MERCHANT_GENERIC_INSTANCE_UNKNOWN,
                                        mi->settings.id);
      goto cleanup;
    }
    if (conflict)
    {
      ret = TALER_MHD_reply_with_error (connection,
                                        MHD_HTTP_CONFLICT,
                                        TALER_EC_MERCHANT_GENERIC_UNIT_BUILTIN,
                                        nud.unit);
      goto cleanup;
    }

    ret = TALER_MHD_reply_static (connection,
                                  MHD_HTTP_NO_CONTENT,
                                  NULL,
                                  NULL,
                                  0);
  }

cleanup:
  if (NULL != nud.unit_name_long_i18n)
  {
    json_decref (nud.unit_name_long_i18n);
    nud.unit_name_long_i18n = NULL;
  }
  if (NULL != nud.unit_name_short_i18n)
  {
    json_decref (nud.unit_name_short_i18n);
    nud.unit_name_short_i18n = NULL;
  }
  GNUNET_JSON_parse_free (spec);
  return ret;
}


/* end of taler-merchant-httpd_private-post-units.c */
