/*
  This file is part of TALER
  Copyright (C) 2014-2023 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as
  published by the Free Software Foundation; either version 3, or
  (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public
  License along with TALER; see the file COPYING.  If not, see
  <http://www.gnu.org/licenses/>
*/
/**
 * @file testing/testing_api_cmd_recoup.c
 * @brief Implement the /recoup test command.
 * @author Marcello Stanisci
 */
#include "taler/platform.h"
#include "taler/taler_json_lib.h"
#include <gnunet/gnunet_curl_lib.h>
#include "taler/taler_testing_lib.h"


/**
 * State for a "pay back" CMD.
 */
struct RecoupState
{
  /**
   * Expected HTTP status code.
   */
  unsigned int expected_response_code;

  /**
   * Command that offers a reserve private key,
   * plus a coin to be paid back.
   */
  const char *coin_reference;

  /**
   * The interpreter state.
   */
  struct TALER_TESTING_Interpreter *is;

  /**
   * Handle to the ongoing operation.
   */
  struct TALER_EXCHANGE_RecoupHandle *ph;

  /**
   * If the recoup filled a reserve, this is set to the reserve's public key.
   */
  struct TALER_ReservePublicKeyP reserve_pub;

  /**
   * Entry in the coin's history generated by this operation.
   */
  struct TALER_EXCHANGE_CoinHistoryEntry che;

  /**
   * Public key of the refunded coin.
   */
  struct TALER_CoinSpendPublicKeyP coin;

  /**
   * Reserve history entry, set if this recoup actually filled up a reserve.
   * Otherwise `reserve_history.type` will be zero.
   */
  struct TALER_EXCHANGE_ReserveHistoryEntry reserve_history;

};


/**
 * Check the result of the recoup request: checks whether
 * the HTTP response code is good, and that the coin that
 * was paid back belonged to the right reserve.
 *
 * @param cls closure
 * @param rr response details
 */
static void
recoup_cb (void *cls,
           const struct TALER_EXCHANGE_RecoupResponse *rr)
{
  struct RecoupState *ps = cls;
  const struct TALER_EXCHANGE_HttpResponse *hr = &rr->hr;
  struct TALER_TESTING_Interpreter *is = ps->is;
  const struct TALER_TESTING_Command *reserve_cmd;
  char *cref;
  unsigned int idx;

  ps->ph = NULL;
  if (ps->expected_response_code != hr->http_status)
  {
    TALER_TESTING_unexpected_status (is,
                                     hr->http_status,
                                     ps->expected_response_code);
    return;
  }

  if (GNUNET_OK !=
      TALER_TESTING_parse_coin_reference (
        ps->coin_reference,
        &cref,
        &idx))
  {
    TALER_TESTING_interpreter_fail (is);
    return;
  }
  (void) idx; /* do NOT use! We ignore 'idx', must be 0 for melt! */

  reserve_cmd = TALER_TESTING_interpreter_lookup_command (is,
                                                          cref);
  GNUNET_free (cref);

  if (NULL == reserve_cmd)
  {
    GNUNET_break (0);
    TALER_TESTING_interpreter_fail (is);
    return;
  }

  switch (hr->http_status)
  {
  case MHD_HTTP_OK:
    /* check old_coin_pub or reserve_pub, respectively */
    {
      const struct TALER_ReservePrivateKeyP *reserve_priv;

      if (GNUNET_OK !=
          TALER_TESTING_get_trait_reserve_priv (reserve_cmd,
                                                &reserve_priv))
      {
        GNUNET_break (0);
        TALER_TESTING_interpreter_fail (is);
        return;
      }
      GNUNET_CRYPTO_eddsa_key_get_public (&reserve_priv->eddsa_priv,
                                          &ps->reserve_pub.eddsa_pub);
      if (0 != GNUNET_memcmp (&rr->details.ok.reserve_pub,
                              &ps->reserve_pub))
      {
        GNUNET_break (0);
        TALER_TESTING_interpreter_fail (is);
        return;
      }
      if (GNUNET_OK ==
          TALER_amount_is_valid (&ps->reserve_history.amount))
        ps->reserve_history.type = TALER_EXCHANGE_RTT_RECOUP;
      /* ps->reserve_history.details.recoup_details.coin_pub; // initialized earlier */
      ps->che.details.recoup.reserve_pub = ps->reserve_pub;
    }
    break;
  case MHD_HTTP_NOT_FOUND:
    break;
  case MHD_HTTP_CONFLICT:
    break;
  default:
    GNUNET_log (GNUNET_ERROR_TYPE_WARNING,
                "Unmanaged HTTP status code %u/%d.\n",
                hr->http_status,
                (int) hr->ec);
    break;
  }
  TALER_TESTING_interpreter_next (is);
}


/**
 * Run the command.
 *
 * @param cls closure.
 * @param cmd the command to execute.
 * @param is the interpreter state.
 */
static void
recoup_run (void *cls,
            const struct TALER_TESTING_Command *cmd,
            struct TALER_TESTING_Interpreter *is)
{
  struct RecoupState *ps = cls;
  const struct TALER_TESTING_Command *coin_cmd;
  const struct TALER_CoinSpendPrivateKeyP *coin_priv;
  const struct TALER_EXCHANGE_DenomPublicKey *denom_pub;
  const struct TALER_DenominationSignature *coin_sig;
  const struct TALER_WithdrawMasterSeedP *seed;
  const struct TALER_HashBlindedPlanchetsP *h_planchets;
  struct TALER_PlanchetMasterSecretP secret;
  char *cref;
  unsigned int idx;
  const struct TALER_ExchangeBlindingValues *ewv;
  struct TALER_DenominationHashP h_denom_pub;

  ps->is = is;
  if (GNUNET_OK !=
      TALER_TESTING_parse_coin_reference (
        ps->coin_reference,
        &cref,
        &idx))
  {
    TALER_TESTING_interpreter_fail (is);
    return;
  }

  coin_cmd = TALER_TESTING_interpreter_lookup_command (is,
                                                       cref);
  GNUNET_free (cref);

  if (NULL == coin_cmd)
  {
    GNUNET_break (0);
    TALER_TESTING_interpreter_fail (is);
    return;
  }
  if (GNUNET_OK !=
      TALER_TESTING_get_trait_coin_priv (coin_cmd,
                                         idx,
                                         &coin_priv))
  {
    GNUNET_break (0);
    TALER_TESTING_interpreter_fail (is);
    return;
  }
  GNUNET_CRYPTO_eddsa_key_get_public (&coin_priv->eddsa_priv,
                                      &ps->coin.eddsa_pub);
  if (GNUNET_OK !=
      TALER_TESTING_get_trait_exchange_blinding_values (coin_cmd,
                                                        idx,
                                                        &ewv))
  {
    GNUNET_break (0);
    TALER_TESTING_interpreter_fail (is);
    return;
  }
  if (GNUNET_OK !=
      TALER_TESTING_get_trait_withdraw_seed (coin_cmd,
                                             &seed))
  {
    GNUNET_break (0);
    TALER_TESTING_interpreter_fail (is);
    return;
  }
  GNUNET_CRYPTO_eddsa_key_get_public (
    &coin_priv->eddsa_priv,
    &ps->reserve_history.details.recoup_details.coin_pub.eddsa_pub);

  if (GNUNET_OK !=
      TALER_TESTING_get_trait_denom_pub (coin_cmd,
                                         idx,
                                         &denom_pub))
  {
    GNUNET_break (0);
    TALER_TESTING_interpreter_fail (is);
    return;
  }
  if (GNUNET_OK !=
      TALER_TESTING_get_trait_denom_sig (coin_cmd,
                                         idx,
                                         &coin_sig))
  {
    GNUNET_break (0);
    TALER_TESTING_interpreter_fail (is);
    return;
  }
  if (GNUNET_OK !=
      TALER_TESTING_get_trait_withdraw_commitment (coin_cmd,
                                                   &h_planchets))
  {
    GNUNET_break (0);
    TALER_TESTING_interpreter_fail (is);
    return;
  }
  GNUNET_log (GNUNET_ERROR_TYPE_INFO,
              "Trying to recoup denomination '%s'\n",
              TALER_B2S (&denom_pub->h_key));
  ps->che.type = TALER_EXCHANGE_CTT_RECOUP;
  ps->che.amount = ps->reserve_history.amount;
  TALER_withdraw_expand_secrets (1,
                                 seed,
                                 &secret);
  TALER_planchet_blinding_secret_create (&secret,
                                         ewv,
                                         &ps->che.details.recoup.coin_bks);
  TALER_denom_pub_hash (&denom_pub->key,
                        &h_denom_pub);
  TALER_wallet_recoup_sign (&h_denom_pub,
                            &ps->che.details.recoup.coin_bks,
                            coin_priv,
                            &ps->che.details.recoup.coin_sig);
  ps->ph = TALER_EXCHANGE_recoup (
    TALER_TESTING_interpreter_get_context (is),
    TALER_TESTING_get_exchange_url (is),
    TALER_TESTING_get_keys (is),
    denom_pub,
    coin_sig,
    ewv,
    &secret,
    h_planchets,
    &recoup_cb,
    ps);
  GNUNET_assert (NULL != ps->ph);
}


/**
 * Cleanup the "recoup" CMD state, and possibly cancel
 * a pending operation thereof.
 *
 * @param cls closure.
 * @param cmd the command which is being cleaned up.
 */
static void
recoup_cleanup (void *cls,
                const struct TALER_TESTING_Command *cmd)
{
  struct RecoupState *ps = cls;
  if (NULL != ps->ph)
  {
    TALER_EXCHANGE_recoup_cancel (ps->ph);
    ps->ph = NULL;
  }
  GNUNET_free (ps);
}


/**
 * Offer internal data from a "recoup" CMD state to other
 * commands.
 *
 * @param cls closure
 * @param[out] ret result (could be anything)
 * @param trait name of the trait
 * @param index index number of the object to offer.
 * @return #GNUNET_OK on success
 */
static enum GNUNET_GenericReturnValue
recoup_traits (void *cls,
               const void **ret,
               const char *trait,
               unsigned int index)
{
  struct RecoupState *ps = cls;

  if (ps->reserve_history.type != TALER_EXCHANGE_RTT_RECOUP)
    return GNUNET_SYSERR; /* no traits */
  {
    struct TALER_TESTING_Trait traits[] = {
      TALER_TESTING_make_trait_reserve_pub (&ps->reserve_pub),
      TALER_TESTING_make_trait_reserve_history (0,
                                                &ps->reserve_history),
      TALER_TESTING_make_trait_coin_history (0,
                                             &ps->che),
      TALER_TESTING_make_trait_coin_pub (0,
                                         &ps->coin),
      TALER_TESTING_trait_end ()
    };

    return TALER_TESTING_get_trait (traits,
                                    ret,
                                    trait,
                                    index);
  }
}


struct TALER_TESTING_Command
TALER_TESTING_cmd_recoup (const char *label,
                          unsigned int expected_response_code,
                          const char *coin_reference,
                          const char *amount)
{
  struct RecoupState *ps;

  ps = GNUNET_new (struct RecoupState);
  ps->expected_response_code = expected_response_code;
  ps->coin_reference = coin_reference;
  if (GNUNET_OK !=
      TALER_string_to_amount (amount,
                              &ps->reserve_history.amount))
  {
    GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                "Failed to parse amount `%s' at %s\n",
                amount,
                label);
    GNUNET_assert (0);
  }
  {
    struct TALER_TESTING_Command cmd = {
      .cls = ps,
      .label = label,
      .run = &recoup_run,
      .cleanup = &recoup_cleanup,
      .traits = &recoup_traits
    };

    return cmd;
  }
}
