"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.VisualizationsRouter = VisualizationsRouter;
var _configSchema = require("@osd/config-schema");
var _custom_panel_adaptor = require("../../adaptors/custom_panels/custom_panel_adaptor");
var _custom_panels = require("../../../common/constants/custom_panels");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function VisualizationsRouter(router) {
  // Fetch all the savedVisualzations
  const customPanelBackend = new _custom_panel_adaptor.CustomPanelsAdaptor();
  router.get({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/visualizations`,
    validate: {}
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const visualizationList = await customPanelBackend.getAllSavedVisualizations(opensearchNotebooksClient);
      return response.ok({
        body: {
          visualizations: visualizationList
        }
      });
    } catch (error) {
      console.error('Issue in fetching saved visualizations:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  // get all saved visualizations by Id
  router.get({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/visualizations/{savedVisualizationId}`,
    validate: {
      params: _configSchema.schema.object({
        savedVisualizationId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const savedVisualization = await customPanelBackend.getSavedVisualizationById(opensearchNotebooksClient, request.params.savedVisualizationId);
      return response.ok({
        body: {
          visualization: savedVisualization
        }
      });
    } catch (error) {
      console.error('Issue in fetching saved visualizations by ids:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  // Add a new visualization to the panel
  router.post({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/visualizations`,
    validate: {
      body: _configSchema.schema.object({
        panelId: _configSchema.schema.string(),
        savedVisualizationId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const newVisualizations = await customPanelBackend.addVisualization(opensearchNotebooksClient, request.body.panelId, request.body.savedVisualizationId);
      return response.ok({
        body: {
          message: 'Visualization Added',
          visualizations: newVisualizations
        }
      });
    } catch (error) {
      console.error('Issue in adding visualization:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  // Add multiple visualizations to the panel
  router.post({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/visualizations/multiple`,
    validate: {
      body: _configSchema.schema.object({
        panelId: _configSchema.schema.string(),
        savedVisualizationIds: _configSchema.schema.arrayOf(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const allVisualizations = await customPanelBackend.addMultipleVisualizations(opensearchNotebooksClient, request.body.panelId, request.body.savedVisualizationIds);
      return response.ok({
        body: {
          message: 'Visualizations Added',
          visualizations: allVisualizations
        }
      });
    } catch (error) {
      console.error('Issue in adding visualization:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  // Replace an existing visualization
  router.post({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/visualizations/replace`,
    validate: {
      body: _configSchema.schema.object({
        panelId: _configSchema.schema.string(),
        savedVisualizationId: _configSchema.schema.string(),
        oldVisualizationId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const newVisualizations = await customPanelBackend.addVisualization(opensearchNotebooksClient, request.body.panelId, request.body.savedVisualizationId, request.body.oldVisualizationId);
      return response.ok({
        body: {
          message: 'Visualization Replaced',
          visualizations: newVisualizations
        }
      });
    } catch (error) {
      console.error('Issue in replacing visualization:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  // changes the position of the mentioned visualizations
  // Also removes the visualizations not mentioned
  router.put({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/visualizations/edit`,
    validate: {
      body: _configSchema.schema.object({
        panelId: _configSchema.schema.string(),
        visualizationParams: _configSchema.schema.arrayOf(_configSchema.schema.object({
          i: _configSchema.schema.string(),
          x: _configSchema.schema.number(),
          y: _configSchema.schema.number(),
          w: _configSchema.schema.number(),
          h: _configSchema.schema.number()
        }))
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const newVisualizations = await customPanelBackend.editVisualization(opensearchNotebooksClient, request.body.panelId, request.body.visualizationParams);
      return response.ok({
        body: {
          message: 'Visualizations Edited',
          visualizations: newVisualizations
        }
      });
    } catch (error) {
      console.error('Issue in Editing visualizations:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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