/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.coordinator.group.runtime;

import java.util.List;
import java.util.Objects;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.coordinator.group.metrics.CoordinatorRuntimeMetrics;
import org.apache.kafka.coordinator.group.runtime.CoordinatorEvent;
import org.apache.kafka.coordinator.group.runtime.CoordinatorEventProcessor;
import org.apache.kafka.coordinator.group.runtime.EventAccumulator;
import org.slf4j.Logger;

public final class MultiThreadedEventProcessor
implements CoordinatorEventProcessor {
    private static final long POLL_TIMEOUT_MS = 300L;
    private final Logger log;
    private final EventAccumulator<TopicPartition, CoordinatorEvent> accumulator;
    private final List<EventProcessorThread> threads;
    private volatile boolean shuttingDown;
    private final CoordinatorRuntimeMetrics metrics;
    private final Time time;

    public MultiThreadedEventProcessor(LogContext logContext, String threadPrefix, int numThreads, Time time, CoordinatorRuntimeMetrics metrics) {
        this(logContext, threadPrefix, numThreads, time, metrics, new EventAccumulator<TopicPartition, CoordinatorEvent>());
    }

    public MultiThreadedEventProcessor(LogContext logContext, String threadPrefix, int numThreads, Time time, CoordinatorRuntimeMetrics metrics, EventAccumulator<TopicPartition, CoordinatorEvent> eventAccumulator) {
        this.log = logContext.logger(MultiThreadedEventProcessor.class);
        this.shuttingDown = false;
        this.accumulator = eventAccumulator;
        this.time = Objects.requireNonNull(time);
        this.metrics = Objects.requireNonNull(metrics);
        this.metrics.registerEventQueueSizeGauge(this.accumulator::size);
        this.threads = IntStream.range(0, numThreads).mapToObj(threadId -> new EventProcessorThread(threadPrefix + threadId)).collect(Collectors.toList());
        this.threads.forEach(Thread::start);
    }

    @Override
    public void enqueueLast(CoordinatorEvent event) throws RejectedExecutionException {
        this.accumulator.addLast(event);
    }

    @Override
    public void enqueueFirst(CoordinatorEvent event) throws RejectedExecutionException {
        this.accumulator.addFirst(event);
    }

    public synchronized void beginShutdown() {
        if (this.shuttingDown) {
            this.log.debug("Event processor is already shutting down.");
            return;
        }
        this.log.info("Shutting down event processor.");
        this.accumulator.close();
        this.shuttingDown = true;
    }

    @Override
    public void close() throws InterruptedException {
        this.beginShutdown();
        for (Thread thread : this.threads) {
            thread.join();
        }
        this.log.info("Event processor closed.");
    }

    private class EventProcessorThread
    extends Thread {
        private final Logger log;

        EventProcessorThread(String name) {
            super(name);
            this.log = new LogContext("[" + name + "]: ").logger(EventProcessorThread.class);
            this.setDaemon(false);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void handleEvents() {
            while (!MultiThreadedEventProcessor.this.shuttingDown) {
                long idleStartTimeMs = MultiThreadedEventProcessor.this.time.milliseconds();
                CoordinatorEvent event = (CoordinatorEvent)MultiThreadedEventProcessor.this.accumulator.poll(300L, TimeUnit.MILLISECONDS);
                long idleEndTimeMs = MultiThreadedEventProcessor.this.time.milliseconds();
                long idleTimeMs = idleEndTimeMs - idleStartTimeMs;
                MultiThreadedEventProcessor.this.metrics.recordThreadIdleTime(idleTimeMs / (long)MultiThreadedEventProcessor.this.threads.size());
                if (event == null) continue;
                try {
                    this.log.debug("Executing event: {}.", (Object)event);
                    long dequeuedTimeMs = MultiThreadedEventProcessor.this.time.milliseconds();
                    MultiThreadedEventProcessor.this.metrics.recordEventQueueTime(dequeuedTimeMs - event.createdTimeMs());
                    event.run();
                    MultiThreadedEventProcessor.this.metrics.recordEventQueueProcessingTime(MultiThreadedEventProcessor.this.time.milliseconds() - dequeuedTimeMs);
                }
                catch (Throwable t) {
                    this.log.error("Failed to run event {} due to: {}.", new Object[]{event, t.getMessage(), t});
                    event.complete(t);
                }
                finally {
                    MultiThreadedEventProcessor.this.accumulator.done(event);
                }
            }
        }

        private void drainEvents() {
            CoordinatorEvent event;
            while ((event = (CoordinatorEvent)MultiThreadedEventProcessor.this.accumulator.poll()) != null) {
                try {
                    this.log.debug("Draining event: {}.", (Object)event);
                    MultiThreadedEventProcessor.this.metrics.recordEventQueueTime(MultiThreadedEventProcessor.this.time.milliseconds() - event.createdTimeMs());
                    event.complete(new RejectedExecutionException("EventProcessor is closed."));
                }
                catch (Throwable t) {
                    this.log.error("Failed to reject event {} due to: {}.", new Object[]{event, t.getMessage(), t});
                }
                finally {
                    MultiThreadedEventProcessor.this.accumulator.done(event);
                }
            }
        }

        @Override
        public void run() {
            this.log.info("Starting");
            try {
                this.handleEvents();
            }
            catch (Throwable t) {
                this.log.error("Exiting with exception.", t);
            }
            if (MultiThreadedEventProcessor.this.shuttingDown) {
                this.log.info("Shutting down. Draining the remaining events.");
                try {
                    this.drainEvents();
                }
                catch (Throwable t) {
                    this.log.error("Draining threw exception.", t);
                }
                this.log.info("Shutdown completed");
            }
        }
    }
}

