/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.fluss.metadata;

import org.apache.fluss.record.MemoryLogRecordsArrowBuilder;
import org.apache.fluss.record.MemoryLogRecordsCompactedBuilder;
import org.apache.fluss.record.MemoryLogRecordsIndexedBuilder;
import org.apache.fluss.row.compacted.CompactedRow;
import org.apache.fluss.row.indexed.IndexedRow;

/**
 * The format of the log records in log store. The supported formats are 'arrow', 'indexed' and
 * 'compacted'.
 */
public enum LogFormat {

    /**
     * The log record batches are stored in Apache Arrow batch format which is a columnar-oriented
     * format. This can have better performance for analytics workloads as it is more efficient to
     * do column projection. This is the default log format.
     *
     * @see MemoryLogRecordsArrowBuilder
     */
    ARROW,

    /**
     * The log record batches are stored in {@link IndexedRow} format which is a row-oriented
     * format. It is more efficient for event-driven workloads and have smaller memory/disk
     * footprint when a log record batch doesn't have many rows. It isn't good at analytics
     * workloads (e.g., doesn't support column projection).
     *
     * @see MemoryLogRecordsIndexedBuilder
     */
    INDEXED,

    /**
     * The log record batches are stored in {@link CompactedRow} format which is a compact
     * row-oriented format optimized for primary key tables to reduce storage while trading CPU for
     * reads.
     *
     * @see MemoryLogRecordsCompactedBuilder
     */
    COMPACTED;

    /**
     * Creates a {@link LogFormat} from the given string. The string must be either 'arrow',
     * 'indexed' or 'compacted'.
     */
    public static LogFormat fromString(String format) {
        switch (format.toUpperCase()) {
            case "ARROW":
                return ARROW;
            case "INDEXED":
                return INDEXED;
            case "COMPACTED":
                return COMPACTED;
            default:
                throw new IllegalArgumentException("Unsupported log format: " + format);
        }
    }
}
