/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hugegraph.loader.reader.hdfs;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileChecksum;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hugegraph.loader.constant.Constants;
import org.apache.hugegraph.loader.exception.LoadException;
import org.apache.hugegraph.loader.progress.FileItemProgress;
import org.apache.hugegraph.loader.progress.InputItemProgress;
import org.apache.hugegraph.loader.reader.Readable;
import org.apache.hugegraph.loader.reader.file.FileLineFetcher;
import org.apache.hugegraph.loader.reader.file.FileReader;
import org.apache.hugegraph.loader.reader.file.OrcFileLineFetcher;
import org.apache.hugegraph.loader.reader.file.ParquetFileLineFetcher;
import org.apache.hugegraph.loader.source.file.Compression;
import org.apache.hugegraph.loader.source.file.FileFilter;
import org.apache.hugegraph.loader.source.hdfs.HDFSSource;
import org.apache.hugegraph.loader.source.hdfs.KerberosConfig;
import org.apache.hugegraph.util.Log;
import org.slf4j.Logger;

public class HDFSFileReader
extends FileReader {
    private static final Logger LOG = Log.logger(HDFSFileReader.class);
    private final FileSystem hdfs;
    private final Configuration conf = this.loadConfiguration();

    public HDFSFileReader(HDFSSource source) {
        super(source);
        try {
            this.enableKerberos(source);
            this.hdfs = FileSystem.get((Configuration)this.conf);
        }
        catch (IOException e) {
            throw new LoadException("Failed to create HDFS file system", e);
        }
        Path path = new Path(source.path());
        HDFSFileReader.checkExist(this.hdfs, path);
    }

    private void enableKerberos(HDFSSource source) throws IOException {
        KerberosConfig kerberosConfig = source.kerberosConfig();
        if (kerberosConfig != null && kerberosConfig.enable()) {
            System.setProperty("java.security.krb5.conf", kerberosConfig.krb5Conf());
            UserGroupInformation.setConfiguration((Configuration)this.conf);
            UserGroupInformation.loginUserFromKeytab((String)kerberosConfig.principal(), (String)kerberosConfig.keyTab());
        }
    }

    public FileSystem fileSystem() {
        return this.hdfs;
    }

    @Override
    public HDFSSource source() {
        return (HDFSSource)super.source();
    }

    @Override
    public void close() {
        super.close();
        try {
            this.hdfs.close();
        }
        catch (IOException e) {
            LOG.warn("Failed to close reader for {} with exception {}", new Object[]{this.source(), e.getMessage(), e});
        }
    }

    @Override
    protected List<Readable> scanReadables() throws IOException {
        Path path = new Path(this.source().path());
        FileFilter filter = this.source().filter();
        ArrayList<Readable> paths = new ArrayList<Readable>();
        if (this.hdfs.isFile(path)) {
            if (!filter.reserved(path.getName())) {
                throw new LoadException("Please check path name and extensions, ensure that at least one path is available for reading");
            }
            paths.add(new HDFSFile(this.hdfs, path));
        } else {
            Path[] subPaths;
            assert (this.hdfs.isDirectory(path));
            FileStatus[] statuses = this.hdfs.listStatus(path);
            for (Path subPath : subPaths = FileUtil.stat2Paths((FileStatus[])statuses)) {
                if (!filter.reserved(subPath.getName())) continue;
                paths.add(new HDFSFile(this.hdfs, subPath));
            }
        }
        return paths;
    }

    @Override
    protected FileLineFetcher createLineFetcher() {
        if (Compression.ORC == this.source().compression()) {
            return new OrcFileLineFetcher(this.source(), this.conf);
        }
        if (Compression.PARQUET == this.source().compression()) {
            return new ParquetFileLineFetcher(this.source(), this.conf);
        }
        return new FileLineFetcher(this.source());
    }

    private Configuration loadConfiguration() {
        Configuration conf = new Configuration();
        conf.addResource(new Path(this.source().coreSitePath()));
        if (this.source().hdfsSitePath() != null) {
            conf.addResource(new Path(this.source().hdfsSitePath()));
        }
        return conf;
    }

    private static void checkExist(FileSystem fs, Path path) {
        try {
            if (!fs.exists(path)) {
                throw new LoadException("Please ensure the file or directory exists: '%s'", path);
            }
        }
        catch (IOException e) {
            throw new LoadException("An exception occurred while checking HDFS path: '%s'", (Throwable)e, path);
        }
    }

    private static class HDFSFile
    implements Readable {
        private final FileSystem hdfs;
        private final Path path;

        private HDFSFile(FileSystem hdfs, Path path) {
            this.hdfs = hdfs;
            this.path = path;
        }

        public FileSystem hdfs() {
            return this.hdfs;
        }

        @Override
        public String name() {
            return this.path.getName();
        }

        @Override
        public Path path() {
            return this.path;
        }

        @Override
        public InputStream open() throws IOException {
            return this.hdfs.open(this.path);
        }

        @Override
        public InputItemProgress inputItemProgress() {
            byte[] bytes;
            long timestamp;
            String name = this.path.getName();
            try {
                timestamp = this.hdfs.getFileStatus(this.path).getModificationTime();
            }
            catch (IOException e) {
                throw new LoadException("Failed to get last modified time for HDFS path '%s'", (Throwable)e, this.path);
            }
            try {
                FileChecksum checksum = this.hdfs.getFileChecksum(this.path);
                if (checksum == null) {
                    throw new LoadException("The checksum of HDFS path '%s' is null", this.path);
                }
                bytes = checksum.getBytes();
            }
            catch (IOException e) {
                throw new LoadException("Failed to calculate checksum for HDFS path '%s'", (Throwable)e, this.path);
            }
            String checkSum = new String(bytes, Constants.CHARSET);
            return new FileItemProgress(name, timestamp, checkSum, 0L);
        }

        public String toString() {
            return "HDFS: " + this.path;
        }
    }
}

